/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/**
 * Form Dialog View Classes
 * #ActivityDialog
 *     #StateDialog
 *     #TaskDialog
 *     #ApprovalDialog
 * #AssignDialog
 *     #AssigneeItem
 */
var AWP = AWP || {};
AWP.View = AWP.View || {};

/**
 * Activity Dialog. It can be Approval, Approver or Task
 * @since 4.0
 */
AWP.View.ActivityDialog = Backbone.View.extend({
    // backbone prop
    tagName: 'form',
    className: 'aui aui-nopad approval',
    // custom prop
    template : AWP.Templates.activityDialog,

    events: {
        'click .accept': 'accept',
        'click .approve': 'approve',
        'click .reject': 'reject',
        'click .cancel': 'close',
        'click .viewparams': 'viewParameters',
        'submit': 'nop',
        'input input, textarea' : 'inputField',
        'keyup .textarea-note' : 'onNoteChange',

        // Assign dialog
        'click .unassign': 'unassign',
        'click .assignApproval': 'assignApproval'
    },

    render: function() {
        this.$el.html(
            this.template({
                model: this.model,
                buttons: this.options.buttons,
                addnote: this.options.addnote,
                disableNote: this.options.disableNote,
                note: this.options.note
            })
        );

        this.$stateProps = this.$(".awp-stateProperties");

        if(this.isRequiredApprovalComment()) {
            if(this.$('.approve, .reject').length){
                this.$('.textarea-note').attr("placeholder", AJS.I18n.getText("approvalsworkflow.configuration.requireApprovalComment.placeholder"));
            }
            else if(this.$('.accept').length){
                this.$('.textarea-note').attr("placeholder", AJS.I18n.getText("approvalsworkflow.configuration.requireApprovalComment.placeHolderStateTransition"));
            }
            this.setButtonActionsDisabled(true)
        }

        // if assign button is shown, limit width of the slide caption to avoid overlay
        if(this.model.get('canAssign')) {
            AJS.$('#' + AWP.Slides.current()  + ' .awp-dialog-title').addClass('awp-slide-title-limit');
        }

        if(this.model.get('minimumRequiredApprovers') && this.model.get('minimumRequiredApprovers') > 0) {
            this.assignApproval();
            AJS.$('.awp-autocomplete-filtered-user').focus();
        }

        return this;
    },

    onNoteChange: function() {
        //Check if this is a regular approval
        if(this.$('.approve, .reject').length){
            //Check that current user has permission to approve
            if(!this.options.buttons.approve.readonly) {
                //Check that comments are required for approval/rejection
                if(this.isRequiredApprovalComment()){
                    var noteTrimmedText = this.$('.textarea-note').val().trim();
                    this.setButtonActionsDisabled(noteTrimmedText === '');
                }
            }
        }

        //Check if this is a set state submission
        else if(this.$('.accept').length){
            //Check that comments are required for acceptance
            if(this.isRequiredApprovalComment()){
                var noteTrimmedText = this.$('.textarea-note').val().trim();
                this.setButtonActionsDisabled(noteTrimmedText === '');
            }
        }
    },

    setButtonActionsDisabled: function (disabled) {
        this.$('.accept,.approve,.reject').attr('disabled', disabled);
    },

    isRequiredApprovalComment: function() {
        return AWP.action.requireApprovalComment;
    },

    inputField: function (e) {
        var $input = this.$(e.currentTarget);
        if($input) {
            this.model.set($input.attr("name"), $input.val());
        }
    },

    setButtonActions: function () {
        var btnView = new AWP.View.Button({
            model: this.options.actions
        });
        this.$('.buttons').html(btnView.render().el);
    },

    assignApproval: function(evt) {
        if(evt) {
            evt.preventDefault();
        }

        // other slides
        this.assignSlide = new AWP.View.AssignSlide({
            // slide prop
            model: this.model,
            hidden: true,
            caption: this.model.get("name")
        });
        this.assignSlide.render();
        this.assignSlide.show();
    },

    destroyAssignSlide: function () {
        if(this.assignSlide) {
            this.assignSlide.remove();
            this.assignSlide = null;
        }
    },

    setActions: function () {
        var buttons = {
            "assign" : {"primary": true}
        };

        return buttons;
    },

    close: function () {
        AWP.View.WorkflowApp.closeDialog();
    },

    error: function(event,error) {
        alert("error");
    },

    accept: function(evt) {
        evt.preventDefault();
        this.model.save({});
    },

    approve : function(e) {
        e.preventDefault();
        this.disableButtons();
        this.model.save({}, {
            action: "approve",
            pageId: AWP.pageId,
            success: _.bind(this.enableButtons, this)
        });
    },

    reject : function(e) {
        e.preventDefault();
        this.disableButtons();
        this.model.save({},{
            action: "reject",
            pageId: AWP.pageId
        });
    },

    unassign: function(event) {
        event.preventDefault();
        this.model.save(
            {assignee: this.model.get("approver").get("name")},
            {
                action: "unassign",
                pageId: AWP.pageId,
                success: function(model, data){
                    // refresh state
                    AWP.View.WorkflowApp.refreshState(data);
                }
            }
        );
    },

    nop: function(event) {
        event.preventDefault();
    },

    viewParameters: function(evt) {
        evt.preventDefault();

        // other slides
        this.assignSlide = new AWP.View.ParamsSlide({
            // slide prop
            model: this.model,
            hidden: true,
            caption: this.model.get("name")
        });
        this.assignSlide.render();
        this.assignSlide.show();
    },

    reloadTasks: function(tasks, textStatus, jqXHR) {
        // reload tasks tab
        AWP.workflowApp.tabMenuView.reloadTab('tasks');
        AWP.workflowApp.tabMenuView.reloadTab('activity');

        // update page message
        AWP.View.WorkflowApp.refreshMessage(tasks);

        // go back to tab menu slide
        AWP.Slides.previous('slide-tabs');
    },

    enableButtons: function () {
        this.$('.aui-button').attr('disabled', false);
    },

    disableButtons: function () {
        this.$('.aui-button').attr('disabled', true);
    }

});


/**
 * @version 2.0
 * @since 4.0
 */
AWP.View.StateDialog = AWP.View.ActivityDialog.extend({
    newStateInput: 'newstate-input',

    initialize: function() {
        this.$('#select-new-state').live('change', _.bind(this.changeState, this));
    },

    accept: function(evt) {
        var state = this.$('select').val(),
            addedNewState = this.$('.newstate-input').val();

        evt.preventDefault();

        // Send new state input if exists and filled
        if(addedNewState) {
            state = addedNewState;
        }

        // changeState before ajax request
        this.model.set("newState", state);
        this.model.save({});
    },

    changeState : function(evt) {
        var state = AJS.$('select').val();

        // awp000 is the option to create a new state
        if(state === 'awp000') {
            this.showNewState();
        }
        // If an existent state is chosen, then remove new state field
        else {
            AJS.$('.newstate-input').remove();
        }
    },

    showNewState: function () {
        // create new state field
        var input = AJS.$('<input>')
            .addClass('text ' + this.newStateInput)
            .attr({
                'name' : 'newstate',
                'placeholder' : AJS.I18n.getText("approvalsworkflow.dialog.statename")
            });
        AJS.$(input)
            .insertAfter('#select-new-state')
            .focus();
    }
});

/**
 * @version 1.0
 * @since 4.4
 */
AWP.View.TaskDialog = AWP.View.ActivityDialog.extend({
    events: function() {
        return _.extend({}, AWP.View.ActivityDialog.prototype.events, {
            'click .complete': 'complete',
            'click .remove': 'remove'
        });
     },

    render: function () {
        // parent class
        AWP.View.ActivityDialog.prototype.render.apply(this, arguments);

        // then specific stuff for this child-class
        // TASK
        // if task dialog and no due date, show due date btn
        // to let user add a due date
        if(this.options.buttons.complete) {
            // if there is a complete button, it means its a task
            var dueDate = new AWP.View.DueDate({
                model: this.model,
                el: this.$('.awp-changeTaskDueDate'),
                onSelect: _.bind(function (date) {
                    // update due date
                    this.model.save({
                        dueDate : date
                    }, {
                        action: 'update',
                        success: _.bind(this._dueDateIsSet, this)
                    });
                }, this)
            });
            dueDate.render();

            if(!dueDate.getFriendlyDueDateValue()) {
                var dueDateTrigger = new AWP.View.DueDateTrigger({
                    dueDateField: this.$('.awp-changeTaskDueDate')
                });
                this.$('.buttons').append(dueDateTrigger.render());
                dueDateTrigger.on('showDueDate', function() {
                    dueDate.editMode();
                });

                dueDate.on('pickedDate', _.bind(function(date) {
                    dueDateTrigger
                        .setDueDate(dueDate.getFriendlyDueDateValue())
                            .showHint()
                                .hideDueDate();
                }, this));
            }

            AJS.messages.hint(this.$('.duedate-remove-hint'), {
                body: AJS.I18n.getText("approvalsworkflow.task.duedate.editremove.hint"),
                closeable: false
             });
        }

        return this;
    },

    complete: function(event) {
        event.preventDefault();
        var note = this.$('.textarea-note').val(),
            update = {};

        if(note !== "") {
            update = {
                "note": note
            };
        }

        this.model.save(update, {
            action: "complete",
            success: this.reloadTasks
        });
        this.destroyAssignSlide();
    },

    remove : function(event) {
        event.preventDefault();
        if(this.removeConfirmed(event)) {
            this.model.save({},{
                action: "remove",
                success: _.bind(function() {
                    this.reloadTasks();
                }, this)
            });
        }
    },

    removeConfirmed: function (event) {
        var $input = AJS.$(event.currentTarget);
        var confirmed = true;
        if($input.attr('data-confirm') !== '') {
            var $msg = this.$el;
            // this.make is deprecated at backbone 0.9.10 (http://backbonejs.org/#changelog)
            var div = this.make('div', {
                "class" : 'aui-lozenge aui-lozenge-error aui-lozenge-subtle'
            }, $input.attr('data-confirm'));
            $msg.prepend(div);

            $input.addClass('aui-button-primary');
            $input.attr('data-confirm', '');
            this.$el.parent().addClass('confirm-delete');
            confirmed = false;
        }

        return confirmed;
    },

    _dueDateIsSet: function (model) {

    }
});


/**
 * @version 2.0
 * @since 4.0
 */
AWP.View.ApprovalDialog = AWP.View.ActivityDialog.extend({
    render: function () {
        // parent class
        AWP.View.ActivityDialog.prototype.render.apply(this, arguments);
        return this;
    }
});

/**
 * @version 1.0
 * @since 4.0
 */
AWP.View.AssignDialog = Backbone.View.extend({
    template : AWP.Templates.assignApprovalDialog,

    events: {
        'click .assign' : 'assign'
    },

    initialize: function () {
        this.assignees = new AWP.Collection.Users();
        this.assignees.bind("add",this.addAssignee,this);
        this.assignees.bind("destroy",this.checkUsersAssigned,this);
    },

    render: function() {
        this.$el.html(this.template({
            requiredApprovers: this.model.get('minimumRequiredApprovers'),
            filterUsers: this.model.get('filterUsers'),
            filterGroups: this.model.get('filterGroups')
        }));

        // tasks cant be multi-assigned
        if(this.model.get('taskId') && this.model.get('assigneeName')) {
            this.assignees.create({
                name: this.model.get('assigneeName'),
                fullName: this.model.get('assigneeFullName'),
                pictureUrl: this.model.get('assigneePictureUrl')
            });
        }

        var self = this;
        this.$('.awp-autocomplete-filtered-user').each(function() {
            var users = "",
                groups = "",
                excludedUsers="";

            if(self.model.get('filterUsers')) {
                users = self.model.get('filterUsers');
            }
            if(self.model.get('filterGroups')) {
                groups = self.model.get('filterGroups');
            }
            if(self.model.get('filterExcludedUsers')) {
                excludedUsers = self.model.get('filterExcludedUsers');
            }

            var view = new AWP.View.AutocompleteUser({
                el: AJS.$(this),
                filterUsers: users,
                filterGroups: groups,
                filterExcludedUsers: excludedUsers
            });
            view.render();
            view.on("select.autocomplete-user", self.createAssignee, self);
        });

        setTimeout(_.bind(function() {
            this.$('.awp-autocomplete-filtered-user').focus();
        }, this), 351);

        return this;
    },

    addAssignee: function(selected) {
        var assignee = new AWP.View.AssigneeItem({
            model: selected
        });
        this.$(".awp-list").append(assignee.render().el);
        this.checkUsersAssigned();
    },

    checkUsersAssigned: function () {
        // tasks cant be multi-assigned
        // tasks can only be assigned to 1 user
        // taskId to check if we're assigning a task or something else
        if(this.model.get('taskId') && this.assignees.length >= 1) {
            this.$('.awp-input-assignuser').hide();
        }
        else {
            this.$('.awp-input-assignuser').show();
        }
    },

    createAssignee: function(selected) {
        AWP.log('createAssignee', selected);

        this.assignees.create({
                    name: selected.get("username"),
                    fullName: selected.get("title"),
                    pictureUrl: selected.get("thumbnailLink").href
                }
        );
        AJS.$(event.currentTarget).val("");
    },

    assign: function(e) {
        e.preventDefault();
        var note = this.$("textarea").val();
        this.model.save({
            note : note,
            assignee : this.assignees.toNamesList()
        }, {
            action : "assign",
            success: function(model, data) {
                // refresh state
                if(data){
                    AWP.View.WorkflowApp.refreshState(data);
                }

                // reload tasks tab
                AWP.workflowApp.tabMenuView.reloadTab('tasks');
                AWP.workflowApp.tabMenuView.reloadTab('activity');

                // go back to tab menu slide
                AWP.Slides.previous('slide-tabs');
            }
        });
    }
});


/**
 * @version 1.0
 * @since 4.0
 */
AWP.View.AssigneeItem = Backbone.View.extend({
    tagName: "li",
    model: AWP.Model.User,
    template: AWP.Templates.assigneeItem,

    events: {
        'click li .delete' : "clear"
    },

    initialize: function() {
        this.model.bind('destroy', this.remove, this);
    },

    clear: function() {
        this.model.destroy();
    },

    render: function() {
        this.$el.html(this.template({assignee: this.model}));
        return this;
    }
});
