/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/**
 * Tasks
 * #TasksTab
 * #TaskForm
 */
var AWP = AWP || {};
AWP.View = AWP.View || {};

AWP.View.TasksTab = Backbone.View.extend({
    caption : AJS.I18n.getText("adhocworkflows.tab.task"),
    id: "tasks",
    icon: "tasks",
    className : "tabs-pane",
    
    canAddTasks: false,
    completedTasks: 0,
    viewCompleted: false,
    
    events: {
        "click .showcomp" : "toggleCompleted",
        "click .addtask" : "createTask",
        "keypress .text-addtask"  : "updateOnEnter"
    },

    initialize: function(canAddTasks) {
        this.canAddTasks = canAddTasks;

        this.tasks = new AWP.Collection.Tasks();
        // TODO once we upgrade Backbone, we should use listenTo
        this.tasks.bind('add', function(model) {
            this.addTask(model);
            this.badge();
        }, this);
        this.tasks.bind('reset', function() {
            this.addAllTasks();
            this.badge();
        }, this);

        this.populate();
    },

    populate: function () {
        AWP.log('populate tasks');
        this.tasks.fetch();
    },
    
    reload: function () {
        AWP.log('reload tasks');
        this.populate();
        this.completedTasks = 0;
    },
    
    badge: function () {
        AWP.log('badge');
        var n = this.tasks.length - this.completedTasks;

        if(n > 0) {
            this.$badge.hide();
            this.$badge.html(n);
            this.$badge.show('fast');
        }
        else {
            this.$badge.html('');
            this.$badge.hide();
        }
        
        // update badge of embedded page (parent page)
        if(parent && parent.AWP.Embedded) {
            parent.AWP.Embedded.Workflow.updateBadge(n);
        }
    },
    
    render: function() {
        // form add task
        if (this.canAddTasks) {
            this.form = new AWP.View.TaskForm({
                model: this.model
            });
            this.$el.append(this.form.render().el);
        }
        // list
        // this.make is deprecated at backbone 0.9.10 (http://backbonejs.org/#changelog)
        var ul = this.make("ul", {
            "class" : "awp-list awp-list-tasks"
        });
        this.$el.append(ul);
        
        this.renderBadge();
        return this;
    },

    renderBadge: function () {
        // create badge element
        // this.make is deprecated at backbone 0.9.10 (http://backbonejs.org/#changelog)
        var sub = this.make('sub', {
            "class" : "awp-badge awp-badge-tab aui-lozenge"
        });
        
        AJS.$('.tasks').append(sub);
        this.$badge = AJS.$('.tasks sub');
        this.$badge.hide();
    },
    
    toggleCompleted: function () {
        this.$el.toggleClass("show-completed");
    },

    /**
     * Show buttons when input in task field
     * Check if using key combo to add a task
     */
    
    updateOnEnter: function(e) {
        // Enter + Shift
        if (e.keyCode == 13 && e.shiftKey) {
            // let continue
        }
        // Only Enter
        else if (e.keyCode == 13) {
            e.preventDefault();
            this.createTask();
        }
    },
    
    createTask: function () {
        var note = assignee = "";
        var dueDate = this.form.getDueDateValue();
        var input = this.form.getInputValue();

        if(input !== '') {
            // task is being assigned to someone
            assignee = this.form.checkMention(input);

            // remove at symbol
            input = input.replace('@' + assignee, assignee);

            // if input is multi line:
            // first line will be the title of the task
            // and next lines aditional notes
            var lines = input.split('\n');
            input = lines[0];
            
            // notes
            if(lines.length > 1) {
                for(i = 1; i < lines.length; i++) {
                    note += lines[i] + "\n";
                }
            }
            
            // magic: create calls Backbone.sync
            var ok = this.tasks.create({
                "name": input,
                "assignee": assignee,
                "note": note,
                "hint" : "",
                "dueDate" : dueDate
            }, {wait: true});

            if(ok) {
                var trackLabel = '';
                this.form.resetForm();

                if(note && assignee) {
                    trackLabel = 'assigned-comment';
                }
                else if(assignee) {
                    trackLabel = 'assigned';
                }
                else if(note) {
                    trackLabel = 'comment';
                }
            }
            else {
                AWP.log('createTask model is not valid');
            }
        }
    },
    
    addTask: function (task) {
        AWP.log('addTask', task);
        
        if(task.get('completed')) {
            this.completedTasks++;
            
            // show completed link
            if(!this.viewCompleted) {
                this.viewCompleted = true;
                this.$el.append(AWP.Templates.viewCompleted({}));
                this.$el.addClass('show-completed-opt');
            }
        }
        
        var view = new AWP.View.TaskList({
            model : task
        });

        this.$("ul").append(view.render().el);
    },
    
    addAllTasks: function() {
        this.$("ul").html('');
        this.tasks.each(this.addTask,this);
    },
    
    focusAddTask: function () {
        AJS.$('.text-addtask').focus();
    }
});

/**
 * @version 1.0
 * @since 4.0
 */
AWP.View.TaskForm = Backbone.View.extend({
    className: 'awp-container',
    model: AWP.Model.Task,
    template: AWP.Templates.taskDialog,
    
    initialInputHeight: 0,
    buttonsShown: false,
    
    events: {
        "mouseover .addtask"  : "changeHintOver",
        "mouseout .addtask"  : "changeHintOut",
        //"keyup .text-addtask"  : "observeMentions",
        "keypress .text-addtask"  : "showButtons",
        "focus .text-addtask" : "makeElastic"
    },

    render: function () {
        this.$el.html(this.template());
        this.$input = this.$('.text-addtask');
        this.$buttons = this.$('.task .buttons');
        this.$hint = this.$('.task .help-addtask');

        this.hideButtons();

        var self = this;
        this.dueDate = new AWP.View.DueDate({
            model: new Backbone.Model(),
            el: this.$('.awp-setTaskDueDate'),
            onSelect: function (date) {
                // update due date
                this.model.set('expirationDateValue', date);
                this.model.set('expirationDate', date);
                self._dueDateIsSet(this.model);
            }
        });
        this.dueDate.render();

        var dueDateTrigger = new AWP.View.DueDateTrigger({
            dueDateField: this.$('.awp-setTaskDueDate')
        });
        this.$('.buttons').append(dueDateTrigger.render());
        dueDateTrigger.on('showDueDate', _.bind(function() {
            this.dueDate.editMode();
        }, this));

        this.dueDate.on('pickedDate', _.bind(function(date) {
            dueDateTrigger
                .setDueDate(date)                    
                    .hideDueDate();
        }, this));

        return this;
    },
    
    hideButtons: function () {
        if(this.buttonsShown) {
            this.$buttons.hide();
            this.buttonsShown = false;
            AJS.$('.awp-list-tasks').removeClass('short');
        }
    },
    
    showButtons: function () {
        if(!this.buttonsShown) {
            this.$buttons.show();
            this.buttonsShown = true;
            AJS.$('.awp-list-tasks').addClass('short');
        }
    },
    
    makeElastic: function () {
        this.$input.elastic();
        this.initialInputHeight = this.$input.height();
        this.$input.off('focus');
    },
    
    /**
     * Check if text is mentioning a user
     * the pattern is @NAME_OF_USER
     */
    checkMention: function (txt) {
        var res = "";
        var user = txt.match(/@[a-zA-Z0-9.-]+/i);
        AWP.log('mention: ' + user);
        
        if(user !== null) {
            res = user
                    .toString()
                        .substring(1);
        }
        return res;
    },
    
    /**
     * TODO If a mention is detected, we can active autocomplete
     */
    observeMentions: function (e) {
        var mention = this.checkMention(this.$input.val());
        if(mention) {
            /*
            var url = '/confluence/rest/adhocworkflows/1/search/user.json?groups=&users=&max-results=10&query=' + mention;
            AJS.$.ajax(url, {
                onsuccess: function(e) {
                    console.log(e)
                }
            });
            //*/
        }
    },
    
    
    changeHintOver: function () {
        this.$hint.html(this.$hint.attr('data-alt-softkey'));
    },

    changeHintOut: function () {
        this.$hint.html(this.$hint.attr('data-alt-mention'));
    },

    resetForm: function () {
        this.$input.val("");
        this.$input.height(this.initialInputHeight + "px");
        this.hideButtons();
    },
    
    getInputValue: function () {
        return this.$input.val();
    },
    
    getDueDateValue: function () {
        var duedate = "";
        if(this.dueDate) {
            duedate = this.dueDate.getDueDateValue();
        }
        return duedate;
    },

    _dueDateIsSet: function (model) {
    }
});