/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var AWP = AWP || {};
AWP.View = AWP.View || {};

AWP.View.DueDate = Backbone.View.extend({
    template: AWP.Templates.dueDateDialog,
    showLabel: true,

    events: {
        'click .awp-duedate-time': 'editMode',
        'mouseover .awp-duedate-remove': 'showRemoveLabel',
        'mouseout .awp-duedate-remove': 'hideRemoveLabel',
        'click .awp-duedate-remove': 'removeDueDate'
    },

    render: function() {
        this.$el.html(this.template({
            dueDate: this.getDueDateValue(),
            friendlyDueDate: this.getFriendlyDueDateValue()
        }));

        // from template
        this.$wrapper = this.$('.awp-duedate-option');
        this.$input = this.$('.awp-duedate-input');
        this.$time = this.$('.awp-duedate-time');
        this.$remove = this.$('.awp-duedate-remove');
        this.$removeLabel = this.$('.awp-duedate-remove-label');

        // Init datepicker only if it is shown. Of course, only init once
        this.initDatePicker = _.once(this._initDatePicker);

        return this.el;
    },

    readMode: function () {
        this.$input.hide();
        this.$time.show();
        (this.getDueDateValue()) ? this.$remove.show() : this.$remove.hide();
    },

    editMode: function () {
        if(!this.$wrapper.hasClass('awp-loading')) {
            this.$time.hide();
            this.$input.show();
            this.initDatePicker();
            this.$input.focus();
            (this.getDueDateValue()) ? this.$remove.show() : this.$remove.hide();
        }
    },

    _initDatePicker: function () {
        var options = {
            dateFormat: "yy-mm-dd",
            onSelect: _.bind(this.pickDate, this),
            onClose: _.bind(this.readMode, this)
        };

        if(this.options.onClose) {
            _.extend(options, {
                onClose: this.options.onClose
            });
        }

        this.$input.datepicker(options, "hide");
    },

    pickDate: function (date) {
        this.$el.addClass('awp-loading');
        if(this.options.onSelect) {
            this.options.onSelect.call(this, date);
        }
        this.trigger('pickedDate', date);
    },

    hideDatepicker: function () {
        this.$input.datepicker("hide");
    },

    getDueDateValue: function () {
        var dueDate = this.model.get('expirationDateValue');
        if(!dueDate) {
            dueDate = this.model.get('dueDate');
        }
        //Check if date value includes time and remove it
        if(dueDate && dueDate.indexOf(' ')>0){
            dueDate = dueDate.slice(0, dueDate.indexOf(' '));
        }
        return dueDate;
    },


    getFriendlyDueDateValue: function () {
        var friendlyDueDate = this.model.get('expirationDate');
        if(!friendlyDueDate) {
            friendlyDueDate = this.model.get('friendlyDueDate');
        }
        return friendlyDueDate;
    },

    showRemoveLabel: function () {
        if(this.showLabel) this.$removeLabel.show();
    },

    hideRemoveLabel: function () {
        if(this.showLabel) this.$removeLabel.hide();
    },

    setRemoveLabel: function (show) {
        this.showLabel = show;
    },

    removeDueDate: function (e) {
        e.preventDefault();
        this.$input.datepicker('setDate', '');
        this.pickDate('');
        this.model.clear();
        this.readMode();
    }
});


AWP.View.DueDateTrigger = Backbone.View.extend({
    className: 'awp-duedate-trigger',
    template: AWP.Templates.dueDateTrigger,
    dueDateValue: '',

    events: {
        'mouseover .awp-duedate-trigger-open': 'showHint',
        'mouseout .awp-duedate-trigger-open': 'hideHint',
        'click .awp-duedate-trigger-open': 'showDueDate'
    },

    render: function () {
        this.$el.html(this.template());


        this.$hint = this.$('.help-duedate');
        this.$duedate = this.options.dueDateField;
        this.hideDueDate();

        if(this.options.dueDateValue) {
            this.setDueDate(this.options.dueDateValue);
        }
        else {
            this.hideHint();
        }

        return this.el;
    },

    showHint: function () {
        this.$hint.show();
        return this;
    },

    hideHint: function () {
        if(!this.dueDateValue) {
            this.$hint.hide();
        }
        return this;
    },

    hideDueDate: function () {
        if(this.$duedate) {
            this.$duedate.hide();
        }
        this.$el.css('display', 'inline-block');
        return this;
    },

    showDueDate: function () {
        if(this.$duedate) {
            this.$duedate.show();
        }
        this.$el.hide();
        this.trigger('showDueDate');
        return this;
    },

    setDueDate: function (date) {
        this.dueDateValue = date;
        this.$hint.html(date);
        this.showHint();
        return this;
    }
});
