/**
 * AJS.Meta is used to access dynamic metadata passed from the
 * server to JavaScript via the page HTML.
 *
 * @since 3.6
 */
(function($) {

    /**
     * Returns a boolean if the passed string is "true" or "false", ignoring case, else returns the original string.
     * @param value
     */
    AJS.asBooleanOrString = function (value) {
        var lc = value ? value.toLowerCase() : "";

        if (lc == "true")  return true;
        if (lc == "false") return false;

        return value;
    };

    // A backing map to use if the user sets data.
    var overrides = {};

    AJS.Meta = $.extend({}, AJS.Meta, {
        /**
         * Sets metadata with a key and value, for use when the state of the page changes after
         * loading from the server
         * @param key
         * @param value
         */
        set: function (key, value) {
            overrides[key] = value;
        },

        /**
         * Returns a value given a key. If no entry exists with the key, undefined is returned.
         * If the string value is "true" or "false" the respective boolean value is returned.
         *
         * @method get
         * @param key
         * @return {String} or {boolean}
         */
        get: function (key) {
            if (typeof overrides[key] != "undefined") return overrides[key];

            var metaEl = $("meta[name='ajs-" + key + "']");
            if (!metaEl.length)
                return undefined;

            var value = metaEl.attr("content");
            return AJS.asBooleanOrString(value);
        },

        /**
         * Returns true if the value for the provided key is equal to "true", else returns false.
         *
         * @method getBoolean
         * @param key
         * @return {boolean}
         */
        getBoolean: function (key) {
            return this.get(key) === true;
        },

        /**
         * Returns a number if the value for the provided key can be converted to one.
         * Good for retrieving content ids to check truthiness (e.g. '0' is truthy but 0 is falsy).
         *
         * @method getNumber
         * @param key
         * @return {number}
         */
        getNumber: function (key) {
            return +this.get(key);
        },

        /**
         * Mainly for use when debugging, returns all Data pairs in a map for eyeballing.
         */
        getAllAsMap: function () {
            var map = {};
            $("meta[name^=ajs-]").each(function () {
                map[this.name.substring(4)] = this.content;
            });
            return $.extend(map, overrides);
        }
    });

    /**
     * Returns Link metadata for a page, commonly found from <link> tags in the <head>.
     */
    AJS.Meta.Links = {

        /**
         * Returns a canonical URI for a Page or BlogPost, if present.
         */
        canonical: function () {
            // e.g. <link href="http://localhost:8080/confluence/display/TST/Home" rel="canonical">
            return $('head link[rel="canonical"]').attr('href');
        },

        /**
         * Returns a shortlink URI for a Page or BlogPost, if present.
         */
        shortlink: function () {
            // e.g. <link href="http://localhost:8080/confluence/x/BAAE" rel="shortlink">
            return $('head link[rel="shortlink"]').attr('href');
        }

    };

})(AJS.$);