/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var AWP = AWP || {};
AWP.Embedded = AWP.Embedded || {};

AWP.Embedded.Workflow = (function(AJS, $) {
    var action = 'workflow',
        dialog = null,
        dialogId = '',
        approval = "",
        task = "",

        modalDialog = null,
        modalDialogId = '',
        modalDialogOpen = false;

    setState = function(pageStatus) {
        var $dialogId = $('#' + dialogId);
        var target = "#title-text";
        var idPageMessage = "#workflow-page-message";

        // 1. Lozenge
        $dialogId.css("display","block");
        if (pageStatus.stateName && pageStatus.stateName !== '' && !pageStatus.hideStates) {
            var finalClass = "";
            if (pageStatus.finalState) {
                finalClass = "final";
            } else if (pageStatus.hasFinalState) {
                finalClass = "notfinal";
            }
            // Refined Wiki compatible
            $("#title-heading .cw-state, .rw_page_title .cw-state").remove();
            if ($(".rw_page_title").length > 0) {
                target = ".rw_page_title";
            }

            /* In-page status */
            if (pageStatus.updatePageStatus) {
                var statusInPage = AJS.$("#content .status-macro:first");
                if (statusInPage.length > 0) {
                    statusInPage.removeClass("aui-lozenge-complete aui-lozenge-success aui-lozenge-current");
                    statusInPage.html(pageStatus.stateName);
                    if (pageStatus.finalState) {
                        statusInPage.addClass("aui-lozenge-success");
                    } else if (pageStatus.hasFinalState) {
                        statusInPage.addClass("aui-lozenge-current");
                    } else {
                        statusInPage.addClass("aui-lozenge-complete");
                    }
                }
            }
        }

        // 2. Messages
        setMessages(pageStatus.messages);

        // 3. Page version
        // todo: check if we are actually viewing an old version of the page
        // todo: check with refined wiki to make sure this renders in the right location
        if (pageStatus.hasFinalState) {
            if (pageStatus.alternateViewUrl) {
                var $viewChangeLink = $("#view-change-link");
                if ($viewChangeLink.length > 0) {
                    var href = $viewChangeLink.attr("href");
                    var $span = $viewChangeLink.parents("span");
                    $span.html(AWP.Templates.versionInfo({
                        changeUrl: href,
                        contextPath: AJS.Confluence.getContextPath(),
                        pageStatus: pageStatus,
                        pageId: AJS.params.pageId
                    }));
                } else {
                    // no version info, try appending alt link to page meta data

                    var $pageMetaInfo = $("li.page-metadata-modification-info");
                    if($pageMetaInfo.length > 0){

                          // check if alt link already embedded to avoid duplications
                          if($('#awp-alternate-view').length > 0) {
                            //console.log('awp-alternate-view', $('#awp-alternate-view'), $('#awp-alternate-view')[0])
                            $('#awp-alternate-view').parent().remove();
                          }

                          $pageMetaInfo.append(AWP.Templates.altLinkNoVersion({
                            contextPath: AJS.Confluence.getContextPath(),
                            pageStatus: pageStatus
                          }));
                    }
                }
            } else {
                // todo has final state, but has never been reached
            }
        } else {
        }

        // 4. The badge
        $(".awp-badge").remove();
        if (pageStatus.actionable || pageStatus.activeTasks > 0) {
            // we are wrapping the awp menu icon to position relative and ensure badge is
            // displayed at top right of the menu icon with any theme (like refined wiki)
            $parent = $dialogId.parent();
            $div = AJS.$('<div></div>', {'id': 'awp-menu-icon'});
            $dialogId.appendTo($div);

            // the badge in the AWP icon
            $div.append(AWP.Templates.badge({message: pageStatus.activeTasks}));
            $parent.prepend($div);

            // if awp dialog is open, we hide the tasks badge
            if(AJS.$('#inline-dialog-adhocWorkflowsDialogPopup').is(":visible")) {
                $(".awp-badge").hide();
            }
        }

        // If has workflow, hide add approval/workflow option from the tools dropdown
        if(!pageStatus.hideStates) {
            $("#addAdhocWorkflow").css("display", "none");
            $("#addApprovalWorkflow").css("display", "none");
        }
        $("#editPageWorkflowLink").css("display", "block");


        // If the user is has the error message because they can't view a draft page, also hide the page meta data line
        if($('#workflow_page_not_permitted').length){
            $(".page-metadata").hide();
        }
    };

    /*
     * Display workflow related messages using AUI message style below the title page
     **/
    setMessages = function (msg) {
        var idPageMessage = "#workflow-page-message";
        if(msg) {
            // if messages returned is not an array, wrap it into one
            if(!jQuery.isArray(msg)) {
                msg = [msg];
            }
            if (msg.length > 0) {
                // when we enable the workflow for the very first time,
                // it's possible #workflow-page-message does not exist,
                // so let's check and add it if needed
                var $workflowMessagesSection = $(idPageMessage);

                if ($workflowMessagesSection.length === 0) {
                    pageMessageContainer = $('<div></div>', {
                        id: 'workflow-page-message'
                    });
                    //$("#main-content").prepend(pageMessageContainer);
                } else {
                    $workflowMessagesSection.html("");
                }
                for (var i = 0; i < msg.length;i++) {
                    var message = msg[i];
                    msg = {
                        title: message.title ? message.title : undefined,
                        closeable : true,
                        body : message.renderedMessage
                    };

                    switch(message.type) {
                        case 'ERROR':
                            AJS.messages.error(idPageMessage, msg);
                            break;
                        case 'SUCCESS':
                            AJS.messages.success(idPageMessage, msg);
                            break;
                        case 'INFO':
                            AJS.messages.info(idPageMessage, msg);
                            break;
                        case 'HINT':
                            AJS.messages.hint(idPageMessage, msg);
                            break;
                        case 'WARNING':
                            AJS.messages.warning(idPageMessage, msg);
                            break;
                        default:
                            AJS.messages.generic(idPageMessage, msg);
                    }
                }

                // force to show old messages, in case new ui is enabled
                AJS.$(idPageMessage).show();
                AJS.$('.cw-top-iframe').hide();
            }
        }
    };

    updateBadge = function (n) {
        var old = $(".awp-badge").html();
        if(n != old) {
            if(n == '0') {
                $(".awp-badge").addClass('awp-badge-0');
                n = '!';
            }
            else {
                $(".awp-badge").removeClass('awp-badge-0');
            }
            $(".awp-badge").html(n);
        }
    };

    generatePopup = function($contents, trigger, doShowPopup) {
        var $trigger = $(trigger);
        $(".awp-badge").hide();

        var loading = $('<div></div>').attr({
            'class' : 'awp-loading'
        }).html('loading');

        $contents
            .height('350px')
            .html(loading)
            .append(AWP.Templates.iframe({
                    pageId: AJS.params.pageId,
                    contextPath: AJS.Confluence.getContextPath(),
                    action: action
                }));
        doShowPopup();
    };

    getModalDialogId = function () {
        return modalDialogId;
    };

    getDialogId = function () {
        return dialogId;
    };

    isModalDialogOpen = function () {
        return modalDialogOpen;
    };

    setModalDialogOpen = function (b) {
        modalDialogOpen = b;
    };

    showModalDlg = function () {
        setModalDialogOpen(true);
        modalDialog.show();
    };

    hideDialog = function () {
        dialog.hide();
    };

    hideModalDialog = function () {
        setModalDialogOpen(false);
        modalDialog.hide();
    };

    setAction = function (s) {
        action = s;
    };

    getTask = function () {
        return task;
    };

    setTask = function (n) {
        task = n;
    };

    setApproval = function (s) {
        approval = s;
    };

    getApproval = function () {
        return approval;
    };

    /**
     * Parses the query of url and delivers a map of param-value
     * i.e. /confluence/pages/viewpage.action?pageId=983079&task=2
     *      You want the value of pageId, just do:
     *      queryString().pageId // returns 983079
     */
    queryString = function() {
        var query_string = {};
        var query = window.location.search.substring(1);
        var vars = query.split("&");
        for ( var i = 0; i < vars.length; i++) {
            var pair = vars[i].split("=");
            // If first entry with this name
            if (typeof query_string[pair[0]] === "undefined") {
                query_string[pair[0]] = pair[1];
                // If second entry with this name
            } else if (typeof query_string[pair[0]] === "string") {
                var arr = [ query_string[pair[0]], pair[1] ];
                query_string[pair[0]] = arr;
                // If third or later entry with this name
            } else {
                query_string[pair[0]].push(pair[1]);
            }
        }
        return query_string;
    };

    /**
     * @param dlgId Element ID that will bind the inline dialog of workflow
     * @param modalDlgId Element ID that will bind the modal dialog of "add a workflow"
     */
    init = function (dlgId, modalDlgId) {
        AJS.log('[AWP] init embedded.workflow');
        // set ids. we'll need it later
        dialogId = dlgId;
        modalDialogId = modalDlgId;

        var dialogSelector = '#' + dialogId;

        // in edit page mode (RTE) we want to open directly the task tab
        $("#rte-button-adhoctasks").click(function () {
            AWP.Embedded.Workflow.setAction('addtask');
        });

        // Dialog at page and edit page (RTE)
        dialog = AJS.InlineDialog($(dialogSelector + ',#rte-button-adhoctasks'), "adhocWorkflowsDialogPopup", generatePopup, {
            width: 420,
            offsetY: 17,
            hideDelay: null,
            hideCallback: function() {
                $(".awp-badge").show('fast');
            },
            initCallback: function () {
                setModalDialogOpen(false);
            }
        });

        modalDialog = new AJS.Dialog({
                width: 350,
                height: 220,
                id: modalDialogId,
                closeOnOutsideClick: true
        });

        // moving to the right for confluence versions < 5.7
        if (AJS.params.buildNumber < 5700){
            $(dialogSelector).parents("li").appendTo($("#navigation ul.ajs-menu-bar"));
        }
        // Refined wiki compatible
        if ($("#rw_page_toolbar").length > 0) {
            // Remove label name from awp button
            $(dialogSelector)
                .find("span")
                    .html("")
                        .end()
                // convert to RW button style
                .addClass("rw_icon_button")
                    .removeClass("aui-button")
                        // move it to page toolbar area
                        .prependTo($("#rw_page_toolbar"));
        }
        // fallback compatibility
        else {
            if ($(".rw_custom_section").length > 0) {
                $(dialogSelector + " span").html("");
                $(dialogSelector).addClass("rw_icon_button").prependTo($("#rw_page_right .rw_custom_section"));
            }
        }
    };

    /**
     * at 5.0, edit page mode (RTE) is invoked without refreshing page,
     * so we have to reinitialize trigger and dialog
     */
    var initRTE = function () {
        var idTaskBtn = 'rte-button-adhoctasks';
        $('#' + idTaskBtn).click(function () {
            AWP.Embedded.Workflow.setAction('addtask');
        });

        // Dialog at page and edit page (RTE)
        dialog = AJS.InlineDialog($('#' + idTaskBtn), "adhocWorkflowsDialogPopup", generatePopup, {
            width: 420,
            offsetY: 17,
            hideDelay: null,
            hideCallback: function() {
                $(".awp-badge").show('fast');
            },
            initCallback: function () {
                setModalDialogOpen(false);
            }
        });
    };

    AJS.toInit(function ($) {
        // Workflows Weekly Statistics
        var params = AJS.params;
        if(params.workflowStatsUAKey) {
            // initialize analytics
            AWP.Analytics.init(params.workflowStatsUAKey, 'AWP', 'Comala Workflows');

            // darkfeatures settings enabled
            if(params.workflowDarkFeaturesEnabled){
                var darkFeaturesEnabled = eval(params.workflowDarkFeaturesEnabled);
                darkFeaturesEnabled.forEach(function(df) {
                    AWP.Analytics.trackEvent("weekly stats", "darkfeature", df.darkFeature, df.enabled ? 1 : 0);
                });
            }

            // License type and size statistics
            if(params.workflowStatsLicenseType || params.workflowStatsLicenseSize) {
                AWP.Analytics.trackEvent("Space Workflows Usage - " + params.workflowStatsLicenseType, params.workflowStatsLicenseSize, params.workflowStatsSpaceUsage);
            }

            // Confluence version vs. Workflows version statistics
            if(params.workflowStatsConfluenceVersion || params.workflowStatsWorkflowsVersion){
                AWP.Analytics.trackEvent("Confluence-Workflows versions", params.workflowStatsConfluenceVersion, params.workflowStatsWorkflowsVersion);
            }

            // Installed Plugins Statistics
            if(params.workflowStatsInstalledPlugins){
                var installedStatsJson = eval(params.workflowStatsInstalledPlugins);
                installedStatsJson.forEach(function(plugin) {
                    AWP.Analytics.trackEvent("Installed Plugin Stats - "+plugin.statParam,
                    plugin.isInstalled,
                    params.workflowStatsLicenseType + " - " + params.workflowStatsLicenseSize);
                });
            }
        }
    });

    return {
        setState: setState,
        setMessages: setMessages,
        setAction: setAction,
        updateBadge: updateBadge,

        // dialog
        dialog: dialog,
        getDialogId: getDialogId,
        hideDialog: hideDialog,

        // modal dialog
        modalDialog: modalDialog,
        getModalDialogId: getModalDialogId,
        isModalDialogOpen: isModalDialogOpen,
        setModalDialogOpen: setModalDialogOpen,
        showModalDlg: showModalDlg,
        hideModalDialog: hideModalDialog,

        // tasks
        setTask: setTask,
        getTask: getTask,

        // approvals
        setApproval: setApproval,
        getApproval: getApproval
    };
})(AJS, AJS.$);
