/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var AWP = AWP || {};
AWP.View = AWP.View || {};

/**
 * @since 1.3
 **/
AWP.View.AutocompleteUser = Backbone.View.extend({
    template : AWP.Templates.autocompleteDropdown,
    changed: true,
    active: false,
    delay: 350, //ms
    _timer: null,

    events: {
        "keyup": "search",
        "keypress": "press"
    },

    initialize: function () {
        this.assignees = new AWP.Collection.AutocompleteUsers();

        this.assignees.request = {
            "max-results": 15,
            users: this.options.filterUsers,
            groups: this.options.filterGroups,
            excludedUsers: this.options.filterExcludedUsers
        };

        // this.$el is the input
        this.$el
            .on("focus", _.bind(this.search, this))
            .on("blur", _.bind(this.hide, this));
    },
    
    render: function() {
        this.$el.after(this.template());
        this.$list = AJS.$('.awp-dd-list', this.$el.parent());
        this.$dd = AJS.$('.awp-dd', this.$el.parent());
        this.$dd
            .on("mouseover", _.bind(this.mouseover, this))
            .on("mouseout", _.bind(this.mouseout, this));
        this.hide();
        return this.el;
    },

    populate: function () {
        this.$list.html('');
        this.assignees.fetch({
            success: _.bind(this.fetchSuccess, this)
        });
    },

    search: function (e) {
        if(this._timer) {
            clearTimeout(this._timer);
        }

        switch(parseInt(e.keyCode)) {
            case 13: // enter
                // do nothing here. it's tracked at onPress
                break;
            case 38: // arrow-up
                this.navUp();
                break;
            case 40: // arrow-down
                this.navDown();
                break;            
            default:
                this._timer = setTimeout(_.bind(this._search, this), this.delay);
        }        
    },

    press: function (e) {
        // If press ENTER key over a user selection, it will pick it
        if(parseInt(e.keyCode) === 13) {
            e.preventDefault();
            this.enter();
        }
    },

    // press arrow key down
    navDown: function() {
        var target,
            $active = this.$list.find('.awp-active');

        // even if we arent focusing the input field, 
        // pressing arrow key down will invoke the list
        // if there are users loaded on list
        if(this.assignees.length > 0) {
            this.show();    
        }
        else {
            return;
        }

        // will start navigating from first user from list
        // if it reaches the last user, it will start from first all over again
        if($active.length === 0 || $active.is(':last-child')) {
            target = this.$list.find('li:first');
        }
        else {
            target = $active.next();    
            
        }

        // select next
        this.$list.find('li').removeClass('awp-active');
        AJS.$(target).addClass('awp-active');
    },

    // press arrow key up
    navUp: function() {
        var target,
            $active = this.$list.find('.awp-active');

        // even if we arent focusing the input field, 
        // pressing arrow key up will invoke the list
        // if there are users loaded on list
        if(this.assignees.length > 0) {
            this.show();    
        }
        else {
            return;
        }

        // will start navigating from last user from list
        // if it reaches the first user, it will start from last all over again
        if($active.length === 0 || $active.is(':first-child')) {
            target = this.$list.find('li:last');
        }
        else {
            target = $active.prev();
            
        }

        // select prev
        this.$list.find('li').removeClass('awp-active');
        AJS.$(target).addClass('awp-active');
    },

    enter: function () {
        var $active = this.$list.find('.awp-active');
        if($active.length > 0) {
            $active.find('a').click();
        }
    },

    _search: function () {
        var val = this.$el.val();
        if(this.assignees.request.query !== val) {
            this.assignees.request.query = val;
            this.populate();
        }
        this.show();
    },

    fetchSuccess: function (collection, response, options) {
        var self = this;
        if(response.result.length > 0) {
            _(response.result).each(function (model) {
                self.addItem(new AWP.Model.User(model));
            });
            this.show();
        }
        else {
            if(this.assignees.request.query) {
                this.addItem(new AWP.Model.User({
                    classname: "no-results",
                    title: 'No users found' // TODO i18n
                }));
                this.show();
            }
            else {
                this.hide();
            }
        }
    },

    addItem: function(item) {
        var view = new AWP.View.AutocompleteUserItem({
            model: item
        });
        view.on("select", this.selectedItem, this);
        this.$list.append(view.render().el);
    },

    show: function () {
        this.$dd.show();
    },

    hide: function (e) {
        if(!this.active) {
            this.$dd.hide();
        }
    },

    mouseover: function () {
        this.active = true;
    },

    mouseout: function () {
        this.active = false;
    },

    selectedItem: function (model) {
        this.active = false;
        this.hide();
        this.trigger("select.autocomplete-user", model);
    }
});

/**
 * @since 1.3
 **/
AWP.View.AutocompleteUserItem = Backbone.View.extend({
    tagName: "li",
    model: AWP.Model.User,
    template: AWP.Templates.autocompleteDropdownItem,

    events: {
        'click a' : "select"
    },

    render: function() {
        this.$el.html(
            this.template(this.model.toJSON())
        );
        return this;
    },

    select: function (e) {
        e.preventDefault();
        this.trigger("select", this.model);
    }
});