/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/* global Backbone, _, AWP, AJS, soydata */

(function(AJS) {
    // Backbone objects: model, collection and views assoc.
    var Parameter = Backbone.Model.extend({
        defaults : {
            mode: 'view' // view || edit
        },
        parse: function(options) {
            var extra = {};
            if (options.type == 'DURATION') {
                var duration = moment.duration(options.value);
                var durationRegex = /^P(T{1}|)\d*[SHDWMY]$/;
                var match = options.value.match(durationRegex);
                if (match) {
                    var periodType = options.value[options.value.length - 1];
                    switch (periodType) {
                        case 'S':
                            extra.valueDurationCount = duration.asSeconds(options.value);
                            extra.valueDurationPeriodName = 'seconds';
                            break;
                        case 'H':
                            extra.valueDurationCount = duration.asHours(options.value);
                            extra.valueDurationPeriodName = 'hours';
                            break;
                        case 'D':
                            extra.valueDurationCount = duration.asDays(options.value);
                            extra.valueDurationPeriodName = 'days';
                            break;
                        case 'W':
                            extra.valueDurationCount = duration.asWeeks(options.value);
                            extra.valueDurationPeriodName = 'weeks';
                            break;
                        case 'M':
                            if (match[1] === 'T') {
                                extra.valueDurationCount = duration.asMinutes(options.value);
                                extra.valueDurationPeriodName = 'minutes';
                            } else {
                                extra.valueDurationCount = duration.asMonths(options.value);
                                extra.valueDurationPeriodName = 'months';
                            }
                            break;
                        default:
                            extra.valueDurationCount = duration.asYears(options.value);
                            extra.valueDurationPeriodName = 'years';
                            break;
                    }
                }
            }
            return _(options).extend(extra);
        }
    }),
        Parameters = Backbone.Collection.extend({
            model: Parameter,
            initialize : function(attributes, options) {
                this.options = options;

                // Init Analytics
                if(AJS.params.workflowStatsUAKey) {
                    AWP.Analytics.init(AJS.params.workflowStatsUAKey, 'AWP', 'Comala Workflows');
                    AWP.Analytics.trackPage('/spaces/workflowparameters', 'Space Parameters');
                }
            },
            url: function() {
                return AJS.Confluence.getContextPath() + "/rest/cw/1/workflows/" + this.options.spaceKey + "/workflowParameter";
            }
        }),
        ParameterView = Backbone.View.extend({
            tagName: 'tr',
            className: 'cw-row',
            initialize: function() {
                this.model.on('change', _.bind(this.render, this)); // Confluence 5.4 / BB 0.9.2 compatibility
                this.template = AWP.Templates.Parameters.parameter;
            },
            events: {
                'click .cw-row__name': 'toggleDescription',
                'click .cw-inline-edit-view': 'edit',
                'keydown .value': 'enterOverride',
                'click .update': 'update',
                'click .cancel': 'cancel',
                'click .delete': 'remove'
            },
            toggleDescription: function() {
                var isCollapsed = this.$('.cw-collapsible').toggleClass('cw-collapsed').hasClass('cw-collapsed');
                this.$('.cw-handle').toggleClass('aui-iconfont-collapsed', isCollapsed).toggleClass('aui-iconfont-expanded', !isCollapsed);
                AWP.Analytics.trackEvent('workflowparam', 'toggle', (isCollapsed)? 'collapsed': 'expanded');
            },
            edit: function() {
                this.model.set('mode', 'edit');
                var $value = this.$('.value');
                $value.focus();
                $value.select();
                AWP.Analytics.trackEvent('workflowparam', 'edit');
                return false;
            },
            cancel: function() {
                this.model.set(_.pick(this.model.previousAttributes(), 'value', 'valueDurationCount', 'valudDurationPeriodName'),{silent: true});
                this.model.set('mode', 'view');
                return false;
            },
            enterOverride: function(e) {
                var code = e.keyCode || e.which;
                if (code === 13) {
                    this.$('.update').click();
                    e.preventDefault();
                    return false;
                }
            },
            update: function() {
                var view = this,
                    spinner = view.$('.cw-spinner');
                spinner.spin();
                var value = this.$('.value').val();
                if (this.$('.valueDurationCount').length && this.$('.valueDurationCount').val() && this.$('.valueDurationPeriodName').length) {
                    value = AJS.format(this.$('.valueDurationPeriodName').val(), this.$('.valueDurationCount').val())
                }
                this.model
                    .save({ value: value })
                    .done(function() {
                        view.model.set('mode', 'view');
                        spinner.spinStop();
                        AWP.Analytics.trackEvent('workflowparam', 'saved');
                    })
                    .fail(function(xhr) {
                        view._showError(AJS.I18n.getText('comalatech.awp.container.workflowparameters.post.error', xhr.responseText));
                        spinner.spinStop();
                        AWP.Analytics.trackEvent('workflowparam', 'error');
                    });
                return false;
            },
            remove: function() {
                this._showConfirmDialog(_.bind(function() {
                    var view = this,
                        spinner = view.$('.cw-spinner');
                    spinner.spin();
                    this.model.destroy()
                        .done(function() {
                            view.$el.hide('slow', function() { $(this).remove(); });
                            spinner.spinStop();
                        })
                        .fail(function(xhr) {
                            view._showError(AJS.I18n.getText('comalatech.awp.container.workflowparameters.delete.error', xhr.responseText));
                            spinner.spinStop();
                        });
                }, this));
                return false;
            },
            _showConfirmDialog: function(onConfirmed) {
                var dialog = new AJS.Dialog({
                    id: "confirm-dialog",
                    width: 400,
                    height: 200,
                    closeOnOutsideClick: true
                });
                dialog.addHeader(AJS.I18n.getText('comalatech.awp.container.workflowparameters.delete.title'));
                dialog.addPanel("panel", "<p>"+AJS.I18n.getText('comalatech.awp.container.workflowparameters.delete.confirm')+"</p>", "panel-body");
                dialog.addButton(AJS.I18n.getText('confirm.name'), function() {
                    onConfirmed();
                    dialog.hide();
                    return false;
                });
                dialog.addLink(AJS.I18n.getText('cancel.name'), function(dialog) { dialog.hide(); }, "", "#");
                //dialog.gotoPage(0);
                dialog.gotoPanel(0, 0);
                dialog.show();
            },
            render: function() {
                this.$el.toggleClass('cw-editing', this.model.get('mode') === 'edit').html(this.template({
                    model: this.model.toJSON(),
                    value: this._getValueTemplate({model: this.model.toJSON()})
                }));
                return this;
            },
            _showError: function(error) {
                AJS.messages.error({
                    title: error,
                    fadeout: true,
                    delay: 1000,
                    duration: 2000
                });
            },
            _getValueTemplate : function(context) {
                try {
                    return this._sanitizedHtml(AWP.Templates.Parameters['parameter' + this.model.get('type').toLowerCase() + this.model.get('mode')](context));
                } catch(ex) {
                    return this._sanitizedHtml(AWP.Templates.Parameters['parameter' + this.model.get('mode')](context));
                }
            },
            _sanitizedHtml: function(soyContent) {
                // Confluence 5.9 soy upgrade to 3.x
                // https://developer.atlassian.com/confdev/development-resources/preparing-for-confluence-5-9#PreparingforConfluence5.9-SOYupdate
                return soydata.VERY_UNSAFE ?
                    soydata.VERY_UNSAFE.ordainSanitizedHtml(soyContent) :
                    new soydata.SanitizedHtml(soyContent);
            }
        }),
        ParametersView = Backbone.View.extend({
            initialize: function(options) {
                this.options = options;
                this.template = AWP.Templates.Parameters.parameterlist;
            },
            render: function() {
                
                this.$el.html(this.template());
                // Prevent form submit with enter key, calling the first available button in the form which would be
                //   unnecessary... but AUI styles are based on form.aui
                this.$('form').on('keyup keypress', function(e) {
                    var code = e.keyCode || e.which;
                    if (code === 13) {
                        e.preventDefault();
                        return false;
                    }
                });
                this.$tbody = this.$('tbody');
                this.$tbody.empty();
                this.collection.each(function(model) {
                    this.$tbody.append(new ParameterView({model: model}).render().el);
                }, this);
                return this;
            }
        });

    AJS.toInit(function($) {
        // Get parameters, build collection and initialize view
        var spaceKey = AJS.Meta.get("spaceKey"),
            workflowParameters = JSON.parse(AJS.Meta.get("workflowParameters")),
            params = new Parameters(workflowParameters, { spaceKey: spaceKey, parse: true }),
            paramsView = new ParametersView({ collection: params });

        $('#workflow-parameters').html(paramsView.render().el);
    });
})(AJS);
