/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var requirejs = require || requirejs;
define(
    'cw',
    ['ajs', 'jquery', 'underscore', 'backbone'],
    function (AJS, $, _, Backbone) {
        var DOCUMENT_ORIGIN =
            document.location.origin ? document.location.origin : document.location.protocol +
                '//' + document.location.hostname +
                (document.location.port ? ':' + document.location.port : '');

        var CW = {};

        /* WRAPPERS 4 AJS & CONFLUENCE */
        CW.getParam = function (key) {
            return AJS.params[key];
        };
        CW.getMeta = function (key) {
            return AJS.Meta.get(key);
        };
        CW.getDarkFeature = function (key) {
            return AJS.DarkFeatures.isEnabled(key);
        };
        CW.message = function () {
            var type = arguments[0];
            if (['error', 'warning', 'success', 'info'].indexOf(type) === -1) {
                // prevent errors with obsolete message types
                type = 'info';
            }
            if (arguments[2]) {
                return AJS.messages[type](arguments[1], arguments[2]);
            } else {
                return AJS.messages[type](arguments[1]);
            }
        };

        CW.onEditorLoad = function (callback) {
            if (AJS.Confluence) {
                AJS.Confluence.EditorLoader.load(callback);
            } else {
                requirejs(['confluence'], function (Confluence) {
                    Confluence.EditorLoader.load(callback);
                });
            }
        };

        CW.I18n = AJS.I18n;
        CW.Labels = AJS.Labels;
        CW.format = AJS.format;
        CW.contextPath = AJS.contextPath();
        CW.toInit = AJS.toInit;
        CW.log = AJS.log;

        CW.request = $.ajax;

        /* UTILS */
        CW.licenseIsValid = function (errorMessage) {
            if (!CW.getParam('isComalaAppLicensed')) {
                var link = CW.getParam('baseUrl') + '/plugins/servlet/upm?fragment=manage/' + CW.getParam('pluginKey');
                CW.showFlag({
                    type: 'warning',
                    title: CW.I18n.getText('workflow.info.disabled'),
                    body: errorMessage || CW.I18n.getText('adhocworkflows.mp.license.unlicensed', link),
                    close: 'manual'
                });
                return false;
            }
            return true;
        };

        CW.useLocalhost = function () {
            return CW.getDarkFeature('comalatech.workflows.riojaUI.useLocalhost');
        };

        CW.getAUIVersion = function () {
            return +$('body').data('aui-version').charAt(0);
        };

        CW.getProduct = function () {
            var product = 'cw';
            if (CW.getParam('isWorkflowsLiteApp')) {
                product = 'lite';
            } else if (CW.getParam('isApprovalsApp')) {
                product = 'approvals';
            }
            return product;
        };

        CW.send = function (target, event, data) {
            let origin = !document.location.origin || document.location.origin === 'null' ?
                document.location.ancestorOrigins : document.location.origin;

            if (!origin) {
                const protocol = document.location.protocol;
                const hostname = document.location.hostname;
                const port = (document.location.port ? ':' + document.location.port : '');
                origin = protocol + '//' + hostname + port;
            }
            target.postMessage(JSON.stringify({
                protocol: 'CDM',
                event: event,
                data: data
            }), origin);
        };

        /**
         * Parses the query of url and delivers a map of param-value
         * i.e. /confluence/pages/viewpage.action?pageId=983079&task=2
         *      You want the value of pageId, just do:
         *      queryString().pageId // returns 983079
         */
        CW.queryString = function (qs) {
            qs = qs || document.location.search;
            qs = qs.split('+').join(' ');

            var params = {};
            var tokens;
            var re = /[?&]?([^=]+)=([^&]*)/g;

            while (tokens = re.exec(qs)) {
                params[CW.sanitize(decodeURIComponent(tokens[1]))] = CW.sanitize(decodeURIComponent(tokens[2]));
            }
            return params;
        };

        /**
         * Sanitize the input value provided with HTML entity encoding
         *
         * @param {string} value
         * @returns {string} value sanitized
         */
        CW.sanitize = function (value) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#x27;',
                "/": '&#x2F;',
            };
            const reg = /[&<>"'/]/ig;
            return value.replace(reg, function (match) { return (map[match]) });
        };

        /*
         * @returns True if user can administer the space OR has "confluence administration" permission OR we're in user personal space
         */
        CW.isSpaceAdmin = function () {
            return CW.getParam('cwIsSpaceAdmin') || CW.isConfluenceAdmin() ||
                (
                    CW.getParam('spaceKey') && CW.getParam('remoteUser') &&
                    CW.getParam('spaceKey') === '~' + CW.getParam('remoteUser')
                );
        };

        /*
         * @returns True if user has "confluence administration" permission.
         */
        CW.isConfluenceAdmin = function () {
            return CW.getParam('cwIsConfluenceAdmin');
        };

        // To handle PostMessage iframe <> parent using our own protocol
        CW.messageHandlers = [];
        CW.connect = function (action, callback, context) {
            if (!CW.messageHandlers[action]) {
                CW.messageHandlers[action] = [];
            }
            // note that you can register multiple callbacks for the same event,
            // but keep in mind that triggering this event will call all callbacks
            CW.messageHandlers[action].push({
                callback: callback,
                context: context
            });
        };

        window.addEventListener('message', function (event) {
            // For Chrome, the origin property is in the event.originalEvent object.
            var origin = event.origin || event.originalEvent;
            if ((origin === DOCUMENT_ORIGIN || CW.useLocalhost()) &&
                event.data.protocol && event.data.protocol === 'CDM' && CW.messageHandlers[event.data.event]) {
                var handlers = CW.messageHandlers[event.data.event];
                if (handlers && handlers.length > 0) {
                    CW.messageHandlers[event.data.event].forEach(function (handler) {
                        handler.callback.call(handler.context, event.data.data, event.source);
                    });
                }
            }
        });

        /* VIEWS */
        CW.StatefulView = Backbone.View.extend({
            defaults: {},
            options: {},
            initialize: function (options) {
                this.options = _.defaults(options || {}, this.defaults);
                _.bindAll(this, 'setState', 'getState', 'resetState', 'stateChanged', 'afterInit');
                this.state = this.getStateModel();
                this.listenTo(this.state, 'change', function (state, options) {
                    var changed = Object.keys(state.changed);
                    if (changed.length > 0) {
                        _.defer(this.stateChanged, changed, options);
                    }
                });
                this.afterInit();
            },
            getStateModel: function () {
                return new Backbone.Model(this.options);
            },
            setState: function (attr, val) {
                return typeof val !== 'undefined' ? this.state.set(attr, val) : this.state.set(attr);
            },
            unsetState: function (attr) { return this.state.unset(attr); },
            getState: function (attr) { return this.state.get(attr); },
            resetState: function () { return this.state.clear(); },
            afterInit: function () {
                // to be overriden
            },
            stateChanged: function () {
                // to be overriden
            }
        });

        CW.iframe = CW.StatefulView.extend({
            className: 'cml-iframe-wrapper',
            defaults: {
                id: null,
                url: '',
                query: {},
                absolute: false,
                customData: null,
                width: '100%',
                height: '100%',
                css: '',
                scrolling: 'no',
                onLoad: function () { }
            },
            getStateModel: function () {
                return new Backbone.Model({ loaded: false });
            },
            afterInit: function () {
                _.bindAll(this, 'load');
                this.params = _.extend({}, this.options.query, {
                    cp: CW.contextPath,
                    xdm_e: CW.getParam('baseUrl'),
                    loc: CW.getParam('userLocale') || CW.getParam('actionLocale'),
                    user: CW.getParam('remoteUser'),
                    // uakey: CW.getParam('workflowStatsUAKey'),
                    rom: CW.getMeta('access-mode') === 'READ_ONLY' ? true : false,
                    atlToken: CW.getParam('atlToken'),
                });
            },
            stateChanged: function (changed) {
                var _self = this;
                var $iframe = _self.$iframe;
                if ($iframe && changed.indexOf('loaded') !== -1) {
                    _.defer(function () {
                        var loading = !_self.getState('loaded');
                        $iframe.toggleClass('cml-loading', loading);
                        if (!loading) {
                            var customData = _self.options.customData;
                            var onLoad = _self.options.onLoad;
                            if (customData) {
                                _self.send('customData', customData);
                            }
                            onLoad();
                        }
                    });
                }
            },
            getUrl: function (query) {
                var params = _.extend({}, this.params, query || {});
                var queryString = _(Object.keys(params)).map(function (paramKey) {
                    return paramKey + '=' + encodeURIComponent(this.params[paramKey]);
                }, this).join('&');
                var baseUrl = (
                    CW.useLocalhost() ? 'http://localhost:3000/confluence' : CW.contextPath
                ) + '/plugins/servlet/workflows/rioja';
                return (this.options.absolute ? '' : baseUrl) + this.options.url + '?' + queryString;
            },
            render: function (query) {
                this.$el.html('<iframe frameborder="0" ALLOWTRANSPARENCY="true"'
                    + ' src="' + this.getUrl(query) + '"'
                    + ' id="' + this.options.id + '_iframe"'
                    + ' width="' + (_.isUndefined(this.options.width) ? '100%' : this.options.width) + '"'
                    + ' height="' + (_.isUndefined(this.options.height) ? '100%' : this.options.height) + '"'
                    + ' scrolling="' + this.options.scrolling + '"'
                    + ' class="cml-iframe' + (this.options.css ? ' ' + this.options.css : '') + ' cml-loading"'
                    + ' />'
                );
                this.$iframe = this.$el.find('iframe');
                this.$iframe.on('load', this.load);
                return this.el;
            },
            load: function (e) {
                this.setState('loaded', true);
            },
            reload: function (query) {
                this.setState('loaded', false);
                this.$iframe.attr('src', this.getUrl(query));
            },
            send: function (event, data) {
                var iframe = this.$iframe.get(0);
                if (iframe && iframe.contentWindow) {
                    CW.send(iframe.contentWindow, event, data);
                }
            }
        });

        CW.InlineDialog = CW.StatefulView.extend({
            defaults: {
                url: '',
                query: {},
                trigger: null,
                id: '',
                width: 450,
                height: 240,
                alignment: 'right middle',
                open: false,
                persistent: true,
                'responds-to': 'toggle'
            },
            tagName: 'aui-inline-dialog',
            className: 'cw-inline-dialog',
            events: {
                'aui-show': 'show',
                'aui-hide': 'hide'
            },
            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['alignment', 'open']))
            },
            afterInit: function () {
                this.attributes = _.pick(
                    this.options,
                    'id', 'alignment', 'open', 'persistent', 'responds-to', 'alignment'
                );
                this.el.setAttribute('alignment', this.options.alignment);
                this.iframe = new CW.iframe(_.extend(
                    {
                        id: this.options.id + '_iframe',
                    },
                    _.pick(this.options, 'url', 'query', 'width', 'height')
                ));
            },
            stateChanged: function (changed) {
                var el = this.el;
                var state = this.state;
                _.each(changed, function (attr) {
                    var curAttr = el.getAttribute(attr);
                    var newAttr = state.get(attr);
                    if (curAttr !== newAttr) {
                        el[attr] = newAttr;
                    }
                });
                if (changed.indexOf('open') !== -1) {
                    // whenever the state dialog is opened, the iframe should refresh
                    this.refresh();
                }
            },
            show: function () {
                if (!this.getState('open')) {
                    this.setState('open', true);
                }
            },
            hide: function () {
                if (this.getState('open')) {
                    this.setState('open', false);
                }
            },
            toggle: function () {
                this.setState('open', !this.getState('open'));
            },
            _renderIframe: function (query) {
                var $container = this.$el.find('.aui-inline-dialog-contents')
                if (!$container.length) {
                    $container = this.$el;
                }
                $container.html(this.iframe.render(query));
            },
            _isIframeAttached: function () {
                return this.$el.find('iframe').length !== 0;
            },
            refresh: function (query) {
                if (this.getState('open')) {
                    if (!this._isIframeAttached()) {
                        this._renderIframe(query);
                    } else {
                        this.iframe.reload(query);
                    }
                }
            },
            send: function (event, data) {
                if (this.getState('open')) {
                    this.iframe.send(event, data);
                }
            },
            render: function () {
                var $trigger = $(this.options.trigger);
                if ($trigger.length > 0 && !$trigger.attr('resolved')) {
                    $trigger
                        .attr('data-aui-trigger', null)
                        .attr('aria-controls', this.id);
                }
                if (this.getState('open') && !this._isIframeAttached()) {
                    this._renderIframe();
                }
                return this.el;
            }
        });

        CW.Byline = Backbone.View.extend({
            tagName: 'ul',
            className: 'cw-bylines',
            add: function (item) {
                var listItem = $('<li class="cw-byline-item" />');
                listItem.append(item.render());
                this.$el.append(listItem);
            },
            render: function () {
                return this.el;
            }
        });

        CW.BylineItem = CW.StatefulView.extend({
            // default options
            defaults: {
                displayState: [],
                // default dialog options
                alignment: 'bottom left',
                width: 475,
                height: 240,
                // default iframe options
                url: '', // if no url defined, dialog is not set
                query: {}
            },

            afterInit: function () {
                this.listenToOnce(this.model, 'sync', this.render);
                this.listenTo(this.model, 'change', this.render);
                this.listenTo(this.model, 'error', CW.errorHandler);
            },

            getStateModel: function () {
                // initialize inline dialog
                this.dialog = new CW.InlineDialog(_.extend({
                    id: this.id + '_dialog',
                    trigger: '#' + this.id + '_dialog'
                }, _.pick(this.options, 'url', 'query', 'height', 'width', 'alignment')));
                return this.dialog.state;
            },

            displayDialog: function () {
                return this.options.url !== '';
            },

            show: function (event) {
                if (event) {
                    event.preventDefault();
                }
                if (this.displayDialog()) {
                    this.dialog.show();
                }
            },

            hide: function () {
                if (this.getState('open')) {
                    this.dialog.hide();
                }
            },

            send: function (event, data) {
                if (this.getState('open')) {
                    this.dialog.send(event, data);
                }
            },

            refresh: function () {
                if (this.getState('open')) {
                    this.dialog.refresh();
                }
            },

            getClassNames: function () {
                return this.className || this.id;
            },

            getCustomStyles: function () {
                return null;
            },

            getIcon: function () {
                return '';
            },

            getDescription: function () {
                return '';
            },

            renderIcon: function (icon, description) {
                var className = 'cw-byline__icon';
                var $icon = this.$el.find('.' + className);
                if ($icon.length < 1) {
                    $icon = $('<span class="' + className + '" />');
                    this.$el.append($icon);
                }
                $icon.html(icon).attr('title', description);
            },

            renderDescription: function (description) {
                var className = 'cw-byline__description';
                var $description = this.$el.find('.' + className);
                if ($description.length < 1) {
                    $description = $('<span class="' + className + '" />');
                    this.$el.append($description);
                }
                if (this.displayDialog()) {
                    className = 'cw-byline__dialog-trigger';
                    var $link = $description.find('.' + className);
                    if ($link.length < 1) {
                        $link = $('<a class="' + className + '" href="#" aria-controls="' + this.dialog.id + '" data-aui-trigger />');
                        $description.html('').append($link);
                    }
                    $link.html(description);
                } else {
                    $description.html(description);
                }
            },

            _renderDialog: function () {
                if (this.displayDialog()) {
                    if (this.dialog.$el.parent().length === 0) {
                        this.$el.append(this.dialog.el);
                    }
                    this.dialog.render();
                } else {
                    this.dialog.remove();
                }
            },

            render: function () {
                var icon = this.getIcon();
                var description = this.getDescription();
                if (icon || description) {
                    this.renderIcon(icon, description);
                    this.renderDescription(description);
                    this._renderDialog();
                } else {
                    this.$el.html('');
                }
                this.$el.attr('class', this.getClassNames());
                this.$el.attr('style', this.getCustomStyles());
                return this.el;
            }
        });

        CW.Dialog = CW.StatefulView.extend({
            tagName: 'section',
            className: 'aui-layer aui-dialog2 cw-modal-dialog',
            defaults: {
                id: '',
                type: '',
                title: '',
                content: '',
                hint: '',
                actions: [],
                size: '',
                width: 600,
                height: 225,
                customData: null,
                closeable: true,
                absolute: false,
                path: '',
                query: {},
                open: false,
                remove: false,
                onLoad: function () { },
                onClose: function () { },
                onOpen: function () { }
            },
            events: {
                'click .aui-dialog2-footer-actions .aui-button': 'clickAction'
            },

            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['open']));
            },

            afterInit: function () {
                _.bindAll(this, 'onShow', 'onHide', 'initializeDialog');

                requirejs(['aui/dialog2'], this.initializeDialog);

                CW.connect('dialogClose', this.hide, this);
                CW.connect('dialogResize', this.resize, this);
            },

            initializeDialog: function (dialog) {
                this.dialog = dialog(this.$el);

                this.dialog.on('show', this.onShow);
                this.dialog.on('hide', this.onHide);

                if (this.options.open) {
                    this.show();
                }
            },

            show: function (query) {
                if (query) {
                    this.options.query = _.extend({}, this.options.query, query);
                }
                if (this.dialog) {
                    this.render();
                    this.dialog.show();
                    this.$el.addClass('cw-modal-dialog-opening');
                    this.setState('open', true);
                } else {
                    // aui resource has not load yet
                    // wait until dialog is initialized to open
                    this.options.open = true;
                }
                return this;
            },

            onShow: function () {
                var _self = this;
                _.defer(function () {
                    _self.$el.removeClass('cw-modal-dialog-opening');
                    _self.$el.addClass('cw-modal-dialog-open');
                    _self.center();
                    _self.options.onOpen();
                });
            },

            hide: function () {
                if (this.getState('open')) {
                    this.$el.addClass('cw-modal-dialog-closing');
                    _.delay(function (dialog) {
                        dialog.hide();
                    }, 300, this.dialog);
                }
                return this;
            },

            onHide: function () {
                this.send('dialogClosed');
                this.setState('open', false);
                this.options.onClose();
                if (this.options.remove) {
                    this.remove();
                } else {
                    this.$el.removeClass('cw-modal-dialog-closing');
                    this.$el.removeClass('cw-modal-dialog-open');
                }
            },

            toggle: function () {
                if (this.getState('open')) {
                    this.hide();
                } else {
                    this.show();
                }
            },

            resize: function (height) {
                var windowHeight = $(window).height() - 150;
                var h = Math.min(height, windowHeight);
                if (this.getState('open') && h) {
                    this.$el.css('height', h + 'px');
                    this.center();
                }
            },

            center: function () {
                // vertically center the dialog
                var dialogHeight = this.$el.height();
                var windowHeight = $(window).height();
                var top = (windowHeight - dialogHeight) / 2;
                this.$el.css('top', top + 'px');
            },

            send: function (event, data) {
                if (this.iframe) {
                    this.iframe.send(event, data);
                }
            },

            getActions: function () {
                return this.options.actions;
            },

            clickAction: function (event) {
                var $target = $(event.target);
                $.each(this.getActions(), function (i, action) {
                    if (action.id === $target.attr('id') && action.onClick) {
                        action.onClick(event);
                        return false;
                    }
                });
            },

            _renderHeader: function () {
                if (this.options.title) {
                    this.$el.append('<header class="aui-dialog2-header">' +
                        (_.isFunction(this.options.title) === true ? '<h2 class="aui-dialog2-header-main">' + this.options.title() + '</h2>' :
                            '<h2 class="aui-dialog2-header-main">' + this.options.title + '</h2>') +
                        (this.options.closeable ? '<a class="aui-dialog2-header-close">' +
                            '<span class="aui-icon aui-icon-small aui-iconfont-close-dialog" />' +
                            '</a>' : '') +
                        '</header>');
                }
            },

            _renderContent: function () {
                if (this.options.content) {
                    var $content = $('<div class="aui-dialog2-content" />');
                    $content.append(this.options.content);
                    this.$el.append($content);
                } else if (this.options.path) {
                    var iframe = new CW.iframe({
                        id: this.options.id + '_content',
                        url: this.options.path,
                        absolute: this.options.absolute,
                        customData: this.options.customData,
                        query: this.options.query,
                        width: '100%',
                        height: '100%',
                        onLoad: this.options.onLoad
                    });
                    this.iframe = iframe;
                    this.$el.append(iframe.render());
                }
            },

            _renderFooter: function () {
                // parse dialog buttons
                var actions = '';
                $.each(this.getActions(), function (i, action) {
                    actions += '<button id="' + action.id + '" class="aui-button aui-button-' + action.style + '">' +
                        action.label + '</button>';
                });
                if (this.options.hint || actions) {
                    this.$el.append('<footer class="aui-dialog2-footer">' +
                        '<div class="aui-dialog2-footer-actions"> ' + actions + '</div>' +
                        '<div class="aui-dialog2-footer-hint">' + this.options.hint + '</div>' +
                        '</footer>');
                }
            },

            render: function () {
                var width = typeof this.options.width === 'string' ? this.options.width : this.options.width + 'px';
                var height = typeof this.options.height === 'string' ? this.options.height : this.options.height + 'px';
                this.$el.attr({
                    'role': 'dialog',
                    'aria-hidden': 'true',
                    'data-aui-remove-on-hide': this.options.remove,
                    'style': 'height: ' + height + ';',
                });

                // add specific class for warning dialogs
                if (this.options.type) {
                    this.$el.addClass('aui-dialog2-' + this.options.type);
                }
                if (this.options.size) {
                    this.$el.addClass('aui-dialog2-' + this.options.size);
                } else {
                    this.$el.css('width', width);
                }

                // reset dialog's contents
                this.$el.html('');

                // append dialog header
                this._renderHeader();
                // append contents or render iframe
                this._renderContent();
                //append dialog footer
                this._renderFooter();

                return this.el;
            }
        });

        CW.Confirm = CW.Dialog.extend({
            defaults: {
                id: 'cw-confirm',
                open: true,
                type: 'warning',
                width: 400,
                height: 'auto',
                title: function () { return CW.I18n.getText('Confirm.title') },
                remove: true,
                content: '',
                hint: '',
                onAccept: function () { },
                onCancel: function () { },
                onOpen: function () { },
                onClose: function () { }
            },
            onAccept: function () {
                this.options.onAccept();
                this.hide();
            },
            onCancel: function () {
                this.options.onCancel();
                this.hide();
            },
            getActions: function () {
                return [{
                    id: 'cw-confirm-accept',
                    style: this.options.type === 'warning' ? 'danger' : 'primary',
                    label: CW.I18n.getText('Confirm.accept'),
                    onClick: _.bind(this.onAccept, this)
                }, {
                    id: 'cw-confirm-cancel',
                    style: 'link',
                    label: CW.I18n.getText('Confirm.cancel'),
                    onClick: _.bind(this.onCancel, this)
                }];
            }
        });

        CW.Flag = CW.StatefulView.extend({
            defaults: {
                type: 'info',
                close: 'auto',
                title: '',
                body: '',
                source: null,
                removeAfterClose: false,
                events: {},
            },
            events: {
                'aui-flag-close': 'closeCallback'
            },
            afterInit: function () {
                _.bindAll(this, 'closeCallback');
                this.delegateEvents(_.extend({}, this.events, this.options.events));
                this.render();
            },
            close: function () {
                this.el.close();
            },
            closeCallback: function (e) {
                if (e.target === this.el) {
                    var target = this.getState('source');
                    if (target) {
                        CW.send(target, 'messageClosed');
                    }
                    if (this.options.removeAfterClose) {
                        this.remove();
                    } else {
                        this.resetState();
                    }
                }
            },
            stateChanged: function (changed, options) {
                // display a new message only if we're not closing the one being shown AND
                // the contents of the new message are different from the previous one being shown (if any)
                if (!options.unset && (
                    changed.indexOf('title') !== -1 ||
                    changed.indexOf('body') !== -1)
                ) {
                    this.render(changed, options);
                }
            },
            render: function (changed, options) {
                var message = _.pick(this.state.attributes, ['type', 'close', 'title', 'body']);
                // display a new message only if the message contains at least a title OR a body
                if (message.title || message.body) {
                    var _self = this;
                    requirejs(['aui/flag'], function (flag) {
                        var element = flag(message);
                        _self.setElement(element);
                        $(element.parentNode).one('aui-flag-close', _self.closeCallback);
                    });
                }
            }
        });

        // provide easy access to the flagView model to render new flags
        var flagView = new CW.Flag();
        CW.showFlag = function (options) {
            return flagView.setState(options);
        }


        CW.errorHandler = function (data, status) {
            console.error("Error, check CW.errorData and CW.errorStatus");
            CW.errorData = data;
            CW.errorStatus = status;
        };

        CW.createNewEvent = function (eventName) {
            var event;
            if (typeof (Event) === 'function') {
                event = new Event(eventName);
            } else {
                // For IE
                event = document.createEvent('Event');
                event.initEvent(eventName, true, true);
            }
            return event;
        };

        CW.observeDOM = (function () {
            var MutationObserver = window.MutationObserver || window.WebKitMutationObserver;

            return function (obj, callback) {
                if (!obj || !obj.nodeType === 1) return; // validation

                if (MutationObserver) {
                    // define a new observer
                    var obs = new MutationObserver(function (mutations, observer) {
                        callback(mutations);
                    })
                    // have the observer observe foo for changes in children
                    obs.observe(obj, { childList: true, subtree: true });
                }

                else if (window.addEventListener) {
                    obj.addEventListener('DOMNodeInserted', callback, false);
                    obj.addEventListener('DOMNodeRemoved', callback, false);
                }
            }
        })();

        CW.showWorkflowTranslator = function () {
            return CW.getParam('cwShowWorkflowTranslator');
        }

        return CW;
    });

var requirejs = require || requirejs;
requirejs(['cw', 'jquery'], function (CW, $) {
    /* AJS toInit Wrapper with common implementations */
    CW.toInit(function () {
        $('body')
            // Add aui version class to body for version dependant style fixes
            .addClass('aui' + CW.getAUIVersion())
            // Add product name for product dependant style fixes
            .addClass(CW.getProduct());

        // Navigating from inside an iframe
        CW.connect('navigate', function (url) {
            document.location.href = url;
        });

        // Generic Dialogs
        CW.connect('dialog', function (data) {
            new CW.Dialog(data);
        });

        // Confirmation Dialogs
        CW.connect('confirm', function (data, source) {
            new CW.Confirm(_.extend({}, data, {
                onAccept: function () { CW.send(source, 'confirmAccept'); },
                onCancel: function () { CW.send(source, 'confirmReject'); }
            }));
        });

        // Updating current URL without triggering a window reload
        CW.connect('history', function (options) {
            var path = typeof options === 'string' ? options : options.path;
            var title = options.title || path;
            var data = options.data || title;

            if (options.type === 'replace') {
                window.history.replaceState(data, title, path);
            } else {
                window.history.pushState(data, title, path);
            }
        });

        // display flag messages
        CW.connect('message', function (data, source) {
            CW.showFlag(_(data).extend({ source: source }));
        });
    });
});
