/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var requirejs = require || requirejs;
requirejs(['cw', 'index/resources', 'jquery'], function (CW, CWIR, $) {
    /* NEW Page Workflow Manager */
    CW.toInit(function () {
        var customButton;

        // Workflow Parameters Dialog
        function ParametersDialog(parameterData) {
            var path = '/parameters/' + CW.getParam('pageId');
            var query = {
                closeStateDialog: CW.getDarkFeature('comalatech.workflows.stateDialog.close')
            };
            if (parameterData) {
                if (parameterData.type && parameterData.type !== 'parameters') {
                    path += '/' + parameterData.type;
                }
            } else {
                query.load = true;
            }
            var Dialog = new CW.Dialog({
                id: 'cw-parametersDialog',
                width: 600,
                height: 250,
                path: path,
                query: query,
                open: true,
                remove: true,
                onLoad: function () {
                    Dialog.send('parameterData', parameterData);
                }
            });
        }

        // check if pageId is available in the context,
        // otherwise status and tasks cannot be loaded
        if (+CW.getParam('pageId') && !$('body').hasClass('contenteditor')) {
            var StatusModel = new CWIR.PageStatus({ pageId: CW.getParam('pageId') });
            var TasksModel = new CWIR.Tasks({ pageId: CW.getParam('pageId') });

            // insert the container element for the bylines
            var container = $('<ul class="cw-bylines" id="cw-bylines"></ul>');
            var Byline = new CW.Byline();
            var $target = $('.page-metadata');
            if ($target.length && $target.is(':visible')) {
                // append it to page metadata, if available
                $target.find('ul:first').after(Byline.$el);
            } else {
                $target = $('.meta-data');
                if ($target.length && $target.is(':visible')) {
                    // bikit zen foundation theme
                    $target.append(Byline.$el);
                } else {
                    // other
                    $target = $('#title-text');
                    if ($target.length && $target.is(':visible')) {
                        $('#title-text').after(Byline.$el);
                    }
                }
            }

            // Byline items
            var StateBylineItem = new CWIR.StateBylineItem({
                id: 'cw-state-byline',
                model: StatusModel,
                displayStates: ['assigned'],
                url: '/state/' + CW.getParam('spaceKey') + '/' + CW.getParam('pageId'),
                query: {
                    // min chars to trigger approval user search
                    showDocumentActivity: CW.getParam('showDocumentActivity'),
                    approvalMinQuery: CW.getParam('approvalMinQuery'),
                    closeStateDialog: CW.getDarkFeature('comalatech.workflows.stateDialog.close'),
                    dateFormat: CW.getParam('cwConfluenceDateFormat'),
                    timeFormat: CW.getParam('cwConfluenceTimeFormat'),
                    dateTimeFormat: CW.getParam('cwConfluenceDateTimeFormat'),
                    locale: CW.getParam('cwConfluenceLocale'),
                    timeZone: CW.getMeta('user-timezone-offset'),
                }
            });
            var TaskBylineItem = new CWIR.TaskBylineItem({
                id: 'cw-tasks-byline',
                model: TasksModel,
                displayStates: ['assigned'],
                url: '/tasks/' + CW.getParam('pageId')
            });
            var SummaryBylineItem = new CWIR.SummaryBylineItem({
                id: 'cw-approval-byline',
                model: StatusModel
            });
            var AlternateBylineItem = new CWIR.AlternateBylineItem({
                id: 'cw-state-alternate',
                model: StatusModel
            });

            // respect the order
            Byline.add(SummaryBylineItem);
            Byline.add(StateBylineItem);
            Byline.add(TaskBylineItem);
            Byline.add(AlternateBylineItem);

            // hide tasks if workflows error
            StatusModel.on('change:errorMessage', function (model) {
                if (model.get('errorMessage')) {
                    TaskBylineItem.$el.hide();
                }
                else {
                    TaskBylineItem.$el.show();
                }
            });

            var isInitialized = false;

            StatusModel.once('sync', function () {
                // (brikit themepress seems to take a time to display .page-metadata)
                if ($('body').hasClass('brikit') && $('.brikit-page-metadata .page-metadata').length) {
                    // Move our bylines to Brikit's custom metadata line
                    $('.brikit-page-metadata .page-metadata').append(Byline.$el);
                }

                if (!CW.getParam('publishedView')) {
                    // Custom button triggers (adhocWorkflowsPopupLink)
                    // find or create the trigger button for the custom inline dialog
                    if (CW.getParam('hasFinalState')) {
                        // clone the inline dialog of the state byline
                        customButton = new CW.InlineDialog(_.extend({}, StateBylineItem.dialog.options, {
                            id: 'customButtonStateDialog',
                            trigger: '#adhocWorkflowsPopupLink, *[rel=#cwState], *[href=#cwState]'
                        }));

                        // (brikit themepress seems to take a time to display .page-metadata)
                        // For Brikit's custom buttons, insert the custom trigger
                        if ($('body').hasClass('brikit') || $('body').hasClass('zen-theme')) {
                            var target = $('#adhocWorkflowsPopupLink');
                            if (!target.length) {
                                // if #adhocWorkflowPopupLink does not exist yet, create it
                                target = $('<span id="adhocWorkflowsPopupLink" />');
                                Byline.$el.after(target);
                            }

                            $('.brikit .action-button, .zen-theme .zen-button')
                                // apply a wrapper to each button to improve dialog position
                                .wrap('<span class="cw-action-button-wrapper" />')
                                .on('mousedown', function (event) {
                                    // insert #adhocWorkflowPopupLink within the wrapper, before the button
                                    // it can't be placed inside the button because the click event propagation
                                    // is blocked by Brikit's code
                                    $(this).before(target);
                                });
                        }
                    }
                }
                isInitialized = true;
            });

            // Refresh both state and task bylines
            var fetch = function (data) {
                if (data && data.state && isInitialized) {
                    // the event that invoked this method provides with updated status data
                    StatusModel.set(StatusModel.parse(data));
                } else {
                    // fetch status data from server
                    StatusModel.fetch({ cache: false });
                }
                if (CW.getParam('isWorkflowsApp')) {
                    TasksModel.fetch({ cache: false });
                }
            };
            var refresh = function (newStateData) {
                fetch(newStateData);
                if (newStateData) {
                    // the event that invoked this method provides with updated status data
                    StateBylineItem.send('refreshState', newStateData);
                    if (customButton) {
                        customButton.send('refreshState', newStateData);
                    }
                } else {
                    // reload status dialogs
                    StateBylineItem.dialog.refresh();
                    if (customButton) {
                        customButton.refresh();
                    }
                }
            }
            CW.connect("refreshState", refresh);
            CW.connect("openStateByline", StateBylineItem.show, StateBylineItem);
            CW.connect("closeStateByline", StateBylineItem.hide, StateBylineItem);

            // tools menu > add task
            if ($('#page-addTask').length > 0) {
                // enables the task byline if hidden
                $('#page-addTask').click(function (e) {
                    e.preventDefault();
                    if (CW.licenseIsValid(StatusModel.get('errorMessage'))) {
                        TasksModel.set('hidden', false);
                        TaskBylineItem.show();
                    }
                });
            }

            StatusModel.on('change:stateName', function () {
                if (this.get('tasksAllowed') === false) {
                    // hide the menu item if the current workflow state does not allow tasks
                    $('#page-addTask').addClass('cw-inactive');
                    // if status does not allow tasks and there is none pending,
                    // hide tasks byline (in case it's displayed)
                    TasksModel.set({ hidden: TasksModel.get('pending') === 0 });
                } else {
                    $('#page-addTask').removeClass('cw-inactive');
                }
            });

            /* WORKFLOW INFO BANNER */
            // make sure we have the workflow info dialog trigger
            var $workflowInfo = $('#adhocWorkflowsInfoBanner');

            // Show / Hide the Info Banner icon
            StatusModel.on('change:showInfoBanner', function (model, show) {
                $workflowInfo.toggleClass('cw-inactive', !show);
                if (show) {
                    // Info Banner Dialog
                    var InfoBannerDialog = new CW.Dialog({
                        id: 'cw-workflowDialog',
                        width: 700,
                        height: 500,
                        path: '/workflow/' + CW.getParam('pageId'),
                        query: {
                            view: CW.getParam('publishedView') ? 'published' : 'draft'
                        }
                    });
                    // Link the Info Banner Button to the dialog
                    $workflowInfo.click(function (e) {
                        e.preventDefault();
                        if (CW.licenseIsValid(StatusModel.get('errorMessage'))) {
                            InfoBannerDialog.show();
                        }
                    });
                    CW.connect("workflowInfo", InfoBannerDialog.show, InfoBannerDialog);
                }
            });

            // Update Info Banner Tooltip with workflow name
            StatusModel.on('change:workflowName', function (model, name) {
                $workflowInfo.attr('title', name || CW.I18n.getText('workflow.info.tooltip'));
            });

            /* WORKFLOW MESSAGES */
            // Implements pageStatus-driven messages between byline and main content
            var messagesView = new CWIR.Messages({ model: StatusModel });

            // following same logic as in adhocworkflows.js
            var $content;
            // global look and feel theme
            if ($("#main-content").length) {
                $content = $("#main-content");
            }
            // documentation theme
            else if ($("#content > .wiki-content").length) {
                $content = $("#content > .wiki-content");
            }
            if ($content) {
                $content.before(messagesView.el);
            }
            if ($content && $("#workflow-page-header")) {
                $content.before($("#workflow-page-header").addClass('space-content'));
            }
            if ($content && $("#workflow-page-footer")) {
                $content.after($("#workflow-page-footer").addClass('space-content'));
            }

            /* PARAMETERS DIALOG */
            CW.connect("parameters", ParametersDialog);

            $('#page-editParameters').click(function (e) {
                e.preventDefault();
                ParametersDialog(StatusModel.get('parameters'));
            });

            /* ADD WORKFLOW */
            // tools menu > add workflow
            // opens the Workflow Picker dialog
            if (!CW.getParam('isApprovalsApp') && $('#page-addWorkflow').length > 0) {
                var canAddSpaceWorkflow = CW.getParam('cwSpaceWorkflowAllowed') && CW.isSpaceAdmin();
                var addWorkflowDialog = new CW.Dialog({
                    id: 'cw-addWorkflowDialog',
                    width: 800,
                    height: 550,
                    path: '/workflow/' + CW.getParam('spaceKey') + '/' + CW.getParam('pageId') + '/add',
                    query: canAddSpaceWorkflow && {
                        contentType: CW.getParam('contentType'),
                        contentTitle: CW.getParam('pageTitle').replace(/^\/\d{4}\/\d{2}\/\d{2}\//g, ''),
                        spaceName: CW.getParam('spaceName')
                    }
                });
                $('#page-addWorkflow').click(function (e) {
                    e.preventDefault();
                    if (CW.licenseIsValid(StatusModel.get('errorMessage'))) {
                        addWorkflowDialog.show();
                    }
                });
                // page workflow add: toggle edit and add workflow menu options
                CW.connect("workflowAdded", function () {
                    $("#page-addWorkflow, #editPageWorkflowLink, #builderPageWorkflowLink, #page-removeWorkflow")
                        .toggleClass("cw-inactive");
                    StatusModel.once('change:showParametersDialog', function (model, showParametersDialog) {
                        $("#page-editParameters").toggleClass("cw-inactive");
                        // Display Parameters Dialog after adding a workflow with parameters
                        if (showParametersDialog) {
                            ParametersDialog(model.get('parameters'));
                        }
                    });

                    // Readack: hide page read confirmation options
                    $('#addPageReadConfirmation').hide();
                }, addWorkflowDialog);
            }

            /* CAS - APPROVALS */
            if (CW.getParam('isApprovalsApp')) {
                var ApprovalsRestApi = CW.getParam('contextPath') + '/rest/approvals/latest/page/' + CW.getParam('pageId') + '/approval';

                /* Add approvals */
                if ($('#page-addApproval').length > 0) {
                    $('#page-addApproval').click(function (e) {
                        e.preventDefault();
                        if (CW.licenseIsValid(StatusModel.get('errorMessage'))) {
                            CW.request({
                                type: 'POST',
                                contentType: 'application/json',
                                url: ApprovalsRestApi,
                                cache: false,
                                success: function (data) {
                                    // TODO response is returning workflow info but not with the same schema model
                                    // being used for the page items (byline, etc.)
                                    // if we refactor the model, we can spare an extra call
                                    StatusModel.fetch();
                                    CW.showFlag({
                                        type: 'success',
                                        title: CW.I18n.getText('Approvals.applied.title'),
                                        body: CW.I18n.getText('Approvals.applied.content'),
                                        close: 'auto'
                                    });
                                    $("#page-addApproval, #page-removeApproval").toggleClass("cw-inactive");
                                }
                            });
                        }
                    });
                }
            }

            /* ERROR: INVALID LICENSE */
            StatusModel.on('error', function (model, jqXHR) {
                if (jqXHR.status === 403) {
                    StatusModel.getMessages(model);
                }
                var res = JSON.parse(jqXHR.responseText);

                // license invalid or error 400
                if (jqXHR.status === 400 && res.messages) {
                    model.set({
                        errorMessage: res.messages[0].html,
                        showInfoBanner: true,
                    });
                    // hide tasks byline
                    TasksModel.set('hidden', true);

                    // red workflow info banner
                    $workflowInfo.addClass('error').attr('title', CW.I18n.getText('workflow.info.disabled'));
                }
            });

            // check queryparams to know if coming from my comala dashboard links
            // lets wait until editor is full loaded to open any byline dialog
            var queryParams = CW.queryString();
            if (queryParams.task || queryParams.approval) {
                CW.onEditorLoad(function () {
                    // detect refering task by URL
                    if (queryParams.task) {
                        TaskBylineItem.show();
                    }
                    // detect refering approval by URL
                    else if (queryParams.approval) {
                        StateBylineItem.show();
                    }
                });
            }

            if (CW.getParam('hasWorkflow') && CW.licenseIsValid(StatusModel.get('errorMessage'))) {
                fetch();
            }
        }
    });
});
