/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

define('cra', ['cw', 'jquery', 'underscore', 'backbone'], function (CW, $, _, Backbone) {
    var CRA = {};

    var readAckInfoDialog = new CW.Dialog({
        id: 'cra-readackreaders',
        width: 800, // large
        height: 450,
        path: '/readack/' + CW.getParam('spaceKey') + '/' + CW.getParam('pageId') + '/readers',
        query: {
            hideReset: !CW.isSpaceAdmin(),
            dateFormat: CW.getMeta('date.format'),
            locale: String(CW.getParam('userLocale') || CW.getParam('actionLocale')).substring(0,2),
            timeZone: CW.getMeta('user-timezone-offset')
        }
    });

    CRA.ReadAckItem = CW.StatefulView.extend({
        // className: 'cra-readackItem-container', // alternative
        className: 'cra-read-section',

        events: {
            'click .cra-read-button': 'read',
            'click .cra-read-info': 'openReadAckInfo',
        },

        afterInit: function () {
            this.model.on('change', this.render, this);
            this.icon = new CRA.ReadAckIcon({
                model: this.model,
                el: '.cra-read-button-icon',
                pageStatus: true,
                userStatus: true
            });
        },

        read: function () {
            new CW.Confirm({
                type: 'info',
                title: CW.I18n.getText('readack.confirm.title'),
                content: CW.I18n.getText('readack.confirm.content'),
                onAccept: (function () {
                    this.model.save();
                }).bind(this)
            });
        },

        display: function () {
            // If label section available, we embed read btn aside
            if ($('#labels-section').length > 0 && !$('#labels-section').is(":hidden")) {
                $('.labels-section-content').css('float', 'right');
                $('#labels-section').prepend(this.$el);
            } else {
                // if no label n like sections are available, we embed at the bottom of the page
                $('#workflow-page-footer').append($('<div/>').css('padding', '20px 0').append(this.$el));
            }

            // if like and labels are available, we adjust them so read btn fits in between
            if ($('#likes-and-labels-container').length > 0 && !$('#likes-and-labels-container').is(":hidden")) {
                clearTimeout(this.deferDisplay);
                $('#likes-section').css('width', '33%');
                $('#labels-section').css('width', '66%');
            } else {
                // the like button is loaded async by Confluence, so we wait until is available to embed
                // our read button aside
                if (!this.deferDisplay) {
                    this.deferDisplay = setTimeout(this.display.bind(this), 500);
                }
            }
        },

        hide: function () {
            this.$el.html('');
        },

        openReadAckInfo: function () {
            readAckInfoDialog.show();
        },

        _renderReadButton: function () {
            var verb = CW.I18n.getText('readack.button.read');
            var content = '<span class="cra-read-button-icon"></span>';
            if (this.model.get('canRead')) {
                content += '<span class="cra-read-button-text">{0}</span>';
                if (this.model.get('hasRead')) {
                    verb = CW.I18n.getText('readack.button.hasRead');
                } else {
                    content = '<a class="cra-read-button" href="#read">' + content + '</a>';
                }
            }
            return CW.format(content, verb);
        },

        _renderSummary: function () {
            // No one has read and you can't either
            var summary = CW.I18n.getText('readack.summary.empty');
            var count = this.model.get('ackUsers');
            if (this.model.get('hasRead')) {
                count--;
                if (count > 0) {
                    // you and more people have read
                    var people = count > 1 ? CW.I18n.getText('readack.summary.others', count) : CW.I18n.getText('readack.summary.another');
                    var message = CW.I18n.getText('readack.summary.multiple', people);
                    if (this.model.get('canViewWorkflow')) {
                        message = '<a href="#info" class="cra-read-info">' + message + '</a>';
                    }
                    summary = message;
                } else {
                    // only you have read
                    var you = CW.I18n.getText('readack.summary.you.read');
                    if (this.model.get('canViewWorkflow')) {
                        you = '<a href="#info" class="cra-read-info">' + you + '</a>';
                    }
                    summary = you;
                }
            } else if (count > 0) {
                // one or more people have read but you
                var people = count > 1 ? CW.I18n.getText('readack.summary.others', count) : CW.I18n.getText('readack.summary.one');
                var message = count > 1 ? CW.I18n.getText('readack.summary.othersButYou', count): CW.I18n.getText('readack.summary.single');
                if (this.model.get('canViewWorkflow')) {
                    message = '<a href="#info" class="cra-read-info">' + message + '</a>';
                }
                summary = message;
            } else if (this.model.get('canRead')) {
                // You can be the first to read
                summary = CW.I18n.getText('readack.summary.canRead');
            }
            return '<span class="cra-read-summary">' + summary + '</span>';
        },

        render: function () {
            this.$el.html(this._renderReadButton() + this._renderSummary());
            this.icon.setElement(this.$el.find('.cra-read-button-icon'));
            if (this.model.get('hasReadAck')) {
                this.display();
            }
            else {
                this.hide();
            }
            return this.el;
        }
    });

    CRA.ReadAckIcon = CW.StatefulView.extend({
        defaults: {
            pageStatus: false,
            userStatus: false
        },

        afterInit: function () {
            this.model.on('change', this.render, this);
        },

        _getIconStyles: function () {
            var pageStatus = ' cra-icon-page-' + this.model.get('pageStatus');
            if (this.model.get('ackUsers') && this.model.get('pendingUsers')) {
                pageStatus += ' cra-icon-page-partial';
            }
            var userStatus = '';
            if (this.model.get('isAssigned')) {
                userStatus = ' cra-icon-user-assigned';
                if (this.model.get('hasRead')) {
                    userStatus += ' cra-icon-user-read';
                } else {
                    userStatus += ' cra-icon-user-unread';
                }
            } else {
                userStatus = ' cra-icon-user-disabled';
            }

            var classNames = 'cra-icon';
            if (this.options.pageStatus && pageStatus) {
                classNames += pageStatus;
            } if (this.options.userStatus && userStatus) {
                classNames += userStatus;
            }
            return classNames;
        },
        _getIcon: function () {
            return '<svg width="20px" height="20px" viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><style type="text/css"> #check-1 { fill: currentColor; } #check-2 { fill: inherit; } </style><g class="checks" fill-rule="nonzero"><path id="check-1" d="M9.36488089,13.0057468 L7.94672128,14.4177594 L7.2646659,13.6777048 C6.89037989,13.271591 6.91618142,12.6389519 7.32229522,12.2646659 C7.72840903,11.8903799 8.36104809,11.9161814 8.7353341,12.3222952 L9.36488089,13.0057468 Z M11.8261529,15.6777648 L11.83287,15.6850571 C12.0844831,15.4375168 12.3578375,15.1684658 12.6482536,14.8824744 C13.6572118,13.8888885 14.6661541,12.8936793 15.6077262,11.9626286 L15.6479974,11.9228059 C17.3168826,10.2724418 18.5842491,9.01130636 19.2880529,8.29776689 C19.6758856,7.90456937 20.3090356,7.90022027 20.7022331,8.28805291 C21.0954306,8.67588555 21.0997797,9.3090356 20.7119471,9.70223311 C20.0003751,10.4236483 18.729081,11.6886919 17.0542979,13.3448885 L17.0139717,13.3847656 C16.071311,14.3168927 15.0614323,15.3130256 14.0515697,16.3075021 C13.6981303,16.6555565 13.3699327,16.9785487 13.0753861,17.2682657 C12.7723912,17.566234 12.7723912,17.566234 12.7111378,17.6263766 C12.2144556,18.1230589 11.4143065,18.1230589 10.8767762,17.5969746 L10.4058392,17.085991 L11.8261529,15.6777648 Z"></path><path id="check-2" d="M4.7353341,12.3222952 C4.36104809,11.9161814 3.72840903,11.8903799 3.32229522,12.2646659 C2.91618142,12.6389519 2.89037989,13.271591 3.2646659,13.6777048 L6.87677623,17.5969746 C7.4143065,18.1230589 8.21445556,18.1230589 8.71113782,17.6263766 C8.77239125,17.566234 8.77239125,17.566234 9.07538613,17.2682657 C9.36993267,16.9785487 9.69813028,16.6555565 10.0515697,16.3075021 C11.0614323,15.3130256 12.071311,14.3168927 13.0139717,13.3847656 L13.0542979,13.3448885 C14.729081,11.6886919 16.0003751,10.4236483 16.7119471,9.70223311 C17.0997797,9.3090356 17.0954306,8.67588555 16.7022331,8.28805291 C16.3090356,7.90022027 15.6758856,7.90456937 15.2880529,8.29776689 C14.5842491,9.01130636 13.3168826,10.2724418 11.6479974,11.9228059 L11.6077262,11.9626286 C10.6661541,12.8936793 9.65721185,13.8888885 8.64825358,14.8824744 C8.35783746,15.1684658 8.08448309,15.4375168 7.83287004,15.6850571 L4.7353341,12.3222952 Z"></path></g></svg>';
        },
        render: function () {
            this.$el.html('<div class="' + this._getIconStyles() + '">' + this._getIcon() + '</div>');
            return this.el;
        }
    });

    CRA.ReadAckInfoBanner = CW.StatefulView.extend({
        events: {
            'click': 'openInfoDialog'
        },

        afterInit: function () {
            this.iconView = new CRA.ReadAckIcon({
                model: this.model,
                pageStatus: true,
                userStatus: false
            });
            this.model.on("change", this.render, this);
        },

        openInfoDialog: function (e) {
            readAckInfoDialog.show();
            e.preventDefault();
        },
        _getVisible: function () {
            return (this.model.get('hasReadAck') && this.model.get('canViewWorkflow'));
        },
        _getTooltip: function () {
            var counter = ' (' + this.model.get('ackUsers') + '/' + this.model.get('total') + ')';
            return CW.I18n.getText('readack.info.tooltip') + counter;
        },
        render: function () {
            this.$el.html(this.iconView.render())
                // show / hide info banner button
                .toggleClass('cw-inactive', !this._getVisible())
                .attr('title', this._getTooltip());
            return this.el;
        }
    });

    CRA.ReadAckModel = Backbone.Model.extend({
        defaults: {
            pageId: CW.getParam('pageId'),
            pageVersion: CW.getParam('pageVersion'),
            hasReadAck: false,
            ackUsers: 0,
            pendingUsers: 0,
            dueDate: 0,
            total: 0,
            parsedDueDate: '',
            userStatus: '',
            canRead: false,
            hasRead: false,
            isOverdue: false,
            isAcknowledged: false,
            canViewWorkflow: false
        },
        idAttribute: "pageId",
        urlRoot: CW.contextPath + "/rest/cw/2/readack/",
        initialize: function () {
            this.once('change:hasReadAck', function (model, hasReadAck) {
                const pageActivityRegex = /viewpageactivity\.action|viewpreviousversions\.action/;
                const isPageActivity = pageActivityRegex.exec(window.location.href);
                if (hasReadAck && !isPageActivity) {
                    // alternative: embed readack item above LIKE button
                    // $('#main-content').after(RAItem.render());
                    if (model.get('canRead')) {
                        // display flag notifying this page has readack
                        CW.showFlag({
                            type: 'info',
                            title: CW.I18n.getText('readack.flag.info.title'),
                            body: CW.I18n.getText('readack.flag.info.body')
                        });
                    }
                }
            });
        },
        save: function () {
            var options = {
                success: function (model, response, options) {
                    // success message when user acknowledges
                    CW.showFlag({
                        type: 'success',
                        title: CW.I18n.getText('readack.flag.success.title'),
                        body: CW.I18n.getText('readack.flag.success.body')
                    });
                }
            };
            return Backbone.Model.prototype.save.call(this, {}, options);
        },
        parse: function (res) {
            if (res) {
                var canRead = res.userStatus === 'pending';
                var hasRead = res.userStatus === 'acknowledged';
                var extra = {
                    hasReadAck: true,
                    canRead: canRead,
                    hasRead: hasRead,
                    isAssigned: (canRead || hasRead),
                    isOverdue: res.pageStatus === 'overdue',
                    isAcknowledged: res.pageStatus === 'acknowledged',
                    total: res.pendingUsers + res.ackUsers
                };
                // read ack has due date
                if (Object.prototype.hasOwnProperty.call(res, 'dueDate')) {
                    var dueDate = new Date(res.dueDate);
                    extra.parsedDueDate = dueDate.toLocaleDateString();
                }
                return _.extend({}, this.defaults, res, extra);
            }
            return this.defaults;
        }
    });

    CRA.PageReadAckModel = Backbone.Model.extend({
        defaults: {
            pageId: null
        },
        idAttribute: 'pageId',
        urlRoot: CW.contextPath + "/rest/cw/2/readack/pageReadAck"
    });

    return CRA;
})

var requirejs = require || requirejs;
requirejs(['cw', 'cra'], function (CW, CRA) {
    CW.toInit(function () {
        // readack from workflow
        var RAModel = new CRA.ReadAckModel({ pageId: CW.getParam('pageId') });
        // Page Read Confirmation
        var PageReadack = new CRA.PageReadAckModel({ pageId: CW.getParam('pageId') });

        // readack items in the page
        new CRA.ReadAckItem({
            model: RAModel
        });
        new CRA.ReadAckInfoBanner({
            el: '#readackInfoBanner',
            model: RAModel
        });

        // update readAck data each time the workflow state changes, but only if final
        CW.connect("refreshState", function(newStateData) {
            if (newStateData && newStateData.state && newStateData.state['final']) {
                RAModel.fetch();
            }
        });

        CW.connect("refreshReadAck", function (data) {
            RAModel.set(RAModel.parse(data));
        });

        // Page Read Confirmation
        $('#addPageReadConfirmation').click(function (e) {
            e.preventDefault();
            // add page readconf
            PageReadack.save({}, {
                type: 'POST',
                success: function (model, data) {
                    // once added, we fetch the readack data to enable the readack elements
                    RAModel.set(_.extend({}, data, {
                        hasReadAck: true
                    }));

                    // update page menu options
                    $('#addPageReadConfirmation').hide();
                    $('#removePageReadConfirmation').show();
                    $('#page-addWorkflow').hide();
                    $('#page-addTask').hide();

                    // success message
                    CW.showFlag({
                        type: 'success',
                        title: CW.I18n.getText('readack.page.flag.added.title'),
                        body: CW.I18n.getText('readack.page.flag.added.description'),
                        close: 'auto'
                    });
                },
                error: function () {
                    CW.showFlag({
                        type: 'error',
                        title: CW.I18n.getText('readack.page.flag.added.error.title'),
                        body: CW.I18n.getText('readack.page.flag.added.error.description'),
                        close: 'auto'
                    });
                }
            });
        });
        $('#removePageReadConfirmation').click(function (e) {
            e.preventDefault();

            new CW.Confirm({
                title: CW.I18n.getText('readack.page.flag.remove.title'),
                content: CW.I18n.getText('readack.page.flag.remove.description'),
                onAccept: function () {
                    // remove page readconf TODO seems to not work yet...
                    PageReadack.destroy({
                        success: function () {
                            RAModel.clear();
                            $('#addPageReadConfirmation').show();
                            $('#removePageReadConfirmation').hide();
                            $('#page-addWorkflow').show();
                            $('#page-addTask').show();

                            // success message
                            CW.showFlag({
                                type: 'success',
                                title: CW.I18n.getText('readack.page.flag.removed.title'),
                                body: CW.I18n.getText('readack.page.flag.removed.description'),
                                close: 'auto'
                            });
                        }
                    });
                }
            });

            
        });

        // Check for AJS.params to perform the initial fetch
        if (CW.getParam('hasReadAck')) {
            RAModel.fetch();
            $('#addPageReadConfirmation').hide();
        }
        
        // if page readack is enabled, hide the page workflow options
        if (CW.getParam('isPageReadAck')) {
            $('#builderPageWorkflowLink').hide();
            $('#editPageWorkflowLink').hide();
            $('#page-removeWorkflow').hide();
        }

        // Cases where we hide the remove readconf option
        // no page readconf
        // it has readconf but from the workflow, not added as page readconf
        if (!CW.getParam('isPageReadAck')) {
            $('#removePageReadConfirmation').hide();
        }
    });
});
