/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

define('index/resources',
    ['cw', 'underscore', 'backbone'],
    function (CW, _, Backbone) {
        var CWIR = {};

        /**
         * Utility to translate a default state (only if the state matches)
         * @param {String} state The state to be translated
         *
         * @returns {String} The state translated
         */
        function translatedState(state) {
            var translate = state;
            switch (translate) {
                case "Draft": translate = CW.I18n.getText('workflow.state.name.draft'); break;
                case "Published": translate = CW.I18n.getText('workflow.state.name.published'); break;
                case "Rejected": translate = CW.I18n.getText('workflow.state.name.rejected'); break;
                case "Approved": translate = CW.I18n.getText('workflow.state.name.approved'); break;
                case "Expired": translate = CW.I18n.getText('workflow.state.name.expired'); break;
                case "In Approval": translate = CW.I18n.getText('workflow.state.name.inapproval'); break;
                case "Obsolete": translate = CW.I18n.getText('workflow.state.name.obsolete'); break;
                case "Review" : translate = CW.I18n.getText('workflow.state.name.review');
            }
            return translate;
        }

        /* MODELS */
        CWIR.PageStatus = Backbone.Model.extend({
            defaults: {
                errorMessage: '',
                stateName: '...'
            },

            urlRoot: function () {
                return CW.contextPath
                    + "/rest/cw/1/content/" + this.get('pageId')
                    + "/status?expand=state,approvals,actions"
                    + "&publishedView=" + (CW.getParam('publishedView') ? 'true' : 'false');
            },

            parse: function (res) {
                if (!res) {
                    return {};
                }
                var extra = {};

                var pageStatus = CW.getParam('publishedView') ? res.publishedState : res.state;
                var alternateStatus = CW.getParam('publishedView') ? res.state : res.publishedState;

                var isMostRecentState = pageStatus && (
                    !alternateStatus || (
                        alternateStatus && pageStatus.name === alternateStatus.name
                    )
                );

                if (pageStatus) {
                    extra = {
                        'showStateByline': !pageStatus.hideStates,
                        'showStateDialog': this._isActionableView(pageStatus, alternateStatus, res.actions),
                        'showInfoBanner': !pageStatus.hideStates,
                        'stateName': pageStatus.name,
                        'colour': pageStatus.colour,
                        'tasksAllowed': !!pageStatus.taskable,
                        'alternate': alternateStatus && alternateStatus.name
                    };
                }

                var showApprovalData = isMostRecentState || !CW.getParam('publishedView');

                extra.assigned = showApprovalData && this._isAssigned(res.approvals);
                extra.approvers = showApprovalData ? this._getApprovers(res.approvals) : [];
                extra.showParametersDialog = this._isInitParameters(res.actions, res.messages);
                extra.parameters = this._getParametersAction(res.actions);

                return _(res).extend(extra);
            },

            _isAssigned: function (approvals) {
                return _(approvals || []).find(function (approval) {
                    return _(approval.approvers || []).find(function (reviewer) {
                        return reviewer.user &&
                            (reviewer.user.name === CW.getParam('remoteUser')
                                && !reviewer.approved && !reviewer.rejected);
                    }) !== undefined;
                });
            },

            _getApprovers: function (approvals) {
                var approvers = {
                    approved: [],
                    rejected: [],
                    pending: [],
                    total: 0
                };
                _(approvals || []).each(function (approval) {
                    _(approval.approvers || []).each(function (approver) {
                        if (approver.user) {
                            var dest;
                            var user = approver.user.shortName || approver.user.name;
                            if (approver.approved) {
                                dest = approvers.approved;
                            } else if (approver.rejected) {
                                dest = approvers.rejected;
                            } else {
                                dest = approvers.pending;
                            }
                            if (dest.indexOf(user) < 0) {
                                dest.push(user);
                                approvers.total++;
                            }
                        }
                    });
                });
                return approvers;
            },

            _getParametersAction: function (actions) {
                return _(actions || []).find(function (action) { return action.type === 'parameters'; });
            },

            _isInitParameters: function (actions, messages) {
                var parametersAction = this._getParametersAction(actions);
                var errorMsg = _(messages || []).find(function (msg) { return msg.type.toLowerCase() === 'error'; });
                return parametersAction && parametersAction.isInit && !errorMsg;
            },

            _isActionableView: function (state, alternateState, actions) {
                if (!CW.getParam('publishedView') || (state && alternateState && state.name === alternateState.name)) {
                    return true;
                }
                return false;
            },
            getMessages: function (model) {

                CW.request({
                    type: 'GET',
                    contentType: 'application/json',
                    url: `${CW.contextPath}/rest/cw/1/content/${model.attributes.pageId}/messages`,
                    cache: false,
                    success: function (messages) {

                        if (messages && messages.length) {
                            _.each(messages, function (message) {
                                var type = ['ERROR', 'SUCCESS', 'INFO', 'WARNING'].indexOf(message.type) !== -1 ?
                                    message.type.toLowerCase() :
                                    'info';
                                var msg = {
                                    closeable: message.closeable,
                                    body: message.html
                                };
                                CW.message(type, '#rioja-page-message', msg);
                            }, this);
                        }
                    }
                });
                return '' ;
            }

        });

        CWIR.Tasks = Backbone.Model.extend({
            defaults: {
                pageId: CW.getParam('pageId'),
                pending: 0,
                total: 0,
                hidden: true
            },

            urlRoot: function () {
                return CW.contextPath + "/rest/adhocworkflows/latest/tasks/" + this.get('pageId');
            },

            parse: function (res) {
                var tasks = res.tasks ? res.tasks : [];
                var pending = _(tasks).filter(function (task) {
                    return !task.completed;
                });
                var extra = {
                    total: tasks.length,
                    pending: pending.length,
                    assigned: _.find(pending, function (task) {
                        return task.assignee && task.assignee.name === CW.getParam('remoteUser');
                    }) !== undefined
                };
                extra.hidden = tasks.length ? false : this.get('hidden');
                return _.extend(res, extra);
            }
        });

        /* VIEWS */
        CWIR.Messages = Backbone.View.extend({
            id: 'rioja-page-message',

            initialize: function (options) {
                if (this.listenTo) {
                    this.listenTo(this.model, 'change', this.render);
                } else {
                    this.model.on('change', this.render, this);
                }
            },

            render: function () {
                this.$el.empty();
                var messages = this.model.get('messages');
                if (messages && messages.length) {
                    _.each(messages, function (message) {
                        var type = ['ERROR', 'SUCCESS', 'INFO', 'WARNING'].indexOf(message.type) !== -1 ?
                            message.type.toLowerCase() :
                            'info';
                        AJS.messages[type]('#' + this.id, {
                            closeable: message.closeable,
                            title: message.title,
                            body: message.html
                        });
                    }, this);
                }
            }
        });

        CWIR.StateBylineItem = CW.BylineItem.extend({
            getClassNames: function () {
                var displayStates = this.options.displayStates;
                var model = this.model;
                var states = [this.id].concat(_(Object.keys(model.attributes))
                    .filter(function (state) {
                        return displayStates.indexOf(state) !== -1 && model.get(state);
                    })
                    .map(function (state) { return 'cw-' + state; }));
                return states.join(' ');
            },

            getCustomStyles: function () {
                var colour = this.model.get('colour');
                return colour ? 'color: ' + colour + ';' : null;
            },

            displayDialog: function () {
                return this.model.get('showStateDialog');
            },

            getDescription: function () {
                var stateName = this.model.get('stateName');
                if (CW.getParam('isApprovalsApp') || CW.getParam('isWorkflowsLiteApp')) {
                    stateName = translatedState(stateName);
                }
                return this.model.get('showStateByline') ? stateName : null;
            }
        });

        CWIR.TaskBylineItem = CW.BylineItem.extend({
            getDescription: function () {
                if (!this.model.get('hidden')) {
                    var count = this.model.get('pending');
                    switch (count) {
                        case 0:
                            return CW.I18n.getText('Tasks.link.not');
                        case 1:
                            return CW.I18n.getText('Tasks.link.single');
                        default:
                            return CW.I18n.getText('Tasks.link.default', count);
                    }
                }
            },

            getClassNames: function () {
                return (this.className || this.id) + (this.model.get('assigned') === true ? ' cw-assigned' : ''); },

            getIcon: function () {
                if (!this.model.get('hidden')) {
                    return '<span class="aui-icon aui-icon-small aui-iconfont-task aui-iconfont-editor-task" />';
                }
            },
        });

        CWIR.SummaryBylineItem = CW.BylineItem.extend({
            className: 'cw-approval-byline',
            renderIcon: function () {
                // remove icon
            },
            getDescription: function () {
                var approvers = this.model.get('approvers');
                if (approvers && approvers.total) {
                    var type = 'pending';

                    if (approvers.rejected.length) {
                        type = 'rejected';
                    } else if (approvers.approved.length) {
                        type = 'approved';
                    }

                    var users = approvers[type];
                    var rest = 0;
                    var names = users[(users.length - 1)];
                    if (users.length === 2) {
                        names = CW.I18n.getText('Approval.summary.and', users[1], users[0]);
                    } else {
                        rest = users.length - 1;
                    }

                    var others = '';
                    if (rest > 0) {
                        others = CW.I18n.getText('Approval.summary.others', rest);
                    }

                    switch (type) {
                        case 'rejected':
                            return CW.I18n.getText('Approval.summary.rejected', names, others);
                        case 'approved':
                            return CW.I18n.getText('Approval.summary.approved', names, others);
                        default:
                            return CW.I18n.getText('Approval.summary.pending');
                    }
                }
            }
        });

        CWIR.AlternateBylineItem = CW.BylineItem.extend({
            className: 'cw-state-alternate',
            renderIcon: function () {
                // remove icon
            },
            getDescription: function () {
                var pageStatus = this.model.get('stateName');
                var alternateStatus = this.model.get('alternate');
                var label = '';
                if (alternateStatus && alternateStatus !== pageStatus) {
                    var link = CW.getParam('alternateViewUrl') || '/pages/releaseview.action?pageId=' + CW.getParam('pageId');
                    if (pageStatus === alternateStatus) {
                        label = CW.getParam('publishedView') ? CW.I18n.getText('Alternate.draft') : CW.I18n.getText('Alternate.public');
                    } else {
                        if (CW.getParam('isApprovalsApp') || CW.getParam('isWorkflowsLiteApp')) {
                            alternateStatus = translatedState(alternateStatus);
                        }
                        label = CW.I18n.getText('Alternate.view', '&quot;<strong>' + alternateStatus + '</strong>&quot;');
                    }
                    return '<a href="' + CW.contextPath + link + '">' + label + '</a>';
                }
            }
        });

        return CWIR;
    }
);
