/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var AWP = AWP || {};
AWP.View = AWP.View || {};

AWP.View.WorkflowApp = Backbone.View.extend({
    model : AWP.Model.Workflow,

    /**
     * Will build the UI elements based on the workflow information 
     */
    initialize : function() {
        var addWorkflowDialogOpen = false;
        
        // approval list
        var workflow = this.model;
        var approvals = workflow.get('approvals');
        
        // Create Initial Slide with Tab Menu
        this.tabMenuView = new AWP.View.TabMenuSlide({
            header: false,
            hideStates: workflow.get('hideStates'),
            state: workflow.get('currentState'),
            stateDescription: workflow.get('stateDescription'),
            currentState: workflow.get('currentState'),
            states: workflow.get('states'),
            displayTracker: workflow.get('displayTracker') && !workflow.get('finalState'),
            stateInfo: _(workflow.toJSON()).pick(
                'hoverDescription',
                'stateDescription',
                'stateName',
                'changeExpirationDate',
                'expirationDate',
                'expirationDateValue'
            )        
        });
        
        // Request embedded Object if opened by modal dialog "add workflow",
        // or by the inline dialog
        if(parent && parent.AWP.Embedded) {
            if(parent.AWP.Embedded.Workflow.isModalDialogOpen()) {
                AWP.log('add workflow dialog open');
                addWorkflowDialogOpen = true;
            }
        }

        // ***************************
        // 0. Workflow Parameters
        // If initParameters=true then none of the other actions could be set until
        // the workflow parameters are submitted
        if(workflow.isInitParameters()) {
            AWP.log('Workflow parameters');
            this.tabMenuView.add(new AWP.View.ParamsTab(workflow.get("parameters"), true)); 
        }
        else {
            // ***************************
            // I. Main Workflow Tab, to be determined by the current workflow state
            
            // #1 Selecting Template Workflow Use case
            // There is no workflow associated to this page, therefore we are showing the list
            // of templates for the user to choose form
            if (workflow.get("templates") && ! AWP.View.WorkflowApp.isAddTask()) {
                AWP.log('1.1.1 Select template')
    
                // No workflow, no tasks no nothing
                // Show Intro state
                if(!workflow.hasStateSelection() && !addWorkflowDialogOpen) {
                    this.tabMenuView.add(new AWP.View.IntroTab({
                        model: workflow
                    }));
                }
                // if opened by modal dialog "add workflow", always show the 
                // select workflow form
                else {
                    this.tabMenuView.add(new AWP.View.SelectWorkflowTab({
                        model: workflow
                    }));
                }
                
            // #2 Select State
            } else if (workflow.hasStateSelection()) {
                AWP.log('2.1.2 Select states');
                
                if(!workflow.isHideStates()) {
                    // [2.1.2] Select states
                    // State selection is the very first thing we'd show in a workflow
                    this.tabMenuView.add(new AWP.View.ApprovalTabState({
                        model: workflow
                    }));
                }
                
            // #3 Approval - Single not Reject
            // In this case, we don't list the single approval and go directly to the dialog or list of assignees
            } else if (approvals.size() == 1 && ! approvals.at(0).get("rejected")) {
                var approvalViewOptions = {
                    model: approvals.at(0)
                };

                if (approvals.at(0).get("approvers").size() > 0) {
                    AWP.log('1.3.1')
                    // [1.3.1] Multiple approvers
                    this.tabMenuView.add(new AWP.View.ApprovalTabApprovers(approvalViewOptions));                   
                } else {
                    AWP.log('1.1.2')
                    // [1.1.2] Single Approval per State 
                    // if there is only one approval, and no approvers, jump to approval dialog                
                    this.tabMenuView.add(new AWP.View.ApprovalTabDialog(approvalViewOptions));
                    this.tabMenuView.setOptions({footer: true});
                }
    
            // #4 Multiple Approvals
            } else if (approvals.size() >= 1) {
                AWP.log('2.2.1')
                // [2.2.1] Multiple approvals
                this.tabMenuView.add(new AWP.View.ApprovalTabApprovals({
                    model: approvals
                }));
            }
    
            if(!addWorkflowDialogOpen) {
                // ***************************
                // II . TASKS TAB
                if (workflow.isShowTasks() || AWP.View.WorkflowApp.isAddTask() || workflow.isHideStates()) {
                    var taskTabView = new AWP.View.TasksTab(this.model.get("taskable"));
                    this.tabMenuView.add(taskTabView);
                }
                
                // ***************************
                // III. ACTIVITY TAB
                // not shown if it's workflow selection
                if (! workflow.get("templates")) {
                    this.tabMenuView.add(new AWP.View.ActivityTab({
                        model: new Backbone.Model() // Loading an empty Model as we wont use any specific
                    }))
                }
                
                // ***************************
                // IV. PARAMETERS TAB
                if (workflow.get("parameters")) {
                    this.tabMenuView.add(new AWP.View.ParamsTab(workflow.get("parameters"))); 
                }
            }
            else {
                if(this.tabMenuView.getTotalTabs() == 0) {
                    this.tabMenuView.add(new AWP.View.SelectWorkflowTab({
                        model: workflow
                    }));
                }

            }
        }
        this.tabMenuView.render();
        
        if (AWP.View.WorkflowApp.isAddApproval()) {
            if(this.model.get("pageStatus")) {
                AWP.View.WorkflowApp.refreshState(this.model.get("pageStatus"));    
            }            
        }
        
        // dialog is opened by the 'add task' button in the Tools dropdown
        // isHideStates = true if there are tasks but no approval workflow
        if (AWP.View.WorkflowApp.isAddTask() || workflow.isHideStates()) {
            // we invoke the task panel
            AJS.$("li.tasks a").click();
        }
    }
},{

    /**
     * Returns true if we are opening this dialog after the user selected to add a task
     * @return {*}
     */
    isAddTask: function() {
        return location.pathname.indexOf("addtask") >= 0;
    },

    /**
     * Returns true if we are opening this dialog after the user selected to add an approval
     * @return {*}
     */
    isAddApproval: function() {
        return location.pathname.indexOf("addapproval") >= 0;
    },

    refreshState : function(state) {
        window.parent.jQuery("iframe.adhoworkflows").trigger("awp.refresh",state)
    },

    refreshMessage : function(state) {
        window.parent.jQuery("iframe.adhoworkflows").trigger("awp.refreshMsg", state.messages)
    },

    closeDialog : function() {
        window.parent.jQuery("iframe.adhoworkflows").trigger("awp.close")
    },

    reloadWorkflow: function() {
        location.reload();
    }


});