/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/**
 * Uber file dealing with the legacy workflow REST API.
 *
 * NOTE: Double and triple check other options before changing this file. This file is Workflow's Backbone -
 *   and you DON'T change backbone, you just DON'T.
 */
(function ($) {

    Backbone.sync = function(method, model, options) {
        // ONLY DEVEL!!!
        if(AWP.debug && console.group) {
            console.group('sync');
            console.log('method:', method);
            console.log('model:', model);
            console.log('type:', model.type);
            console.log('options:', options);
            console.groupEnd();
        }

        switch (method) {
            case "read":
                    switch(model.type) {
                        case "workflow":
                            loadLegacyAction(model,options);
                            break;
                        case "workflow-templates":
                            loadWorkflowTemplates(model,options);
                            break;
                        case "autocomplete-users":
                            loadAutocompleteUsers(model, options);
                            break;
                        default:
                            loadLegacyTasks(model,options);
                            break;
                    }
                    break;
            case "update":
                    switch(model.type) {
                        case "approval":
                        case "approver":
                            doApprovalAction(model,options);
                            break;
                        case "workflow-template":
                            doAddWorkflow(model,options);
                            break;
                        case "task":
                            doTaskAction(model,options);
                            break;
                    }
                    break;
            case "create":
                    switch(model.type) {
                        case "task":
                            doCreateTask(model,options);
                            break;
                        case "parameters":
                            doParametersAction(model,options);
                            break;
                        case "state":
                            doState(model,options);
                            break;
                        default:
                            if (model.get("newState")) {
                                doStateAction(model,options);
                            }
                            break;
                    }
                    break;
        }
    };

/*
 * NOTE: Double and triple check other options before changing this file. This file is Workflow's Backbone -
 *   and you DON'T change backbone, you just DON'T.
 */
    var doApprovalAction = function(model, options) {
        var request = _.pick(model.attributes,"name","note","user","password","assignee");
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/approval/" + AWP.pageId + "/" + options.action;
        ajax(url, request, function(pageStatus, textStatus, jqXHR) {
            handleAjaxWorkflowResponse(pageStatus, textStatus, jqXHR);
            if(options.success) {
                options.success.call(this, pageStatus, textStatus, jqXHR);
            }
        });
    };

    var doStateAction = function(method, options) {
        var request = {
            name: method.get("newState"),
            note: method.get("note"),
            assignee: method.get("assignee")
        };
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/state/" + AWP.pageId + "/change";
        ajax(url, request, handleAjaxWorkflowResponse);
    };

    var doState = function(model, options) {
        var request = _.pick(model.attributes,"expiryDate");
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/state/" + AWP.pageId + "/" + options.action;
        ajax(url, request, options.success);
    };

    var doAddWorkflow = function(model, options) {
        var request = {
            workflowId: model.get("id") || 0,
            stateNames: options.stateNames
        };
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/workflow/" + AWP.pageId + "/add";
        ajax(url, request, handleAjaxWorkflowResponse);
    };

    var doParametersAction = function(model, options) {
        var request = {
            fields : model.get('fields')
        };

        var url = AWP.contextPath + "/rest/adhocworkflows/latest/forms/" + AWP.pageId + "/submit";
        ajax(url, request, handleAjaxParamReponse);
    };

/*
 * NOTE: Double and triple check other options before changing this file. This file is Workflow's Backbone -
 *   and you DON'T change backbone, you just DON'T.
 */
    var doTaskAction = function(model, options) {
        AWP.log('doTaskAction', model);
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/tasks/" + AWP.pageId + "/" + model.get("taskId") + "/" + options.action;
        var request = {
            names: [model.get("name")],
            assignee: model.get("assignee"),
            note: model.get("note"),
            dueDate: model.get("dueDate")
        };
        ajax(url, request, function(pageStatus, textStatus, jqXHR) {
            handleAjaxTasksUpdateReponse(pageStatus, textStatus, jqXHR);
            if(options.success) {
                options.success.call();
            }
        });

    };

    var doCreateTask = function(model, options) {
        var name = $.trim(model.get("name"));
        var request = {
            names: [name],
            assignee: model.get("assignee"),
            note: model.get("note"),
            dueDate: model.get("dueDate")
        };

        var url = AWP.contextPath + "/rest/adhocworkflows/latest/tasks/" + AWP.pageId + "/create";
        ajax(url, request, function(response, textStatus, jqXHR) {
            if(response.tasks) {
                // get last task added (the new one)
                var task = response.tasks.slice(-1)[0];
                if(task) {
                    model.set(normalizeTaskModel(task));
                    model.collection.add(model);
                }
            }
            handleAjaxTasksCreateReponse(response, textStatus, jqXHR);
        });


        function normalizeTaskModel(obj) {
            var normObj = _(obj).pick('completed', 'date', 'hint', 'lastAction');

            var canAssign = (_.findWhere(obj.actions, {id: "assign"}))? true : false,
                canComplete = (_.findWhere(obj.actions, {id: "complete"}))? true : false,
                canDelete = (_.findWhere(obj.actions, {id: "remove"}))? true : false;

            normObj = _.extend(normObj, {
                actorFullName:      obj.actor.fullName || "",
                actorName:          obj.actor.name || "",
                actorPictureUrl:    obj.actor.pictureUrl || "",
                canAssign:          canAssign,
                canComplete:        canComplete,
                canDelete:          canDelete,
                id:                 'task-' + obj.id,
                taskId:             obj.id
            });
            return normObj;
        }
    };

/*
 * NOTE: Double and triple check other options before changing this file. This file is Workflow's Backbone -
 *   and you DON'T change backbone, you just DON'T.
 */
    var loadLegacyAction = function(model,options) {
        AWP.contextPath = $('meta[name=context-path]').attr("content");
        AWP.pageId = location.pathname.match(/\/adhocworkflows\/[a-zA-Z]*\/(.*)/)[1];
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/workflow/" + AWP.pageId + "/actions";

        // admin override mode: if admin param exist, pass the param to the rest call.
        if(parent && parent.AWP.params) {
            if(parent.AWP.params.admin) {
                url += '?admin=true';
            }
        }

        $.ajax({
            type: "GET",
            cache: false,
            url: url,
            success: function (action, textStatus, jqXHR) {
                AWP.action = action;
                model.set({
                    pageId: AWP.pageId,
                    states: action.states,
                    approvals: new AWP.Collection.Approvals(makeApprovals(action.approvals)),
                    messageHtml: action.messageHtml,
                    statusCode: jqXHR.status,
                    adhoc: action.adhoc,
                    requireApprovalComment: action.requireApprovalComment
                });
                if (action.pageStatus) {
                    model.set({
                        pageStatus: action.pageStatus,
                        currentState: action.pageStatus.stateName,
                        stateName: action.pageStatus.stateName,
                        finalState: action.pageStatus.finalState,
                        stateDescription: action.pageStatus.stateDescription,
                        hoverDescription: action.pageStatus.hoverDescription,
                        changeExpirationDate: action.pageStatus.changeExpirationDate,
                        expirationDate: action.pageStatus.expirationDate,
                        expirationDateValue: action.pageStatus.expirationDateValue,
                        taskable: action.pageStatus.taskable,
                        activeTasks : action.pageStatus.activeTasks || 0,
                        hideStates : action.pageStatus.hideStates,
                        workflowStates: action.pageStatus.orderedWorkflowStates,
                        displayTracker: action.pageStatus.displayProgressTracker
                    });
                } else if (action.tasksAllowed) {
                    model.set({taskable: true});
                }
                if (action.templates) {
                    model.set({
                        templates: new AWP.Collection.WorkflowTemplates(action.templates)
                    });
                }
                if (action.spaceAdmin){
                    model.set({isSpaceAdmin: true});
                }
                if (action.spaceKey){
                    model.set({spaceKey: action.spaceKey});
                }
                if (action.inputFields && action.inputFields.length > 0) {
                    model.set({
                        parameters: action.inputFields,
                        initParameters: action.initParameters === true ? true : false
                    });
                }
                options.success();
            },
            error: function(qXHR, textStatus, errorThrown) {
                var action = JSON.parse(qXHR.responseText);

                model.set({
                    pageId: AWP.pageId,
                    states: action.states,
                    approvals: new AWP.Collection.Approvals(makeApprovals(action.approvals)),
                    messageHtml: action.messageHtml,
                    statusCode: qXHR.status
                });
                if (action.tasksAllowed) {
                    model.set({taskable: true});
                }
                if (action.templates) {
                    model.set({
                        templates: new AWP.Collection.WorkflowTemplates(action.templates)
                    });
                }

                options.error(action);
            }
        });
    };

/*
 * NOTE: Double and triple check other options before changing this file. This file is Workflow's Backbone -
 *   and you DON'T change backbone, you just DON'T.
 */
    var loadLegacyTasks = function(model, options) {
        var url = AWP.contextPath + "/rest/adhocworkflows/latest/tasks/" + AWP.pageId;
        $.ajax({
            type: "GET",
            cache: false,
            url: url,
            success: function (tasks, textStatus, jqXHR) {
                options.success(makeTasks(tasks.tasks));
            },
            error: function() {
                options.error();
                console.error("load legacy tasks - oops");
            }
        });
    };

    /*
     * Generates an array of AWP.Model.Approver from an array of Legacy approvers
     */
    var makeApprovers = function(legacyApprovers) {
        var approvers = [];
        if (! legacyApprovers) {
            return approvers;
        }
        _.each(legacyApprovers,function(legacyApprover) {
            var approver = new AWP.Model.Approver(_.pick(legacyApprover,"id","name","approved","rejected"));
            var allowedActions = [];
            _.each(legacyApprover.actions,function(action) {
                allowedActions.push(action.id);
            });
            approver.set({
                approver: new AWP.Model.User(legacyApprover.user),
                canApprove: _.contains(allowedActions,"approve"),
                canReject: _.contains(allowedActions,"reject"),
                canUnassign: _.contains(allowedActions,"delete"),
                requiresUserId: legacyApprover.signatureType == "USERID_AND_PASSWORD",
                requiresPassword: (legacyApprover.signatureType == "PASSWORD" || legacyApprover.signatureType == "USERID_AND_PASSWORD")
            });
            approvers.push(approver);
        });
        return approvers;
    };

    /*
     * Generates an array of AWP.Model.Approvals from an array of Legacy approvals
     */
    var makeApprovals = function(legacyApprovals) {
        var approvals = [];
        if (! legacyApprovals) {
            return approvals;
        }

        _.each(legacyApprovals,function(legacyApproval) {
            var approval = new AWP.Model.Approval(_.pick(legacyApproval,"id","name","shortName","hint","approved","rejected","minimumRequiredApprovers","filterUsers","filterGroups","filterExcludedUsers"));
            var allowedActions = [];
            _.each(legacyApproval.actions,function(action) {
                allowedActions.push(action.id);
            });
            approval.set({
                approvers: new AWP.Collection.Approvers(makeApprovers(legacyApproval.approvers)),
                canApprove: _.contains(allowedActions,"approve"),
                canReject: _.contains(allowedActions,"reject"),
                canAssign: _.contains(allowedActions,"assign"),
                requiresUserId: legacyApproval.signatureType == "USERID_AND_PASSWORD",
                requiresPassword: legacyApproval.signatureType == "PASSWORD" || legacyApproval.signatureType == "USERID_AND_PASSWORD"
            });
            approvals.push(approval);
        });
        return approvals;
    };

    var makeTasks = function(legacyTasks) {
        var tasks = [];
        if (! legacyTasks) {
            return tasks;
        }
        _.each(legacyTasks,function(legacyTask) {
            var task = new AWP.Model.Task(_.pick(legacyTask,"name","date","hint","lastAction","dueDate","friendlyDueDate"));
            var allowedActions = [];
            _.each(legacyTask.actions,function(action) {
                allowedActions.push(action.id);
            });

            task.set({
                "taskId" : legacyTask.id,
                id: "task-" + legacyTask.id,
                actorName: legacyTask.actor.name,
                actorFullName: legacyTask.actor.fullName,
                actorPictureUrl: legacyTask.actor.pictureUrl,
                canComplete: _.contains(allowedActions,"complete"),
                canDelete: _.contains(allowedActions,"remove"),
                canAssign: _.contains(allowedActions,"assign"),
                completed: legacyTask.completed,
                note: legacyTask.comment // All should be treated as notes, not comments!
            });

            if (legacyTask.user) {
                task.set({
                    assigneeName: legacyTask.user.name,
                    assigneeFullName: legacyTask.user.fullName,
                    assigneePictureUrl: legacyTask.user.pictureUrl
                });
            }
            tasks.push(task);
        });
        return tasks;
    };

    var loadAutocompleteUsers = function (model, options) {
        var request = model.request;
        AJS.$.ajax({
            type: "GET",
            contentType: "application/json; charset=utf-8",
            url: AWP.contextPath + model.url,
            data: AJS.$.param(request),
            success: options.success
        });
    };

    var ajax = function (url, request, responseHandler) {
        AJS.$.ajax({
            type: "POST",
            cache: false,
            contentType: "application/json; charset=utf-8",
            url: url,
            data: JSON.stringify(request),
            success: responseHandler
        });
    };

    var getErrorMessage = function(pageStatus) {
        // note that status may include more than one message,
        // but we only care if there's a error message
        for (var i = 0; i < pageStatus.messages.length; i++) {
            if (pageStatus.messages[i].type == 'ERROR') {
                return pageStatus.messages[i].renderedMessage;
            }
        }
        return undefined;
    };

    var handleAjaxWorkflowResponse = function(pageStatus, textStatus, jqXHR) {
        if (AWP.debug) {
            AJS.log("Ajax Response",pageStatus,textStatus,jqXHR);
        }
        if (jqXHR.status == 200) {
            var errorMessage = getErrorMessage(pageStatus);
            if (errorMessage) {
                var err = new AWP.View.Message({
                    type: 'error',
                    title: errorMessage,
                    closeable: true,
                    disabled: false
                });
                AJS.$('body').prepend(err.render());
            } else if (pageStatus.stateName != AWP.workflow.get("currentState") || pageStatus.messages.length > 0 ) {
                AWP.View.WorkflowApp.refreshState(pageStatus);
                AWP.View.WorkflowApp.closeDialog();
            } else {
                AWP.View.WorkflowApp.reloadWorkflow();
            }
        } else {
            AJS.log("status",jqXHR.status);
        }
    };

    var handleAjaxTasksUpdateReponse = function(tasks, textStatus, jqXHR) {
        handleAjaxTasksCreateReponse(tasks, textStatus, jqXHR);

        if (jqXHR.status == 200 && tasks.stateName) {
            AWP.View.WorkflowApp.refreshState(tasks);
        }

        if(!tasks.finalState) {
            // go back to tab menu slide
            AWP.Slides.previous('slide-tabs');
        }
        else {
            AWP.View.WorkflowApp.closeDialog();
        }
    };

    var handleAjaxTasksCreateReponse = function(tasks, textStatus, jqXHR) {
        if (AWP.debug) {
            AJS.log("Ajax Response",tasks,textStatus,jqXHR);
        }
        if (jqXHR.status === 200) {
            // reload tasks tab
            AWP.workflowApp.tabMenuView.reloadTab('activity');
            // update page message
            AWP.View.WorkflowApp.refreshMessage(tasks);
        } else {
            AJS.log("status",jqXHR.status);
            options.error(textStatus);
        }
    };

    var handleAjaxParamReponse = function(tasks, textStatus, jqXHR) {
        if (AWP.debug) {
            AJS.log("Ajax Response",tasks,textStatus,jqXHR);
        }
        if (jqXHR.status === 200) {
            AWP.View.WorkflowApp.refreshState(tasks);
            AWP.View.WorkflowApp.reloadWorkflow();
        } else {
            AJS.log("status",jqXHR.status);
            options.error(textStatus);
        }
    };

    /**
     * handles the AJAX exception and the message is shown in the .awp-slide-message div.
     * AWP.ajaxError is set for debugging
     * @param event
     * @param jqxhr
     * @param settings
     * @param exception
     */
    var errorHandler = function(event, jqXHR, settings, exception) {
        AWP.ajaxError = {
            event: event,
            jqxhr: jqXHR,
            settings: settings,
            exception: exception
        };
        var title = '',
            description = '',
            status = jqXHR.status;

        switch(status) {
            case 400:
                if(JSON.parse(jqXHR.responseText).messageHtml) {
                    title = JSON.parse(jqXHR.responseText).messageHtml;
                }
                break;
            case 500: default:
                if (jqXHR.responseText.indexOf("<html>") >= 0) {
                    title = AJS.I18n.getText("adhocworkflows.error.title");
                    description = AJS.I18n.getText("adhocworkflows.error.description");
                } else {
                    title = jqXHR.responseText;
                }
        }

        if(title !== '') {
            var err = new AWP.View.Message({
                title: title,
                description: description
            });
            AJS.$('body').prepend(err.render());
        }
        AWP.log("AWP.ajaxError", AWP.ajaxError);
    };

    $(document).ajaxError(errorHandler);

})(AJS.$);
