/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/**
 * #Slide
 *   #TabMenuSlide
 *   #DialogSlide
 *     #TaskDialogSlide
 *   #AssignSlide
 *   #ApproverSlide
 *   #SimpleApprovalSlide
 *   #SelectWorkflowSlide
 *   
 * Helpers:
 *   Slides
 */
var AWP = AWP || {};
AWP.View = AWP.View || {};


AWP.View.Slide = Backbone.View.extend({
    // backbone prop
    id: 'slide',
    className: "awp-slide",
    // custom prop
    template: AWP.Templates.slide,
    classContent: "awp-slide-body",
    
    events: {
        "click .awp-slide-back": "back",
        "click .cancel": "back",
        "click .close": "close"
    },
    
    options : {
        target: 'body',                     // Where will the slide be inserted
        caption: '',                        // Caption inserted at the caption block
        backCaption: AJS.I18n.getText("adhocworkflows.slide.back.workflow"),  // Description of header (back link)
        header: true,                       // Header is generated/shown
        footer: false,                      // Footer is generated/shown
        hidden: false                       // Slide is hidden at the beginning
    },
    
    initialize: function (customOpt) {
        this.setOptions(customOpt);
        AWP.Slides.register(this.id);
    },
    
    render : function() {
        this.renderSlide();
        return this;
    },
    
    renderSlide: function () {
        // to ensure no slide duplications,
        // if there is a slide with same ID, we remove it
        if(AJS.$('#' + this.id)) {
            AJS.$('#' + this.id).remove();
        }
        this.$el.append(this.template(this.options));
        AJS.$(this.options.target).append(this.el);
        if(this.options.hidden) {
            this.$el.hide();
        }

        this.$body = this.$('.awp-slide-body');
        this.$footer = this.$('.awp-slide-footer');
        
        return this;
    },
    
    getIdContent: function () {
        return AJS.$("#" + this.id + " ." + this.classContent);
    },
    
    close: function () {
        AWP.View.WorkflowApp.closeDialog();
    },
    
    /**
     * Go back to previous slide using the slide effect transition
     */
    back: function (event) {
        event.preventDefault();
        AWP.Slides.previous();
    },
    
    /**
     * Show current Slide using the slide effect transition
     */
    show: function () {
        AWP.Slides.next(this.el);
    },
    
    setOptions: function (customOpt) {
        if(customOpt) {
            this.options = _.extend(this.options, customOpt);
        }
    }
});


/**
 * Tab menu slide
 */
AWP.View.TabMenuSlide = AWP.View.Slide.extend({
    id: "slide-tabs",
    tabs: [],

    events: {
        "click .menu-item a"        : "setCaption",
        "click .menu-item.tasks"    : "focusAddTask",
        "tabSelect"                 : "tabSelect"
    },
    
    add: function (view) {
        this.tabs.push(view);
    },

    tabSelect: function(event,selected) {
        var tab = AJS.$(selected.tab).attr('href');
        if(tab) {
            this.updateDisplayStateAndFooter(tab.substring(1));
            this.updateContentHeight();
        }
    },
    
    render: function () {
        this.renderSlide();
        
        this.$body.addClass("aui-tabs horizontal-tabs");
        
        // tab menu
        this.$body.append(AWP.Templates.tabs({
            model: this.tabs,
            currentState: this.options.currentState
        }));

        // tab contents
        _.each(this.tabs, function(tab, i) {
            $tab = tab.render().$el;
            $tab.addClass("tabs-pane");
            // first tab is shown by default
            if(i === 0) {
                $tab.addClass("active-pane");
                this.updateDisplayStateAndFooter(tab.id);
            }
            this.$body.append($tab);
        }, this);
        
        AJS.$(this.options.target).append(this.el);
        
        // apply AUI's tab funcionality
        AJS.tabs.setup();
        // adjust position of tab menu according to number of tabs
        this.$('.tabs-menu').addClass('tabs-menu-' + this.tabs.length);
        
        // if tasks is the first active tab, focus the task field
        if(this.$('.active-pane').attr('id') == 'tasks') {
            this.focusAddTask();
        }

        this.renderFooter();
        this.updateContentHeight();

        return this;
    },

    // Footer can contain the state info with due date 
    renderFooter: function () {
        var opt = this.options;
        // State Info: Show description and due date button
        if(opt.stateInfo.stateDescription || opt.stateInfo.expirationDate || opt.stateInfo.changeExpirationDate) {
            var stateInfo = new AWP.View.StateInfo({
                model: new AWP.Model.State(this.options.stateInfo)
            });
            this.$footer
                .append(stateInfo.render())
                .addClass('awp-slide-footer-show');
        }
        else {
            this.$footer.addClass('awp-slide-footer-nostateinfo');
        }

        // Progress tracker - Workflow states
        // add state progress tracker only if the current state is one of the workflow's states
        if(opt.displayTracker && opt.states && opt.states.length > 0 && opt.states.indexOf(opt.state) >= 0) {
            var progressTracker = new AWP.ProgressTracker(opt.states, opt.state, opt.stateDescription);
            this.$footer
                .append(progressTracker.render())
                .addClass('awp-slide-footer-show');

            this.$el.addClass('awp-slide-tracker');
        }
        else {
            this.$footer.addClass('awp-slide-footer-noprogresstracker');
        }
    },

    updateDisplayStateAndFooter: function (tab) {
        // We only display the footer with the state info & progress tracker
        // if we're on a state-related tab:
        // Same with the breadcrumb on top of the title
        if(tab == 'approvals' || tab == 'select-state' || tab == 'params') {
            if(this.options.displayTracker || this.options.stateInfo.changeExpirationDate) {
                this.$('.awp-breadcrumbs').show();
                this.$footer.show();
            }
            else {
                this.$('.awp-breadcrumbs').hide();
                this.$footer.hide();
            }
        }
        else {
            this.$('.awp-breadcrumbs').hide();
            this.$footer.hide();
        }        
    },
    
    getTotalTabs: function () {
        return this.tabs.length;
    },
    
    tab: function (i) {
        return this.tabs[i];
    },
    
    /**
     * Reloads tab with id tabId
     */
    reloadTab: function(tabId) {
        var i = 0,
            tabView = '';
        
        while(tabId !== '' && i < this.getTotalTabs()) {
            if(this.tab(i).id == tabId) {
                tabId = '';
                this.tab(i).reload();
            }
            i++;
        }
    },
    
    setCaption: function (e) {
        var caption = this.$(e.currentTarget).attr("title");
        this.$('.awp-slide-title').html(caption);
    },
    
    focusAddTask: function () {
        AJS.$('.text-addtask').focus();
    },

    updateContentHeight: function () {
        // Adjust body's height to fit in between header and footer
        var slide = AJS.$('.awp-slide').height(),
            header = AJS.$('.tab-heading').outerHeight(true),
            footer = AJS.$('.awp-slide-footer').outerHeight(),
            tabPanes = this.$('.tabs-pane');

        if(!AJS.$('.awp-slide-footer').is(":visible")) {
            footer = 0;
        }

        tabPanes.each(function() {
            // adjustment of padding
            var pad = (AJS.$(this).hasClass('awp-container')) ? 20 : 1;
            // set height of each tab pane
            AJS.$(this).height(slide - header - footer - pad);
        });
    }
});


AWP.View.DialogSlide = AWP.View.Slide.extend({
    id : 'slide-dialog',
    
    initialize: function () {
        this.options.footer = true;
        this.innerContent = new AWP.View.ActivityDialog({
            model : this.model,
            buttons: this.options.buttons,
            addnote: this.options.addnote,
            note: this.options.note,
            canAssign: this.options.canAssign
        });
        
        AWP.Slides.register(this.id);
    },
    
    render : function() {
        this.renderSlide();
        this.getIdContent()
            .html(this.innerContent.render().el)
                .addClass("awp-container");
        
        return this;
    }
});

AWP.View.TaskDialogSlide = AWP.View.DialogSlide.extend({
    id : 'slide-task',
    
    initialize: function () {
        this.options.footer = true;
        this.innerContent = new AWP.View.TaskDialog({
            model : this.model,
            buttons: this.options.buttons,
            addnote: this.options.addnote,
            note: this.options.note,
            canAssign: this.options.canAssign
        });
        
        AWP.Slides.register(this.id);
    }
});


AWP.View.AssignSlide = AWP.View.Slide.extend({
    id : 'slide-assign',
    
    initialize : function() {
        this.innerContent = new AWP.View.AssignDialog({
            model : this.model
        });
        
        AWP.Slides.register(this.id);
    },
    
    render : function() {
        this.renderSlide();
        this.getIdContent()
            .html(this.innerContent.render().el)
                .addClass("awp-container");
        return this;
    },
    
    /**
     * When assigning, we want that the cancel button goes back to previous slide
     * not to close dialog
     */
    close: function () {
        this.back();
    }
});


AWP.View.ApproverSlide = AWP.View.Slide.extend({
    model: AWP.Model.Approver,
    id : 'slide-approver',
    
    initialize: function () {
        // content
        this.innerContent = new AWP.View.List({
            model: this.model.get('approvers')
        });
 
        // assign button
        if(this.model.get('canAssign')) {
            _.defer(_.bind(function() {
                this.$('.awp-container').prepend(AWP.Templates.assignBtn());
                this.$('.assignMoreApproval input').on('click', _.bind(this.assignApproval, this));
            }, this));
        }
        
        AWP.Slides.register(this.id);
    },
    
    render : function() {
        this.renderSlide();
        this.getIdContent().html(this.innerContent.render().el);
        return this;
    },

    // copied from tab-approval.js
    assignApproval: function(event) {
        event.preventDefault();
        
        // other slides
        var assignSlide = new AWP.View.AssignSlide({
            // slide prop
            model: this.model,
            hidden: true,
            caption: this.model.get("name")
        });
        assignSlide.render();
        assignSlide.show();
    }
});

AWP.View.SimpleApprovalSlide = AWP.View.Slide.extend({
    id : 'slide-simpleapproval',
    
    render: function () {
        this.renderSlide();
        
        var content = AWP.Templates.addApproval();
        this.getIdContent().html(content);
        this.$('.awp-slide-title').addClass('intro-approval-heading');
        
        this.$('.accept').on('click', this.addSimpleWorkflow);
    },
    
    addSimpleWorkflow: function (evt) {
        evt.preventDefault();
        // TODO Can improve this
        // Not sure what params to send to backbone.sync for adding a simple workflow
        // so atm i trigger the existent event from confluence page
        window.parent.jQuery('#addApprovalWorkflow').click();
    }
});

AWP.View.selectWorkflowSlide = AWP.View.Slide.extend({
    id : 'slide-selectWorkflow',
    
    render: function () {
        this.renderSlide();
        
        var slide = new AWP.View.SelectWorkflowTab({
            model: this.model
        });
        this.getIdContent().html(slide.render().el);
    }
});

AWP.View.ParamsSlide = AWP.View.Slide.extend({
    id : 'slide-params',
    
    initialize: function () {
        var fieldsModel = new AWP.Model.Parameters({
            fields: this.model.get('parameters')
        });
        
        // content
        this.innerContent = new AWP.View.Parameters({
            model: fieldsModel
        });
        
        AWP.Slides.register(this.id);
    },
    
    render : function() {
        this.renderSlide();
        this.getIdContent().html(this.innerContent.render().el);
        return this;
    }
});



/**
 * @requires jquery, jquery ui effect
 */
AWP.Slides = (function($) {
    debug = false;   // ONLY DEVEL !!
    
    duration = 350; // ms slide transition duration
    buffer = [];
    
    register = function (n) {
        buffer.push(n);
        if(debug) AWP.log('[slide+]', buffer);
    };
    
    unregister = function () {
        var s = '';
        if(buffer.length > 1) {
            s = buffer.pop();
            if(debug) AWP.log('[slide-]', buffer, '>>', s);
        }
        return s;
    };
    
    /**
     * Show a new slide using the slide effect transition
     */
    next = function (slide) {
        if(slide) {
            $footer = AJS.$('.awp-slide-footer');
            var current = _.last(buffer, 2);
            showNext(hash(current[0]), slide);
            if(debug) AWP.log('[next slide] ', current[0], ' >> ', AJS.$(slide).attr('id'));
        }
    };
    
    /**
     * Go back to previous slide using the slide effect transition
     */
    previous = function (slide) {
        var current = unregister();
        if(current) {
            var prev = _.last(buffer);
            if(slide) {
                var target = current;
                while(prev !== slide && target !== '') {
                    target = unregister();
                    prev = _.last(buffer);
                }
            }
            showPrevious(hash(current), hash(prev));
            if(debug) AWP.log('[prev slide] ', prev, ' << ', current);
        }
    };
    
    /**
     * Current slide showing
     */
    current = function () {
        return _.last(buffer);
    };
    
    /**
     * Slide fx: Left to Right
     */
    showNext = function (from, to, onComplete) {
        $(to).effect('slide', {direction: "right"}, duration, function() {
            if(onComplete) {
                onComplete.call('slow');
            }
        });
        $(from).effect('slide', {direction: "left", mode: 'hide'}, duration);
    };

    /**
     * Slide fx: Right to Left
     */
    showPrevious = function (from, to, onComplete) {
        $(to).effect('slide', {direction: "left"}, duration, function() {
            if(onComplete) {
                onComplete.call('slow');
            }
        });
        $(from).effect('slide', {direction: "right", mode: 'hide'}, duration);
    };
    
    hash = function (s) {
        return '#' + s;
    };
    
    return {
        register: register,
        next: next,
        previous: previous,
        current: current
    };
})(jQuery);