/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var AHP = AHP || {};
var _paq = window._paq || [];
define(
    'cp',
    ['ajs', 'jquery', 'underscore', 'backbone'],
    function(AJS, $, _, Backbone) {
        var DOCUMENT_ORIGIN =
            document.location.origin ? document.location.origin : document.location.protocol +
            '//' + document.location.hostname +
            (document.location.port ? ':' + document.location.port : '');

        var CP = {};

        CP.createNewEvent = function(eventName) {
            var event;
            if (typeof(Event) === 'function') {
                event = new Event(eventName);
            } else {
                // For IE
                event = document.createEvent('Event');
                event.initEvent(eventName, true, true);
            }
            return event;
        }

        // This event is the same for any of our apps
        var matomoLoadedEvent = CP.createNewEvent('MatomoLoaded');

        /* WRAPPERS 4 AJS & CONFLUENCE */
        CP.getParam = function (key) {
            return AJS.params[key];
        };
        CP.getMeta = function (key) {
            return AJS.Meta.get(key);
        };
        CP.getDarkFeature = function (key) {
            return AJS.DarkFeatures.isEnabled(key);
        };
        CP.message = function () {
            var type = arguments[0];
            if (['error', 'warning', 'success', 'info'].indexOf(type) === -1) {
                // prevent errors with obsolete message types
                type = 'info';
            }
            if (arguments[2]) {
                return AJS.messages[type](arguments[1], arguments[2]);
            } else {
                return AJS.messages[type](arguments[1]);
            }
        };

        if (AJS.Confluence) {
            CP.Confluence = AJS.Confluence;
        } else {
            requirejs(['confluence'], function (Confluence) {
                CP.Confluence = Confluence;
            });
        }

        CP.onEditorLoad = function (callback) {
            CP.Confluence.EditorLoader.load(callback);
        };

        CP.isPageGadget = function () {
            return AJS.PageGadget || (window.parent.AJS && window.parent.AJS.PageGadget);
        };

        CP.template = function(template, options) {
            if (Object.prototype.hasOwnProperty.call(AHP.Templates, template)) {
                return AHP.Templates[template](options);
            } else {
                return 'Template <strong>' + template +'</strong> not found.';
            }
        };

        CP.I18n = AJS.I18n;
        CP.Labels = AJS.Labels;
        CP.format = AJS.format;
        CP.contextPath = AJS.contextPath();
        CP.toInit = AJS.toInit;
        CP.log = AJS.log;
        CP.bind = AJS.bind;
        CP.escapeHtml = AJS.escapeHtml;

        CP.request = $.ajax;

        /* UTILS */
        CP.useLocalhost = function () {
            return CP.getDarkFeature('comalatech.workflows.riojaUI.useLocalhost');
        };

        CP.getAUIVersion = function () {
            return +$('body').data('aui-version').charAt(0);
        };

        /**
         * Parses the query of url and delivers a map of param-value
         * i.e. /confluence/pages/viewpage.action?pageId=983079&task=2
         *      You want the value of pageId, just do:
         *      queryString().pageId // returns 983079
         */
        CP.queryString = function(qs) {
            qs = qs || document.location.search;
            qs = qs.split('+').join(' ');

            var params = {};
            var tokens;
            var re = /[?&]?([^=]+)=([^&]*)/g;

            while (tokens = re.exec(qs)) {
                params[decodeURIComponent(tokens[1])] = decodeURIComponent(tokens[2]);
            }
            return params;
        };

        // To handle PostMessage iframe <> parent using our own protocol
        CP.messageHandlers = [];
        CP.connect = function(action, callback, context) {
            if (!CP.messageHandlers[action]) {
                CP.messageHandlers[action] = [];
            }
            // note that you can register multiple callbacks for the same event,
            // but keep in mind that triggering this event will call all callbacks
            CP.messageHandlers[action].push({
                callback: callback,
                context: context
            });
        };

        window.addEventListener('message', function(event) {
            // For Chrome, the origin property is in the event.originalEvent object.
            var origin = event.origin || event.originalEvent;
            if (origin === DOCUMENT_ORIGIN &&
                event.data.protocol && event.data.protocol === 'comala' && CP.messageHandlers[event.data.event]){
                var handlers = CP.messageHandlers[event.data.event];
                if (handlers && handlers.length > 0) {
                    CP.messageHandlers[event.data.event].forEach(function(handler) {
                        handler.callback.call(handler.context, event.data.data, event.source);
                    });
                }
            }
        });

        CP.send = function(target, event, data) {
            target.postMessage(JSON.stringify({
                protocol: 'comala',
                event: event,
                data: data
            }), '*');
        };


        /* VIEWS */
        CP.StatefulView = Backbone.View.extend({
            defaults: {},
            options: {},
            initialize: function (options) {
                this.options = _.defaults(options || {}, this.defaults);
                _.bindAll(this, 'setState', 'getState', 'resetState', 'stateChanged', 'afterInit');
                this.state = this.getStateModel();
                this.listenTo(this.state, 'change', function (state, options) {
                    var changed = Object.keys(state.changed);
                    if (changed.length > 0) {
                        _.defer(this.stateChanged, changed, options);
                    }
                });
                this.afterInit();
            },
            getStateModel: function () {
                return new Backbone.Model(this.options);
            },
            setState: function (attr, val) {
                return typeof val !== 'undefined' ? this.state.set(attr, val) : this.state.set(attr);
            },
            unsetState: function (attr) { return this.state.unset(attr); },
            getState: function (attr) { return this.state.get(attr); },
            resetState: function () { return this.state.clear(); },
            afterInit: function () {
                // to be overriden
            },
            stateChanged: function () {
                // to be overriden
            }
        });

        CP.iframe = CP.StatefulView.extend({
            className: 'cml-iframe-wrapper',
            defaults: {
                id: null,
                url: '',
                query: {},
                absolute: false,
                customData: null,
                width: '100%',
                height: '100%',
                css: '',
                scrolling: 'no',
                onLoad: function () { }
            },
            getStateModel: function () {
                return new Backbone.Model({ loaded: false });
            },
            afterInit: function () {
                _.bindAll(this, 'load');
                this.params = _.extend({}, this.options.query, {
                    cp: CP.contextPath,
                    // xdm_e: CP.getParam('baseUrl'), removed due to Customer's firewall issues SUPPORT-11718
                    loc: CP.getParam('userLocale') || CP.getParam('actionLocale'),
                    user: CP.getParam('remoteUser'),
                    rom: CP.getMeta('access-mode') === 'READ_ONLY' ? true : false,
                    uakey: CP.getParam('publishingStatsMatomoKey'),
                    atlToken: CP.getParam('atlToken'),
                });
            },
            stateChanged: function (changed) {
                var _self = this;
                var $iframe = _self.$iframe;
                if ($iframe && changed.indexOf('loaded') !== -1) {
                    _.defer(function () {
                        var loading = !_self.getState('loaded');
                        $iframe.toggleClass('cml-loading', loading);
                        if (!loading) {
                            var customData = _self.options.customData;
                            var onLoad = _self.options.onLoad;
                            if (customData) {
                                _self.send('customData', customData);
                            }
                            onLoad();
                        }
                    });
                }
            },
            getUrl: function (query) {
                var params = _.extend({}, this.params, query || {});
                var queryString = _(Object.keys(params)).map(function (paramKey) {
                    return paramKey + '=' + encodeURIComponent(this.params[paramKey]);
                }, this).join('&');
                var baseUrl = (
                    CP.useLocalhost() ? 'http://localhost:3000/confluence' : CP.contextPath
                ) + '/plugins/servlet/publishing/client';
                return (this.options.absolute ? '' : baseUrl) + this.options.url + '?' + queryString;
            },
            render: function (query) {
                this.$el.html('<iframe frameborder="0" ALLOWTRANSPARENCY="true"'
                    + ' src="' + this.getUrl(query) + '"'
                    + ' id="' + this.options.id + '_iframe"'
                    + ' width="' + (_.isUndefined(this.options.width) ? '100%' : this.options.width) + '"'
                    + ' height="' + (_.isUndefined(this.options.height) ? '100%' : this.options.height) + '"'
                    + ' scrolling="' + this.options.scrolling + '"'
                    + ' class="cml-iframe' + (this.options.css ? ' ' + this.options.css : '') + ' cml-loading"'
                    + ' />'
                );
                this.$iframe = this.$el.find('iframe');
                this.$iframe.on('load', this.load);
                return this.el;
            },
            load: function (e) {
                this.setState('loaded', true);
            },
            reload: function (query) {
                this.setState('loaded', false);
                this.$iframe.attr('src', this.getUrl(query));
            },
            send: function (event, data) {
                var iframe = this.$iframe.get(0);
                if (iframe && iframe.contentWindow) {
                    CP.send(iframe.contentWindow, event, data);
                }
            }
        });

        CP.InlineDialog = CP.StatefulView.extend({
            defaults: {
                url: '',
                query: {},
                content: '',
                trigger: null,
                id: '',
                width: 450,
                height: 240,
                alignment: 'right middle',
                open: false,
                persistent: null,
                'responds-to': 'toggle',
                hideOthers: true,
                onShow: function () { },
                onHide: function () { }
            },
            tagName: 'aui-inline-dialog',
            className: 'cp-inline-dialog',
            events: {
                'aui-show': 'show',
                'aui-hide': 'hide'
            },
            trigger: null,
            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['alignment', 'open']))
            },
            afterInit: function () {
                _.bindAll(this, 'show', 'hide', 'render', 'setPosition', '_updateTrigger');
                this.attributes = _.pick(
                    this.options,
                    'alignment', 'persistent', 'responds-to'
                );
                this.$el.attr(this.attributes);
                this.$el.addClass('cp-inline-dialog');

                if (this.options.url) {
                    this.iframe = new CP.iframe(_.extend(
                        {
                            id: this.options.id + '_iframe',
                        },
                        _.pick(this.options, 'url', 'query', 'width', 'height')
                    ));
                }
                this.parseTriggers();
                // update the inline dialog position when several triggers open the same
                var eventName = this.options['responds-to'] === 'toggle' ? 'mousedown' : 'mouseenter';
                $(document).on(eventName, this.options.trigger, this._updateTrigger);
                if (eventName === 'mouseenter') {
                    $(document).on('mouseout', this.options.trigger, this.hide);
                }
            },
            stateChanged: function (changed) {
                var $el = this.$el;
                var state = this.state;
                _.each(changed, function (attr) {
                    var curAttr = $el.attr(attr);
                    var newAttr = state.get(attr);
                    if (curAttr !== newAttr) {
                        $el.attr(attr, newAttr);
                    }
                });
                if (changed.indexOf('open') !== -1) {
                    // whenever the state dialog is opened, the iframe should refresh
                    if (this.getState('open')) {
                        this.onShow();
                    } else {
                        this.onHide();
                    }
                }
            },
            parseTriggers: function () {
                var $triggers = $(this.options.trigger);
                var id = this.id;
                $triggers.each(function () {
                    $trigger = $(this);
                    if (!$trigger.attr('resolved')) {
                        var $newTrigger = $trigger.clone();
                        $newTrigger
                            .attr('data-aui-trigger', true)
                            .attr('aria-controls', id);
                        $trigger.replaceWith($newTrigger);
                    }
                });
            },
            show: function () {
                if (!this.getState('open')) {
                    this.setState('open', true);
                }
            },
            onShow: function () {
                var $container = this.getContainer();
                if (this.options.hideOthers) {
                    $('aui-inline-dialog[open]').not(this.$el).removeAttr('open');
                }
                if (this._isReusable()) {
                    _.delay(this.setPosition);
                }
                if (this._isReusable() || $container.children().length === 0) {
                    this.render();
                }
                this.options.onShow(this);
            },
            hide: function () {
                if (this.getState('open')) {
                    this.setState('open', false);
                }
            },
            onHide: function () {
                this.options.onHide(this);
                if (this.options.removeOnHide) {
                    this.remove();
                }
            },
            toggle: function () {
                this.setState('open', !this.getState('open'));
            },
            getContainer: function () {
                var $container = this.$el.find('.aui-inline-dialog-contents')
                if (!$container.length) {
                    $container = this.$el;
                }
                return $container;
            },
            _isReusable: function () {
                return $(this.options.trigger).length > 1;
            },
            _updateTrigger: function (e) {
                if (!e.target.isEqualNode(this.trigger)) {
                    this.trigger = e.target;

                    if (this._isReusable() && this.getState('open')) {
                        // if the element clicked was another trigger
                        // re-open the inline dialog after the position has been updated
                        var _self = this;
                        this.$el.one('aui-hide', function () {
                            _.delay(_self.show, 300);
                        });
                    }
                }
            },
            _renderIframe: function (query) {
                var $container = this.getContainer();
                $container.html(this.iframe.render(query));
            },
            _isIframeAttached: function () {
                return this.$el.find('iframe').length !== 0;
            },
            send: function (event, data) {
                if (this.iframe && this.getState('open')) {
                    this.iframe.send(event, data);
                }
            },
            setPosition: function (trigger) {
                var $trigger = $(trigger || this.trigger);
                var tOffset = $trigger.offset();
                var tHeight = $trigger.outerHeight();
                var tWidth = $trigger.outerWidth();
                var eHeight = this.$el.outerHeight();
                var eWidth = this.$el.outerWidth();
                var alignment = this.getState('alignment').split(' ');
                var edge = alignment[0];
                var edgePos = alignment[1];
                var top = tOffset.top;
                var left = tOffset.left;
                switch (edge) {
                    case 'top':
                        top -= eHeight;
                        break;
                    case 'left':
                        left -= eWidth;
                        break;
                    case 'right':
                        left += tWidth;
                        break;
                    case 'bottom':
                        top += tHeight;
                        break;

                }
                switch (edgePos) {
                    case 'right':
                        left -= (eWidth - tWidth);
                        break;
                    case 'center':
                        left -= (eWidth - tWidth) / 2;
                        break;
                    case 'bottom':
                        top += (eHeight - tHeight);
                        break;
                    case 'middle':
                        top += (eHeight - tHeight) / 2;
                        break;
                }
                this.$el.css({
                    transform: 'translateX(' + left + 'px) translateY(' + top + 'px) translateZ(0px)'
                });
            },
            render: function () {
                var $container = this.getContainer();

                if (this.iframe) {
                    if (this.getState('open')) {
                        if (!this._isIframeAttached()) {
                            this._renderIframe(query);
                        } else {
                            this.iframe.reload(query);
                        }
                    }
                } else if (_.isFunction(this.options.content)) {
                    this.options.content(this);
                } else {
                    $container.html(this.options.content);
                }
                return this.el;
            }
        });

        CP.Dialog = CP.StatefulView.extend({
            tagName: 'section',
            className: 'aui-layer aui-dialog2 cp-modal-dialog',
            defaults: {
                id: '',
                type: '',
                title: '',
                content: '',
                hint: '',
                help: '',
                actions: [],
                size: '',
                width: 600,
                height: 225,
                customData: null,
                closeable: true,
                absolute: false,
                path: '',
                query: {},
                open: false,
                remove: false,
                onLoad: function () { },
                onClose: function () { },
                onOpen: function () { }
            },
            events: {
                'click .aui-dialog2-footer-actions .aui-button': 'clickAction'
            },

            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['open']));
            },

            afterInit: function () {
                _.bindAll(this, 'onShow', 'onHide', 'initializeDialog');

                requirejs(['aui/dialog2'], this.initializeDialog);

                CP.connect('dialogClose', this.hide, this);
                CP.connect('dialogResize', this.resize, this);
            },

            initializeDialog: function (dialog) {
                this.dialog = dialog(this.$el);

                this.dialog.on('show', this.onShow);
                this.dialog.on('hide', this.onHide);

                if (this.options.open) {
                    this.show();
                }
            },

            show: function (query) {
                if (query) {
                    this.options.query = _.extend({}, this.options.query, query);
                }
                if (this.dialog) {
                    this.render();
                    this.dialog.show();
                    this.$el.addClass('cp-modal-dialog-opening');
                    this.setState('open', true);
                } else {
                    // aui resource has not load yet
                    // wait until dialog is initialized to open
                    this.options.open = true;
                }
                return this;
            },

            onShow: function () {
                var _self = this;
                _.defer(function () {
                    _self.$el.removeClass('cp-modal-dialog-opening');
                    _self.$el.addClass('cp-modal-dialog-open');
                    _self.center();
                    _self.options.onOpen();
                });
            },

            hide: function () {
                if (this.getState('open')) {
                    this.$el.addClass('cp-modal-dialog-closing');
                    _.delay(function (dialog) {
                        dialog.hide();
                    }, 300, this.dialog);
                }
                return this;
            },

            onHide: function () {
                this.send('dialogClosed');
                this.setState('open', false);
                this.options.onClose();
                if (this.options.remove) {
                    this.remove();
                } else {
                    this.$el.removeClass('cp-modal-dialog-closing');
                    this.$el.removeClass('cp-modal-dialog-open');
                }
            },

            toggle: function () {
                if (this.getState('open')) {
                    this.hide();
                } else {
                    this.show();
                }
            },

            resize: function (height) {
                var windowHeight = $(window).height() - 150;
                var h = Math.min(height, windowHeight);
                if (this.getState('open') && h) {
                    this.$el.css('height', h + 'px');
                    this.center();
                }
            },

            center: function () {
                // vertically center the dialog
                var dialogHeight = this.$el.height();
                var windowHeight = $(window).height();
                var top = (windowHeight - dialogHeight) / 2;
                this.$el.css('top', top + 'px');
            },

            send: function (event, data) {
                if (this.iframe) {
                    this.iframe.send(event, data);
                }
            },

            getActions: function () {
                return this.options.actions;
            },

            clickAction: function (event) {
                var $target = $(event.target);
                $.each(this.getActions(), function (i, action) {
                    if (action.id === $target.attr('id') && action.onClick) {
                        action.onClick(event);
                        return false;
                    }
                });
            },

            setContent: function (content, actions, title) {
                this.options.content = content;
                if (actions) {
                    this.options.actions = actions;
                }
                if (title) {
                    this.options.title = title;
                }
            },

            _renderHeader: function () {
                if (this.options.title) {
                    this.$el.append('<header class="aui-dialog2-header">' +
                        '<h2 class="aui-dialog2-header-main">' + this.options.title + '</h2>' +
                        (this.options.closeable ? '<a class="aui-dialog2-header-close">' +
                            '<span class="aui-icon aui-icon-small aui-iconfont-close-dialog" />' +
                            '</a>' : '') +
                        '</header>');
                }
            },

            _renderContent: function () {
                if (this.options.content) {
                    var $content = $('<div class="aui-dialog2-content" />');
                    $content.append(this.options.content);
                    this.$el.append($content);
                } else if (this.options.path) {
                    var iframe = new CP.iframe({
                        id: this.options.id + '_content',
                        url: this.options.path,
                        absolute: this.options.absolute,

                        customData: this.options.customData,
                        query: this.options.query,
                        width: '100%',
                        height: '100%',
                        onLoad: this.options.onLoad
                    });
                    this.iframe = iframe;
                    this.$el.append(iframe.render());
                }
            },



            _renderFooter: function () {
                // parse dialog buttons
                var actions = '';
                $.each(this.getActions(), function (i, action) {
                    actions += '<button id="' + action.id + '" class="aui-button aui-button-' + action.style + '">' +
                        action.label + '</button>';
                });
                if (this.options.hint || actions) {
                    this.$el.append('<footer class="aui-dialog2-footer">' +   
                        this.options.help +
                        '<div class="aui-dialog2-footer-actions"> ' + actions + '</div>' +
                        '<div class="aui-dialog2-footer-hint">' + this.options.hint + '</div>' +
                        '</footer>');
                }
            },

            update: function () {
                // reset dialog's contents
                this.$el.html('');

                // append dialog header
                this._renderHeader();
                // append contents or render iframe
                this._renderContent();
                //append dialog footer
                this._renderFooter();
            },

            render: function () {
                var width = typeof this.options.width === 'string' ? this.options.width : this.options.width + 'px';
                var height = typeof this.options.height === 'string' ? this.options.height : this.options.height + 'px';
                this.$el.attr({
                    'role': 'dialog',
                    'aria-hidden': 'true',
                    'data-aui-remove-on-hide': this.options.remove
                });

                // add specific class for warning dialogs
                if (this.options.type) {
                    this.$el.addClass('aui-dialog2-' + this.options.type);
                }
                if (this.options.size) {
                    this.$el.addClass('aui-dialog2-' + this.options.size);
                } else {
                    this.$el.css({
                        width: width,
                        height: height
                    });
                }

                // reset dialog's contents
                this.$el.html('');

                // append dialog header
                this._renderHeader();
                // append contents or render iframe
                this._renderContent();
                //append dialog footer
                this._renderFooter();

                return this.el;
            }
        });

        CP.Confirm = CP.Dialog.extend({
            defaults: {
                id: 'cp-confirm',
                open: true,
                type: 'warning',
                width: 400,
                height: 'auto',
                title: function () { return CP.I18n.getText('Confirm.title') },
                remove: true,
                content: '',
                hint: '',
                onAccept: function () { },
                onCancel: function () { },
                onOpen: function () { },
                onClose: function () { }
            },
            onAccept: function () {
                this.options.onAccept();
                this.hide();
            },
            onCancel: function () {
                this.options.onCancel();
                this.hide();
            },
            getActions: function () {
                return [{
                    id: 'cp-confirm-accept',
                    style: this.options.type === 'warning' ? 'danger' : 'primary',
                    label: CP.I18n.getText('Confirm.accept'),
                    onClick: _.bind(this.onAccept, this)
                }, {
                    id: 'cp-confirm-cancel',
                    style: 'link',
                    label: CP.I18n.getText('Confirm.cancel'),
                    onClick: _.bind(this.onCancel, this)
                }];
            }
        });

        CP.Flag = CP.StatefulView.extend({
            defaults: {
                type: 'info',
                close: 'auto',
                title: '',
                body: '',
                source: null,
                removeAfterClose: false,
                events: {},
            },
            events: {
                'aui-flag-close': 'closeCallback'
            },
            afterInit: function () {
                _.bindAll(this, 'closeCallback');
                this.delegateEvents(_.extend({}, this.events, this.options.events));
                this.render();
            },
            close: function () {
                this.el.close();
            },
            closeCallback: function (e) {
                if (e.target === this.el) {
                    var target = this.getState('source');
                    if (target) {
                        CP.send(target, 'messageClosed');
                    }
                    if (this.options.removeAfterClose) {
                        this.remove();
                    } else {
                        this.resetState();
                    }
                }
            },
            stateChanged: function (changed, options) {
                // display a new message only if we're not closing the one being shown AND
                // the contents of the new message are different from the previous one being shown (if any)
                if (!options.unset && (
                    changed.indexOf('title') !== -1 ||
                    changed.indexOf('body') !== -1)
                ) {
                    this.render(changed, options);
                }
            },
            render: function (changed, options) {
                var message = _.pick(this.state.attributes, ['type', 'close', 'title', 'body']);
                // display a new message only if the message contains at least a title OR a body
                if (message.title || message.body) {
                    var _self = this;
                    requirejs(['aui/flag'], function (flag) {
                        var element = flag(message);
                        _self.setElement(element);
                        $(element.parentNode).one('aui-flag-close', _self.closeCallback);
                    });
                }
            }
        });

        // provide easy access to the flagView model to render new flags
        var flagView = new CP.Flag();
        CP.showFlag = function (options) { return flagView.setState(options); }

        CP.errorHandler = function(data, status) {
            console.error("Error, check CP.errorData and CP.errorStatus");
            CP.errorData = data;
            CP.errorStatus = status;
        };

        CP.Analytics = {
            matomoTracker: null,
            lang: null,
            url: null,
            siteId: null,
            init: function (domain, siteId, lang, section) {
                // Matomo tracking code
                this.lang = lang;
                this.siteId = siteId;
                this.url = 'https://' + domain + '/matomo.php';
                if (domain && siteId) {
                    (function () {
                        window.piwikMediaAnalyticsAsyncInit = function () {
                            // This global function definition will be overridden by the last app loaded, so
                            // the only thing we can do is trigger an event to let our components know that Matomo lib is loaded
                            try {
                                document.dispatchEvent(matomoLoadedEvent);
                            } catch (e) {
                                console.error(e);
                            }
                        }
                        var d = document, g = d.createElement('script'), s = d.getElementsByTagName('script')[0];
                        g.type = 'text/javascript';
                        g.async = true;
                        g.defer = true;
                        g.src = '//cdn.matomo.cloud/' + domain + '/matomo.js';
                        s.parentNode.insertBefore(g, s);
                    })();

                    document.addEventListener('MatomoLoaded',function (e) {
                        // Now we can init the tracker
                        CP.Analytics.initTracker(section);
                    }, false);
                }
                return this;
            },

            /**
             * To keep customer privacy, it removes page title from url as it might contain confidential info
             * @param url
             * @returns {string}
             */
            removePageNameFromUrl: function(url) {
                var index = url.indexOf("/display/");
                if (index !== -1){
                    url = url.substr(0, index + 9);
                }
                return url;
            },

            /**
             * Once we know the Matomo lib is loaded we can initialize our tracker.
             */
            initTracker: function(section) {
                if (!this.matomoTracker) {
                    this.matomoTracker = Matomo.getTracker(this.url, this.siteId);
                    if (this.lang) {
                        this.matomoTracker.setCustomVariable(1, 'plugin language', this.lang, 'visit');
                    }
                    this.matomoTracker.disableCookies();
                    this.matomoTracker.setCustomUrl('http:/' + this.removePageNameFromUrl(window.location.pathname));
                    CP.Analytics.trackPage(section);
                }
            },

            /**
             * TODO To be implemented
             * @param campaign
             * @param source
             * @param medium
             * @returns {CP.Analytics}
             */
            setCampaignParams: function (campaign, source, medium) { return this; },

            /**
             * this is still not used
             * @param title
             */
            trackPage: function (title) {
                if (this.matomoTracker) {
                    this.matomoTracker.trackPageView(title);
                }
            },

            /**
             * @param category
             * @param action
             * @param label
             * @param value
             * @param callback
             */
            trackEvent: function (category, action, label, value) {
                if (this.matomoTracker) {
                    this.matomoTracker.trackEvent(category, action, label, value);
                }
            },

            /**
             * TODO To be implemented
             * @param prefix
             * @returns {CP.Analytics}
             */
            trackName: function (prefix) { return this; },
        };

        return CP;
    }
);

/* COMMON IMPLEMENTATIONS */
var requirejs = require || requirejs;
requirejs(['cp'], function (CP) {
    CP.toInit(function ($) {
        // Navigating from inside an iframe
        CP.connect('navigate', function(url) {
            document.location.href = url;
        });

        // display flag messages
        CP.connect('message', function (data, source) {
            CP.showFlag(_(data).extend({ source: source }));
        });

        // Updating current URL without triggering a window reload
        CP.connect('history', function(options) {
            var path = typeof options === 'string' ? options : options.path;
            var title = options.title || path;
            var data = options.data || title;

            if (options.type === 'replace') {
                window.history.replaceState(data, title, path);
            } else {
                window.history.pushState(data, title, path);
            }
        });
    });
});
