/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 */

var requirejs = requirejs || require;
var checklists = {};

define(
    'checklist',
    ['chk', 'jquery', 'underscore'],
    function(CHK, $, _) {

        function setSortValue(cellId, value) {
            tdElement = document.getElementById('sort_' + cellId);
            tdElement.abbr = value;
        };

        var Checklist = function (name, entityId) {
            this.name = name;
            this.entityId = entityId;

            function makeUri(pageId, columnKey, cellId, action) {
                var uri = CHK.contextPath + "/plugins/checklists/" + action + ".action" +
                    "?entityId=" + entityId +
                    "&columnKey=" + columnKey +
                    "&cellId=" + cellId +
                    "&pageId=" + pageId;
                return uri;
            };

            function updateSortValue(pageId, columnKey, cellId, columnType, optionIds, storeInRows) {
                var uri = makeUri(pageId, columnKey, cellId, "getSortValue") +
                    "&columnType=" + columnType + "&storeInRows=" + storeInRows + "&optionIds=" +
                    encodeURIComponent(optionIds);
                $.get(uri, function (data) {
                    setSortValue(cellId, data.responseText);
                });
            };

            function changeLabel(pageId, labelName, cellId, value, checkValue) {
                var uri = makeUri(pageId, labelName, cellId, "changeLabel") + "&value=" + value;
                $.get(uri, function () {
                    var newValue = !value;
                    var callback = "checklists." + name + ".changeLabel('"
                        + pageId + "','"
                        + labelName + "','"
                        + cellId + "',"
                        + newValue + ",'')";
                    var $checkbox = $('input[name = ' + cellId + ']');
                    $checkbox.attr("onchange", callback);

                    // Switch icons
                    if (newValue) {
                        setSortValue(cellId, "1");
                        $checkbox.next()
                            .removeClass('aui-iconfont-remove chk-icon-unchecked')
                            .addClass('aui-iconfont-approve chk-icon-checked')
                            .text('✔');
                    } else {
                        setSortValue(cellId, "0");
                        $checkbox.next()
                            .removeClass('aui-iconfont-approve chk-icon-checked')
                            .addClass('aui-iconfont-remove chk-icon-unchecked')
                            .text('✖');
                    }

                    triggerChangeEvent('label', cellId, newValue);
                });
                return false;
            }

            function changeCheckbox(pageId, columnKey, cellId, value, checkValue) {
                var uri = makeUri(pageId, columnKey, cellId, "changeCheckbox") + "&storeInRows=true";
                if (!value) {
                    uri += "&value=" + checkValue;
                }
                $.get(uri, function (data) {
                    var newValue = !value;
                    var callback = 'checklists.' + name + ".changeCheckbox('"
                        + pageId + "','"
                        + columnKey + "','"
                        + cellId + "',"
                        + newValue + ",'"
                        + checkValue + "')";
                    var $checkbox = $('#' + cellId);
                    $checkbox.attr("onchange", callback);

                    // Switch icons
                    if (newValue) {
                        setSortValue(cellId, "1");
                        $checkbox.next()
                            .removeClass('aui-iconfont-remove chk-icon-unchecked')
                            .addClass('aui-iconfont-approve chk-icon-checked')
                            .text('✔');
                    } else {
                        setSortValue(cellId, "0");
                        $checkbox.next()
                            .removeClass('aui-iconfont-approve chk-icon-checked')
                            .addClass('aui-iconfont-remove chk-icon-unchecked')
                            .text('✖');
                    }

                    triggerChangeEvent('checkbox', cellId, value);
                });
                return false;
            }

            function changeSelect(pageId, columnKey, cellId, optionIds, storeInRows) {
                var selectElem = document.getElementById(cellId);

                // sanitize old '[~user]' values, convert them to the new format: '👤user'
                var search = /\[~(\w+)\]/g;
                var replace = '👤$1';

                var value = selectElem.options[selectElem.selectedIndex].value.replace(search, replace);
                optionIds = optionIds.replace(search, replace);

                var uri = makeUri(pageId, columnKey, cellId, "changeSelect")
                    + "&storeInRows=" + storeInRows + "&value=" + encodeURIComponent(value)
                    + "&optionIds=" + encodeURIComponent(optionIds);

                $.get(uri, function (data) {
                    updateSortValue(pageId, columnKey, cellId, 'select', optionIds, storeInRows);
                    $(selectElem).blur();
                    triggerChangeEvent('select', cellId, value);
                });
                return false;
            }

            function changeInput(pageId, columnKey, cellId, storeInRows) {
                var value = document.getElementById(cellId).value;
                var label = document.getElementById("label_" + cellId).firstChild;
                var uri = makeUri(pageId, columnKey, cellId, "changeInput") +
                    "&value=" + value + "&storeInRows=" + storeInRows;
                $.get(uri, function () {
                    updateSortValue(pageId, columnKey, cellId, 'select', "", storeInRows);
                    triggerChangeEvent('input', cellId, value);
                    label.nodeValue = value;
                });
            };

            function editWiki(pageId, columnKey, cellId, rows, cols, storeInRows) {
                var uri = makeUri(pageId, columnKey, cellId, "editWiki") +
                    "&checklistId=" + name + "&rows=" + rows + "&cols=" + cols +
                    "&storeInRows=" + storeInRows;
                $.get(uri, function (data) {
                    var originalContent = encodeURIComponent(document.getElementById("wiki_" + cellId).innerHTML);
                    document.getElementById("wiki_" + cellId).innerHTML = "";
                    document.getElementById("form_" + cellId).innerHTML = data;
                    var $textarea = $('#' + cellId);
                    $textarea.data('originalContent', originalContent);
                    $textarea.focus();
                });

                return false;
            };

            function setWikiInput(pageId, columnKey, cellId, rows, cols, storeInRows) {
                var value = document.getElementById(cellId).value;
                var uri = makeUri(pageId, columnKey, cellId, "setWikiInput") +
                    "&checklistId=" + name + "&rows=" + rows + "&cols=" + cols +
                    "&storeInRows=" + storeInRows + "&value=" + encodeURIComponent(value);
                $.get(uri, function (data) {
                    document.getElementById("form_" + cellId).innerHTML = "";
                    document.getElementById("wiki_" + cellId).innerHTML = data;
                    updateSortValue(pageId, columnKey, cellId, 'wikiinput', "", storeInRows);
                    triggerChangeEvent('wikiinput', "wiki_" + cellId, value);
                });
                return false;
            };

            function triggerChangeEvent(type, cellId, value) {
                $('td:has(#' + cellId + ')').trigger('changed.chk', value, {
                    cellType: type,
                    cellId: cellId,
                    checklist: name
                });
            };
            
            this.changeLabel = changeLabel;
            this.setWikiInput = setWikiInput;
            this.editWiki = editWiki;
            this.changeSelect = changeSelect;
            this.changeInput = changeInput;
            this.changeCheckbox = changeCheckbox;
        };

        return Checklist;
    }
);

requirejs(
    ['chk', 'checklist', 'jquery'],
    function(CHK, Checklist, $) {

        CHK.toInit(function () {

            //Initialize checklists
            $('.chk-wrap').each(function(i, item) {
                var $checklist = $(item);
                var checklist = $checklist.data('checklist');
                var entity = $checklist.data('entity');
                var context = $checklist.data('context');

                // get column types
                var $table = $checklist.find('.chk-table');
                var $columns = $table.find('th');
                var columnTypes = [$checklist.data('sortType')];
                $columns.each(function(i, item) {
                    var sortType = $(item).data('sortType')
                    if (columnTypes.indexOf(sortType) === -1) {
                        columnTypes.push(sortType);
                    }
                });

                checklists[checklist] = new Checklist(checklist, entity, context);
                // customizeMacroTable($table.prop('id'), columnTypes, 1, '', true, 'Click to sort', '', false, false, false, false, 0, false, '');
            });

            // EVENT LISTENERS
            // This ensures the textarea disappears also when no content has changed when textarea looses focus
            $(document).on('focus', 'table.chk-table textarea', function () {
                // keep the original value
                $(this).data('originalValue', this.value);
            });

            $(document).on('blur', 'table.chk-table textarea', function () {
                // retrieve the original value
                var originalValue = $(this).data('originalValue');
                // and compare to the current one
                if (originalValue === this.value) {
                    // If they are the same we just replace the textarea with the original HTML content
                    var cellId = this.id;
                    document.getElementById("form_" + cellId).innerHTML = "";
                    document.getElementById("wiki_" + cellId).innerHTML = decodeURIComponent($(this).data('originalContent'));
                }
            });
        });
    }
);
