/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

define(
    'cb',
    ['ajs', 'jquery', 'underscore', 'backbone'],
    function (AJS, $, _, Backbone) {
        var DOCUMENT_ORIGIN =
            document.location.origin ? document.location.origin : document.location.protocol +
                '//' + document.location.hostname +
                (document.location.port ? ':' + document.location.port : '');
        const CB = {};

        /* WRAPPERS 4 AJS & CONFLUENCE */
        CB.getParam = function (key) {
            return AJS.params[key];
        };
        CB.getMeta = function (key) {
            return AJS.Meta.get(key);
        };
        CB.getDarkFeature = function (key) {
            return AJS.DarkFeatures.isEnabled(key);
        };
        CB.message = function () {
            var type = arguments[0];
            if (['error', 'warning', 'success', 'info'].indexOf(type) === -1) {
                // prevent errors with obsolete message types
                type = 'info';
            }
            if (arguments[2]) {
                return AJS.messages[type](arguments[1], arguments[2]);
            } else {
                return AJS.messages[type](arguments[1]);
            }
        };
        CB.getVersion = function() {
            return parseInt(AJS.version.substr(0, AJS.version.indexOf('.')), 10);
        };
        CB.setupTabs = function() { return AJS.tabs.setup() };
        CB.changeTab = function(tab) { return AJS.tabs.change(tab); };
        CB.loadTemplate = function (template) { return AJS.template.load(template) };

        CB.I18n = AJS.I18n;
        CB.Labels = AJS.Labels;
        CB.format = AJS.format;
        CB.contextPath = AJS.contextPath();
        CB.toInit = AJS.toInit;
        CB.log = AJS.log;
        CB.tabs = AJS.tabs;
        CB.bind = AJS.bind;

        CB.request = $.ajax;

        if (AJS.Confluence) {
            CB.Confluence = AJS.Confluence;
        } else {
            requirejs(['confluence'], function (Confluence) {
                CB.Confluence = Confluence;
            });
        }

        /* UTILS */
        CB.getAUIVersion = function () {
            return +$('body').data('aui-version').charAt(0);
        };

        CB.send = function (target, event, data) {
            var origin = document.location.origin;
            if (!origin) {
                var protocol = document.location.protocol;
                var hostname = document.location.hostname;
                var port = (document.location.port ? ':' + document.location.port : '');
                origin = protocol + '//' + hostname + port;
            }
            target.postMessage(JSON.stringify({
                protocol: 'comala',
                event: event,
                data: data
            }), origin);
        };

        /**
         * Parses the query of url and delivers a map of param-value
         * i.e. /confluence/pages/viewpage.action?pageId=983079&task=2
         *      You want the value of pageId, just do:
         *      queryString().pageId // returns 983079
         */
        CB.queryString = function (qs) {
            qs = qs || document.location.search;
            qs = qs.split('+').join(' ');

            var params = {};
            var tokens;
            var re = /[?&]?([^=]+)=([^&]*)/g;

            while (tokens = re.exec(qs)) {
                params[decodeURIComponent(tokens[1])] = decodeURIComponent(tokens[2]);
            }
            return params;
        };

        CB.isSpaceAdmin = function () {
            return CB.getParam('isSpaceAdmin') || CB.getParam('isConfluenceAdmin') || (
                CB.getParam('spaceKey') && CB.getParam('remoteUser') &&
                CB.getParam('spaceKey') === '~' + CB.getParam('remoteUser')
            );
        };

        // To handle PostMessage iframe <> parent using our own protocol
        CB.messageHandlers = [];
        CB.connect = function (action, callback, context) {
            if (!CB.messageHandlers[action]) {
                CB.messageHandlers[action] = [];
            }
            // note that you can register multiple callbacks for the same event,
            // but keep in mind that triggering this event will call all callbacks
            CB.messageHandlers[action].push({
                callback: callback,
                context: context
            });
        };

        window.addEventListener('message', function (event) {
            // For Chrome, the origin property is in the event.originalEvent object.
            var origin = event.origin || event.originalEvent;
            var handlers = CB.messageHandlers[event.data.event] || [];
            var protocol = event.data.protocol;
            var data = event.data.data;
            var source = event.source;
            if (origin === DOCUMENT_ORIGIN && protocol === 'cb' && handlers.length > 0) {
                handlers.forEach(function (handler) {
                    handler.callback.call(handler.context, data, source);
                });
            }
        });

        /* VIEWS */
        CB.StatefulView = Backbone.View.extend({
            defaults: {},
            options: {},
            initialize: function (options) {
                this.options = _.defaults(options || {}, this.defaults);
                _.bindAll(this, 'setState', 'getState', 'resetState', 'stateChanged', 'afterInit');
                this.state = this.getStateModel();
                this.listenTo(this.state, 'change', function (state, options) {
                    var changed = Object.keys(state.changed);
                    if (changed.length > 0) {
                        _.defer(this.stateChanged, changed, options);
                    }
                });
                this.afterInit();
            },
            getStateModel: function () {
                return new Backbone.Model(this.options);
            },
            setState: function (attr, val) {
                return typeof val !== 'undefined' ? this.state.set(attr, val) : this.state.set(attr);
            },
            unsetState: function (attr) { return this.state.unset(attr); },
            getState: function (attr) { return this.state.get(attr); },
            resetState: function () { return this.state.clear(); },
            afterInit: function () {
                // to be overriden
            },
            stateChanged: function () {
                // to be overriden
            }
        });

        CB.iframe = CB.StatefulView.extend({
            className: 'cml-iframe-wrapper',
            defaults: {
                id: null,
                url: '',
                query: {},
                absolute: false,
                customData: null,
                reloadEvent: '',
                width: '100%',
                height: '100%',
                css: '',
                scrolling: 'no',
                onLoad: function () { },
                onResize: function () { },
            },
            getStateModel: function () {
                return new Backbone.Model({ loading: true });
            },
            afterInit: function () {
                _.bindAll(this, 'onLoad', 'onUnload', 'send', 'listen', 'listener', 'resize');
                this.params = _.extend({}, this.options.query, {
                    cp: CB.contextPath,
                    xdm_e: CB.getParam('baseUrl'),
                    loc: CB.getParam('userLocale') || CB.getParam('actionLocale'),
                    user: CB.getParam('remoteUser'),
                    rom: CB.getMeta('access-mode') === 'READ_ONLY' ? true : false,
                    atlToken: CB.getParam('atlToken'),
                });
                if (this.options.reloadEvent) {
                    this.listen(this.options.reloadEvent, this.onLoad);
                }
                this.listen('resize', this.resize);
            },
            stateChanged: function (changed) {
                var _self = this;
                var $iframe = _self.$iframe;
                if ($iframe && changed.indexOf('loading') !== -1) {
                    _.defer(function () {
                        $iframe.toggleClass('cml-loading', _self.getState('loading'));
                    });
                }
            },
            getUrl: function (query) {
                var params = _.extend({}, this.params, query || {});
                var queryString = _(Object.keys(params)).map(function (paramKey) {
                    return paramKey + '=' + encodeURIComponent(this.params[paramKey]);
                }, this).join('&');
                var baseUrl = CB.contextPath + '/plugins/servlet';
                return (this.options.absolute ? '' : baseUrl) + this.options.url + '?' + queryString;
            },
            _renderIframe: function (query) {
                this.$el.html('<iframe frameborder="0" ALLOWTRANSPARENCY="true"'
                    + ' src="' + this.getUrl(query) + '"'
                    + ' id="' + this.options.id + '_iframe"'
                    + ' width="' + (_.isUndefined(this.options.width) ? '100%' : this.options.width) + '"'
                    + ' height="' + (_.isUndefined(this.options.height) ? '100%' : this.options.height) + '"'
                    + ' scrolling="' + this.options.scrolling + '"'
                    + ' class="cml-iframe' + (this.options.css ? ' ' + this.options.css : '') + ' cml-loading"'
                    + ' />'
                );
                this.$iframe = this.$el.find('iframe');
                this.$iframe.on('load', this.onLoad);
            },
            render: function(query) {
                $hiddenParent = this.$el.parents('*:hidden').last();
                if ($hiddenParent.length > 0) {
                    // if the board is within a hidden div (i.e. the expand macro)
                    var classNameList = $hiddenParent.attr('class');
                    var _self = this;
                    var observer = new MutationObserver(function (mutations) {
                        if (mutations[0].oldValue === classNameList) {
                            // when the parent is shown
                            if (!_self.$iframe) {
                                // on the first expand:
                                // delay initial iframe render
                                _self._renderIframe(query);
                            }
                        }
                    });
                    // observe when the class name list of the parent changes
                    observer.observe($hiddenParent.get(0), { attributeFilter: ['class'], attributeOldValue: true });
                } else {
                    this._renderIframe(query);
                }
                return this.el;
            },
            onLoad: function (e) {
                this.setState('loading', false);
                $(this.$iframe.get(0).contentWindow).one('unload', this.onUnload);
                var onLoad = this.options.onLoad;
                var customData = this.options.customData;
                if (customData) {
                    if (_.isFunction(customData)) {
                        customData = customData();
                    }
                    this.send('customData', customData);
                }
                if (onLoad) {
                    _.bind(onLoad, this)();
                }
            },
            onUnload: function (e) {
                this.setState('loading', true);
            },
            reload: function (query) {
                if (this.$iframe) {
                    this.$iframe.attr('src', this.getUrl(query));
                }
            },
            send: function (event, data) {
                if (this.$iframe) {
                    var iframe = this.$iframe.get(0);
                    if (iframe && iframe.contentWindow) {
                        CB.send(iframe.contentWindow, event, data);
                    }
                }
            },
            listen: function (event, callback) {
                CB.connect(event, this.listener(callback), this);
            },
            listener: function (callback) {
                return function (data, source) {
                    if (
                        this.$iframe && source &&
                        this.$iframe.get(0) === source.frameElement
                    ) {
                        callback(data);
                    }
                };
            },
            resize: function (height) {
                this.$iframe.attr('height', height);
                _.delay(function(_self) {
                    _.bind(_self.options.onResize, _self)();
                }, 66, this);
            }
        });

        CB.InlineDialog = CB.StatefulView.extend({
            defaults: {
                url: '',
                query: {},
                content: '',
                trigger: null,
                id: '',
                width: 450,
                height: 240,
                alignment: 'right middle',
                open: false,
                persistent: null,
                'responds-to': 'toggle',
                hideOthers: true,
                onShow: function() {},
                onHide: function() {}
            },
            tagName: 'aui-inline-dialog',
            className: 'cb-inline-dialog',
            events: {
                'aui-show': 'show',
                'aui-hide': 'hide'
            },
            trigger: null,
            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['alignment', 'open']))
            },
            afterInit: function () {
                _.bindAll(this, 'show', 'hide', 'render', 'setPosition', '_updateTrigger');
                this.attributes = _.pick(
                    this.options,
                    'alignment', 'persistent', 'responds-to'
                );
                this.$el.attr(this.attributes);
                this.$el.addClass('cb-inline-dialog');
                if (this.options.url) {
                    this.iframe = new CB.iframe(_.extend(
                        {
                            id: this.options.id + '_iframe',
                        },
                        _.pick(this.options, 'url', 'query', 'width', 'height')
                    ));
                }
                this.parseTriggers();
                // update the inline dialog position when several triggers open the same
                var eventName = this.options['responds-to'] === 'toggle' ? 'mousedown' : 'mouseenter';
                $(document).on(eventName, this.options.trigger, this._updateTrigger);
                if (eventName === 'mouseenter') {
                    $(document).on('mouseout', this.options.trigger, this.hide);
                }
            },
            stateChanged: function (changed) {
                var $el = this.$el;
                var state = this.state;
                _.each(changed, function (attr) {
                    var curAttr = $el.attr(attr);
                    var newAttr = state.get(attr);
                    if (curAttr !== newAttr) {
                        $el.attr(attr, newAttr);
                    }
                });
                if (changed.indexOf('open') !== -1) {
                    // whenever the state dialog is opened, the iframe should refresh
                    if (this.getState('open')) {
                        this.onShow();
                    } else {
                        this.onHide();
                    }
                }
            },
            parseTriggers: function() {
                var $triggers = $(this.options.trigger);
                var id = this.id;
                $triggers.each(function () {
                    $trigger = $(this);
                    if (!$trigger.attr('resolved')) {
                        var $newTrigger = $trigger.clone();
                        $newTrigger
                            .attr('data-aui-trigger', true)
                            .attr('aria-controls', id);
                        $trigger.replaceWith($newTrigger);
                    }
                });
            },
            show: function () {
                if (!this.getState('open')) {
                    this.setState('open', true);
                }
            },
            onShow: function () {
                var $container = this.getContainer();
                if (this.options.hideOthers) {
                    $('aui-inline-dialog[open]').not(this.$el).removeAttr('open');
                }
                if (this._isReusable()) {
                    _.delay(this.setPosition, 30);
                }
                if (this._isReusable() || $container.children().length === 0) {
                    this.render();
                }
                this.options.onShow(this);
            },
            hide: function () {
                if (this.getState('open')) {
                    this.setState('open', false);
                }
            },
            onHide: function() {
                this.options.onHide(this);
                if (this.options.removeOnHide) {
                    this.remove();
                }
            },
            toggle: function () {
                this.setState('open', !this.getState('open'));
            },
            getContainer: function() {
                var $container = this.$el.find('.aui-inline-dialog-contents')
                if (!$container.length) {
                    $container = this.$el;
                }
                return $container;
            },
            _isReusable: function() {
                return $(this.options.trigger).length > 1;
            },
            _updateTrigger: function(e) {
                if (!e.target.isEqualNode(this.trigger)) {
                    this.trigger = e.target;

                    if (this._isReusable() && this.getState('open')) {
                        // if the element clicked was another trigger
                        // re-open the inline dialog after the position has been updated
                        var _self = this;
                        this.$el.one('aui-hide', function () {
                            _.delay(_self.show, 300);
                        });
                    }
                }
            },
            _renderIframe: function (query) {
                var $container = this.getContainer();
                $container.html(this.iframe.render(query));
            },
            _isIframeAttached: function () {
                return this.$el.find('iframe').length !== 0;
            },
            send: function (event, data) {
                if (this.iframe && this.getState('open')) {
                    this.iframe.send(event, data);
                }
            },
            setPosition: function(trigger) {
                var $trigger = $(trigger || this.trigger);
                var tOffset = $trigger.offset();
                var tHeight = $trigger.outerHeight();
                var tWidth = $trigger.outerWidth();
                var eHeight = this.$el.outerHeight();
                var eWidth = this.$el.outerWidth();
                var alignment = this.getState('alignment').split(' ');
                var edge = alignment[0];
                var edgePos = alignment[1];
                var top = tOffset.top;
                var left = tOffset.left;
                switch (edge) {
                    case 'top':
                        top -= eHeight;
                        break;
                    case 'left':
                        left -= eWidth;
                        break;
                    case 'right':
                        left += tWidth;
                        break;
                    case 'bottom':
                        top += tHeight;
                        break;

                }
                switch (edgePos) {
                    case 'right':
                        left -= (eWidth - tWidth);
                        break;
                    case 'center':
                        left -= (eWidth - tWidth) / 2;
                        break;
                    case 'bottom':
                        top += (eHeight - tHeight);
                        break;
                    case 'middle':
                        top += (eHeight - tHeight) / 2;
                        break;
                }
                this.$el.css({
                    top: top,
                    left: left,
                    transform: 'translateZ(0px)'
                });
            },
            render: function () {
                if (this.getState('open')) {
                    var $container = this.getContainer();
                    if (this.iframe) {
                        if (!this._isIframeAttached()) {
                            this._renderIframe(query);
                        } else {
                            this.iframe.reload(query);
                        }
                    } else if (_.isFunction(this.options.content)) {
                        this.options.content(this);
                    } else if (_.isElement(this.options.content)) {
                        $container.html(this.options.content);
                    }
                }
                return this.el;
            }
        });


        CB.Dialog = CB.StatefulView.extend({
            tagName: 'section',
            className: 'aui-layer aui-dialog2 cml-modal-dialog',
            defaults: {
                id: '',
                type: '',
                title: '',
                content: '',
                hint: '',
                actions: [],
                size: '',
                width: 600,
                height: 225,
                customData: null,
                closeable: true,
                absolute: false,
                path: '',
                reloadEvent: '',
                query: {},
                open: false,
                remove: false,
                onLoad: function () { },
                onClose: function () { },
                onOpen: function () { }
            },
            events: {
                'click .aui-dialog2-footer-actions .aui-button': 'clickAction'
            },

            getStateModel: function () {
                return new Backbone.Model(_.pick(this.options, ['open']));
            },

            afterInit: function () {
                _.bindAll(this, 'onShow', 'onHide', 'initializeDialog');

                requirejs(['aui/dialog2'], this.initializeDialog);

                CB.connect('dialogClose', this.hide, this);
                CB.connect('dialogResize', this.resize, this);
            },

            initializeDialog: function (dialog) {
                this.dialog = dialog(this.$el);

                this.dialog.on('show', this.onShow);
                this.dialog.on('hide', this.onHide);

                if (this.options.open) {
                    this.show();
                }
            },

            show: function (query) {
                if (query) {
                    this.options.query = _.extend({}, this.options.query, query);
                }
                if (this.dialog) {
                    this.render();
                    this.dialog.show();
                    this.$el.addClass('cml-modal-dialog-opening');
                    this.setState('open', true);
                } else {
                    // aui resource has not load yet
                    // wait until dialog is initialized to open
                    this.options.open = true;
                }
                return this;
            },

            onShow: function () {
                var _self = this;
                _.defer(function () {
                    _self.$el.removeClass('cml-modal-dialog-opening');
                    _self.$el.addClass('cml-modal-dialog-open');
                    _self.center();
                    _self.options.onOpen();
                });
            },

            hide: function () {
                if (this.getState('open')) {
                    this.$el.addClass('cml-modal-dialog-closing');
                    _.delay(function (dialog) {
                        dialog.hide();
                    }, 300, this.dialog);
                }
                return this;
            },

            onHide: function () {
                this.send('dialogClosed');
                this.setState('open', false);
                this.options.onClose();
                if (this.options.remove) {
                    this.remove();
                } else {
                    this.$el.removeClass('cml-modal-dialog-closing');
                    this.$el.removeClass('cml-modal-dialog-open');
                }
            },

            toggle: function () {
                if (this.getState('open')) {
                    this.hide();
                } else {
                    this.show();
                }
            },

            resize: function (options) {
                var height = typeof options === 'object' ? options.height : options;
                var width = typeof options === 'object' ? options.width : this.$el.width();
                var windowHeight = $(window).height() - 50;
                var windowWidth = $(window).width() - 50;
                var h = Math.min(height, windowHeight);
                var w = Math.min(width, windowWidth);
                if (this.getState('open') && h) {
                    this.$el.css('height', h + 'px');
                    this.$el.css('width', w + 'px');
                    this.center();
                }
            },

            center: function () {
                // vertically center the dialog
                var dialogHeight = this.$el.height();
                var windowHeight = $(window).height();
                var top = (windowHeight - dialogHeight) / 2;
                this.$el.css('top', top + 'px');
            },

            send: function (event, data) {
                if (this.iframe) {
                    this.iframe.send(event, data);
                }
            },

            getActions: function () {
                return this.options.actions;
            },

            clickAction: function (event) {
                var $target = $(event.target);
                $.each(this.getActions(), function (i, action) {
                    if (action.id === $target.attr('id') && action.onClick) {
                        action.onClick(event);
                        return false;
                    }
                });
            },

            _renderHeader: function () {
                if (this.options.title) {
                    this.$el.append('<header class="aui-dialog2-header">' +
                        (_.isFunction(this.options.title) === true ? '<h2 class="aui-dialog2-header-main">' + this.options.title() + '</h2>' :
                            '<h2 class="aui-dialog2-header-main">' + this.options.title + '</h2>') +
                        (this.options.closeable ? '<a class="aui-dialog2-header-close">' +
                            '<span class="aui-icon aui-icon-small aui-iconfont-close-dialog" />' +
                            '</a>' : '') +
                        '</header>');
                }
            },

            renderContent: function () {
                if (this.options.content) {
                    var $content = $('<div class="aui-dialog2-content" />');
                    $content.append(this.options.content);
                    this.$el.append($content);
                } else if (this.options.path) {
                    var iframe = new CB.iframe({
                        id: this.options.id + '_content',
                        url: this.options.path,
                        absolute: this.options.absolute,
                        customData: this.options.customData,
                        reloadEvent: this.options.reloadEvent,
                        query: this.options.query,
                        width: '100%',
                        height: '100%',
                        onLoad: this.options.onLoad
                    });
                    this.iframe = iframe;
                    this.$el.append(iframe.render());
                }
            },

            _renderFooter: function () {
                // parse dialog buttons
                var actions = '';
                $.each(this.getActions(), function (i, action) {
                    actions += '<button id="' + action.id + '" class="aui-button aui-button-' + action.style + '">' +
                        action.label + '</button>';
                });
                if (this.options.hint || actions) {
                    this.$el.append('<footer class="aui-dialog2-footer">' +
                        '<div class="aui-dialog2-footer-actions"> ' + actions + '</div>' +
                        '<div class="aui-dialog2-footer-hint">' + this.options.hint + '</div>' +
                        '</footer>');
                }
            },

            render: function () {
                var width = typeof this.options.width === 'string' ? this.options.width : this.options.width + 'px';
                var height = typeof this.options.height === 'string' ? this.options.height : this.options.height + 'px';
                this.$el.attr({
                    'role': 'dialog',
                    'aria-hidden': 'true',
                    'data-aui-remove-on-hide': this.options.remove,
                    'data-aui-modal': !this.options.closeable,
                    'style': 'height: ' + height + ';',
                });

                // add specific class for warning dialogs
                if (this.options.type) {
                    this.$el.addClass('aui-dialog2-' + this.options.type);
                }
                if (this.options.size) {
                    this.$el.addClass('aui-dialog2-' + this.options.size);
                } else {
                    this.$el.css('width', width);
                }

                // reset dialog's contents
                this.$el.html('');

                // append dialog header
                this._renderHeader();
                // append contents or render iframe
                this.renderContent();
                //append dialog footer
                this._renderFooter();

                return this.el;
            }
        });

        CB.Confirm = CB.Dialog.extend({
            defaults: {
                id: 'cb-confirm',
                open: true,
                type: 'warning',
                width: 400,
                height: 'auto',
                title: CB.I18n.getText('Confirm.title'),
                remove: true,
                closeable: false,
                content: '',
                checks: [],
                hint: '',
                accept: CB.I18n.getText('Confirm.accept'),
                cancel: CB.I18n.getText('Confirm.cancel'),
                onAccept: function () { },
                onCancel: function () { },
                onOpen: function () { },
                onClose: function () { }
            },
            _getChecks: function() {
                var checks = [];
                this.$el.find('input').each(function(i, field) {
                    checks[i] = field.checked;
                });
                return checks;
            },
            onAccept: function () {
                this.options.onAccept(this._getChecks());
                this.hide();
            },
            onCancel: function () {
                this.options.onCancel(this._getChecks());
                this.hide();
            },
            getActions: function () {
                return [{
                    id: 'cb-confirm-accept',
                    style: 'primary',
                    label: this.options.accept,
                    onClick: _.bind(this.onAccept, this)
                }, {
                    id: 'cb-confirm-cancel',
                    style: 'subtle',
                    label: this.options.cancel,
                    onClick: _.bind(this.onCancel, this)
                }];
            },
            renderContent: function () {
                var $content = $('<div class="aui-dialog2-content" />');
                $content.append('<p>' + this.options.content + '</p>');
                this.$el.append($content);
                var id = this.options.id;
                if (this.options.checks.length > 0) {
                    var $form = $('<form class="aui"></form>');
                    $content.append($form);
                    $(this.options.checks).each(function(i, check) {
                        var checkId = id + '-check-' + i;
                        $form.append(
                            '<div class="checkbox">' +
                                '<input type="checkbox" class="checkbox" name="' + checkId + '" id="' + checkId + '"/> ' + 
                                '<label for="' + checkId + '">' + check + '</label>' +
                            '</div>'
                        );
                    });
                }
            }
        });

        CB.Flag = CB.StatefulView.extend({
            defaults: {
                type: 'info',
                close: 'auto',
                title: '',
                body: '',
                source: null,
                removeAfterClose: false,
                events: {},
            },
            events: {
                'aui-flag-close': 'closeCallback'
            },
            afterInit: function () {
                _.bindAll(this, 'closeCallback');
                this.delegateEvents(_.extend({}, this.events, this.options.events));
                this.render();
            },
            close: function () {
                this.el.close();
            },
            closeCallback: function (e) {
                if (e.target === this.el) {
                    var target = this.getState('source');
                    if (target) {
                        CB.send(target, 'messageClosed');
                    }
                    if (this.options.removeAfterClose) {
                        this.remove();
                    } else {
                        this.resetState();
                    }
                }
            },
            stateChanged: function (changed, options) {
                // display a new message only if we're not closing the one being shown AND
                // the contents of the new message are different from the previous one being shown (if any)
                if (!options.unset && (
                    changed.indexOf('title') !== -1 ||
                    changed.indexOf('body') !== -1)
                ) {
                    this.render(changed, options);
                }
            },
            render: function (changed, options) {
                var message = _.pick(this.state.attributes, ['type', 'close', 'title', 'body']);
                // display a new message only if the message contains at least a title OR a body
                if (message.title || message.body) {
                    var _self = this;
                    requirejs(['aui/flag'], function (flag) {
                        var element = flag(message);
                        _self.setElement(element);
                        $(element.parentNode).one('aui-flag-close', _self.closeCallback);
                    });
                }
            }
        });

        // provide easy access to the flagView model to render new flags
        var flagView = new CB.Flag();
        CB.showFlag = function (options) { return flagView.setState(options); }

        return CB;
    }
);

var requirejs = require || requirejs;
requirejs(['cb', 'jquery'], function (CB, $) {
    /* AJS toInit Wrapper with common implementations */
    CB.toInit(function () {
        $('body')
            // Add aui version class to body for version dependant style fixes
            .addClass('aui' + CB.getAUIVersion());

        // Navigating from inside an iframe
        CB.connect('navigate', function (url) {
            document.location.href = url;
        });

        // Generic Dialogs
        CB.connect('dialog', function (data) {
            new CB.Dialog(data);
        });

        // Confirmation Dialogs
        CB.connect('confirm', function (data, source) {
            new CB.Confirm(_.extend({}, data, {
                onAccept: function (args) { CB.send(source, 'confirmAccept', args); },
                onCancel: function (args) { CB.send(source, 'confirmReject', args); }
            }));
        });

        // Updating current URL without triggering a window reload
        CB.connect('history', function (options) {
            var path = typeof options === 'string' ? options : options.path;
            var title = options.title || path;
            var data = options.data || title;

            if (options.type === 'replace') {
                window.history.replaceState(data, title, path);
            } else {
                window.history.pushState(data, title, path);
            }
        });

        // display flag messages
        CB.connect('message', function (data, source) {
            CB.showFlag(_(data).extend({ source: source }));
        });

        CB.createNewEvent = function(eventName) {
            var event;
            if (typeof(Event) === 'function') {
                event = new Event(eventName);
            } else {
                // For IE
                event = document.createEvent('Event');
                event.initEvent(eventName, true, true);
            }
            return event;
        }

        /* WRAPPERS 4 AJS & CONFLUENCE */
        CB.getParam = function (key) {
            return AJS.params[key];
        };

        return CB;
        }
    );
});

var requirejs = require || requirejs;
requirejs(['cb', 'jquery'], function (CB) {

    CB.toInit(function () {
    });
});