/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var Canvas = Canvas || {};
Canvas.Wizard = Canvas.Wizard || {};

Canvas.Wizard.TasksBoard = Backbone.View.extend({
    className: 'cnv-wizard-tasks',

    users: [],
    j: 1,
    boardTitle: AJS.I18n.getText('canvas.blueprint.tasksboard.description'),
    template: Canvas.WizardTemplates.tasksBoard,
    templateUser: Canvas.WizardTemplates.tasksUser,
    templateUserInfo: Canvas.WizardTemplates.tasksUserInfo,

    board: {"leftJustified": true, "type":"LABEL","row":[{"column":[
        {"header":"Inbox","container":[{"key":"inbox","caption":" ","style":"s029"}]},
        {"header":"To do","container":[{"key":"todo","caption":" ","style":"s025"}]},
        {"header":"In progress","container":[{"key":"inprogress","caption":" ","style":"s027"}]},
        {"header":"Complete","container":[{"key":"complete","caption":" ","style":"s026"}]}]}],
        "name": AJS.I18n.getText('canvas.blueprint.tasksboard.title'),
        "matchRowHeight":true,"cardsOnlyBoard":true},

    events: {
        "selected.autocomplete-user" : "selected",
        "click .cnv-icon-remove" : "removeUserField",
        "click .cnv-icon-cancel" : "cancelEdit",
        "click .cnv-taskuser-info" : "edit",
        "click .cnv-wizard-header" : "editTitle",
        "blur .cnv-wizard-header-edit .cnv-text" : "saveTitle",
        "submit .cnv-wizard-header-edit" : "saveTitle",
        "focus .autocomplete-user" : "autocomplete"
    },

    initialize: function() {
        // init to ensure if user goes back and forth through wizard panels, always with start fresh.
        this.users = [];
        this.j = 1;
        this.render();
        this.$error = this.$(".error");
        this.$preview = this.$(".boardPreview");
        this.$uiStep1 = this.$('.uiguide-step1');
        this.$uiStep1.fadeIn();

        this.renderPreview(this.board);

        // by default, first user is current user
        this.firstUser();
    },

    autocomplete: function (evt) {
        var $field = this.$(evt.target);

        if(!$field.hasClass('cnv-autocompletable')) {
            this._autocomplete();
            $field.addClass('cnv-autocompletable');
        }
    },

    edit: function (evt) {
        var $fieldgroup = AJS.$(evt.target).parents('.field-group');
        
        $fieldgroup
            .addClass('editing')
            .find('.text')
                .focus()
                .select();
    },

    editTitle: function (evt) {
        var $header = AJS.$(evt.target).parents('.cnv-wizard-header');

        $header
            .addClass('editing')
            .find('.cnv-text')
                .focus()
                .select();
    },

    saveTitle: function (evt) {
        evt.preventDefault();

        var title = this.$('.cnv-wizard-header-edit .cnv-text').val();
        this.boardTitle = title;
        this.board.name = title;
        this.$('.cnv-wizard-header .cnv-label').html(title);
        this.closeTitle(evt);

        return false;
    },

    closeTitle: function (evt) {
        evt.preventDefault();
        AJS.$('.cnv-wizard-header', this.$el).removeClass('editing');
    },

    cancelEdit: function (evt) {
        AJS.$(evt.target)
            .parents('.field-group')
                .removeClass('editing');
    },

    firstUser: function () {
        var username = window.parent.AJS.params.remoteUser,
            $user = AJS.$('#user1', this.$el);

        $user.val(username);

        fullname = username;
        this._selected($user, username, fullname);
        var $this = this;
        AJS.$.getJSON(AJS.Confluence.getContextPath() + "/rest/prototype/1/user/current", function (userDetails, textStatus, jqXHR) {
                    if (userDetails != undefined) {
                        $this._selected($user, userDetails.name, userDetails.displayName);
                    }
                }
            );
    },

    _autocomplete: function() {
        // new autocomplete method is introduced in Conf 5.1-m2 
        // autocompleteUser is not working anymore (not backward comp.
        // seems to be replaced by autocompleteUserOrGroup
        if(Confluence.Binder.autocompleteUserOrGroup) {
            Confluence.Binder.autocompleteUserOrGroup();
        }
        else {
            Confluence.Binder.autocompleteUser();
        }
    },

    selected: function(event, selected) {
        event.preventDefault();

        this.$uiStep1.hide();

        var $user = AJS.$(event.target),
            username = $user.val(),
            fullname = selected.content.title;

        this._selected($user, username, fullname);
    },

    _selected: function ($user, username, fullname) {
        var $fieldgroup = $user.parents('.field-group');

        // new user
        if($fieldgroup.hasClass('active')) {
            this.addUserField(username, fullname);    
            $fieldgroup.removeClass('active');
        }
        // edit user
        else {
            this.changeUserField($user.attr("data-cml-user"), username, fullname);
            $user.parents('.field-group')
                    .removeClass('editing');
        }

        $fieldgroup.find('.cnv-taskuser-info')
                        .html(this.templateUserInfo({
                            user: username,
                            fullname: fullname,
                            baseUrl: AJS.params.baseUrl
                        }))
        
        $user.attr("data-cml-user", username);

        var board = this.makeBoard();
        // adding a new row for preview
        board = this.addExtraRow(board);
        this.renderPreview(board.toJSON());
    },

    addExtraRow: function (b) {
        // adding a new row for preview
        var boardP = new Canvas.Model.Board(_.clone(this.board));
        var emptyRow = boardP.get("row").models[0];
        emptyRow.get("column").each(function(column) {
            column.set({
                header: ""
            });
        });
        b.get("row").add(emptyRow);

        return b;
    },

    clearError: function() {
        this.$error.html("");
    },

    error: function(message) {
        this.$error.html(message);
    },

    addUserField: function (user, fullname) {
        // register new user
        this.users.push({
            name: fullname,
            user: user
        });

        // add new input field for new row
        this.j++;
        this.$('form fieldset').append(this.templateUser({n: this.j}));
        AJS.$('#user' + this.j).focus();
    },

    changeUserField: function (replace, toUsername, toFullname) {
        var i = 0,
            found = false;

        if(replace != '') {
            // find slot and replace the users name
            while(!found && i < this.users.length) {
                if(this.users[i].user== replace) {
                    this.users[i].user = toUsername;
                    this.users[i].name = toFullname;
                    found = true;
                }
                i++;
            }    
        }             
    },

    removeUserField: function (evt) {
        var id = this.$(evt.target).attr('data-cml-remove');
        var $field = this.$('#' + id);

        // not allow to remove first user
        if(id != 'user1') {
            if($field.val() != '') {
                var i = 0,
                    found = false;

                // find user to remove
                while(!found && i < this.users.length) {
                    if(this.users[i] == $field.attr('data-cml-user')) {
                        found = true;
                    }
                    else {
                        i++;
                    }
                }
                // move position-1 all users at right of array
                while(i < this.users.length) {
                    this.users[i] = this.users[i+1];
                    i++;
                }
                trash = this.users.pop();

                // remove row
                $field.parents('.field-group').remove();

                var board = this.makeBoard();
                board = this.addExtraRow(board);
                this.renderPreview(board.toJSON());

                this._autocomplete();
            }
        }
    },

    convertToLabel: function(input) {
        var specialCharacter = /[\s:;\.?&\[\]\(\)#\^!<>~\*]+/g; // allowing the @ character deliberately
        var lo = input.toLowerCase();
        return lo.replace(specialCharacter, '')
    },

    makeBoard: function() {
        var count = 0,
            board = new Canvas.Model.Board(_.clone(this.board)),
            emptyRow = board.get("row").models[0];

        board.get("row").remove(emptyRow);

        for (var i = 0; i < this.users.length; i++) {
            var user = this.users[i].user;
            var fullname = this.users[i].name;
            var row = emptyRow.clone();
            board.get("row").add(row);
            if (count > 0) {
                row.get("column").each(function(column) {
                    column.set({header: ""});
                });
            }
            var first = true;
            row.get("column").each(function(column) {
                column.get("container").each(function(container) {
                    container.set({
                        key: container.get("key") + ",@" + user
                    });
                    if (first) {
                        container.set({caption: "@" + user});
                        first = false;
                    }
                });
            });
            count++;
        }

        return board;
    },

    render: function() {
        this.$el.html(this.template({}));
        return this;
    },

    renderPreview: function(board) {
        this.$preview.html(Canvas.WizardTemplates.boardPreview({
            board: board,
            showCaptions : false
        }));
    }
});