/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

Canvas = Canvas || {};
Canvas.View = Canvas.View || {};
Canvas.View.BoardPicker = Backbone.View.extend({
	className: Canvas.CML.blueprint ? "aui-dialog cnv-blueprint" : "aui-dialog aui-theme-default",
    LOCAL_REPOSITORY_URL: AJS.$('meta[name=context-path]').attr("content") + "/rest/adhoclists/1/exchange/boards",
    // Must be http
    // See point 7 at http://blogs.msdn.com/b/ieinternals/archive/2010/05/13/xdomainrequest-restrictions-limitations-and-workarounds.aspx
    REMOTE_REPOSITORY_URL: "http://exchange.adhoccanvas.com/rest/exchange/1/boards",
    REMOTE_REPOSITORY_HTTPS:  "https://wiki.comalatech.com/rest/exchange/1/boards",

    height: Canvas.CML.blueprint ? "342" : "372",
    welcomeCookie: 'com.comalatech.adhoccanvas.picker.welcome',
    welcomeSkip: false,     // Skip the welcome panel
    waitTimeout: 10000,     // 10 seconds
    
    events: {
        "click a.close" : "close",
        "click button.insert" : "insert",
        "click .back" : "back"
    },
    
    back: function (evt) {
        if(AJS.$(evt.currentTarget).prop('disabled') != 'disabled') {
            window.history.back();
        }
    },
    
    initialize: function() {
        this.welcomeSkip = (Canvas.CML.Cookies.read(this.welcomeCookie) === 'skip');
        this.menu = new AUI.PageMenu(this.height);
        this.panels = new AUI.DialogPanels(this.height);
        this.buttons = new AUI.DialogButtonPanel();

        this.localCategories = new Canvas.Collection.Categories(this.LOCAL_REPOSITORY_URL);
        var remoteUrl = window.location.protocol === "https:" ? this.REMOTE_REPOSITORY_HTTPS : this.REMOTE_REPOSITORY_URL;
        this.remoteCategories = new Canvas.Collection.Categories(remoteUrl, true);

        this.localCategories.bind("reset",this.renderMenu,this);
        this.remoteCategories.bind("reset",this.renderRemoteMenu,this);

        this.history = new Canvas.Router();
        Backbone.history.start({
            pushState: true,
            root: AJS.Confluence.getContextPath()+'/plugins/servlet/adhoccanvas/picker/'
        });
    },

    setChosenTable: function(table, hideInsert) {
        this.table = table;
        if (! hideInsert) {
            this.$insert.removeAttr("disabled");
        }
        this.$back.removeAttr("disabled");
        // wizard stuff
        window.parent.jQuery(".cnv-wizard-picker").trigger("canvas:enableCreate",table);
    },

    disableInsert: function() {
        this.$insert.prop("disabled","disabled");
        this.$back.prop("disabled","disabled");
        window.parent.jQuery(".cnv-wizard-picker").trigger("canvas:disableCreate");
        // wizard stuff
    },

    configure: function() {
        this.configure = new Canvas.View.ConfigurePanel({model: this.table});
        AUI.DialogPanels.addPanel(this.configure.render().el);
        AJS.tabs.setup();
        AUI.DialogPanels.selectPanel(this.configure.$el);
    },

    insert: function() {
        if (this.table.collection && this.table.collection.exchange) {
            Canvas.CML.analytics.trackEvent('board','insert cex',this.table.get("name"));
        }
        if (this.table.build) {
            this.table.build();
        }
        var board = this.table.get("board");
        if (board) {
            board = board.toJSON();
            if (this.table.get("id") == 0) {
                delete board['description'];
            }
            var table = {
                    table: board,
                    filter: this.table.get("filter").toJSON(),
                    id: this.table.get("id")
            };
            window.parent.jQuery("iframe.canvas").trigger("canvas.select",table);        
        }
        else {
            // No board was retrieved
            AJS.$('#cnv-message-bar-import').html('');
            AJS.messages.warning('#cnv-message-bar-import', {
                title: 'The markup syntax is not valid',
                shadowed: false
            })
        }
    },

    close: function() {
        window.parent.jQuery("iframe.canvas").trigger("canvas.close")
    },

    render: function() {
        var dialog = new AUI.DialogComponents({
            title: AJS.I18n.getText('adhoclists.rte.boards.insert'),
            hideTitle: Canvas.CML.blueprint
        });
        dialog.render();
        this.renderButtons();
        dialog.$el.append(this.menu.render().el);
        dialog.$el.append(this.panels.render().el);

        // Help link
        AJS.$('h2', dialog.$el).prepend(this.helpLink());

        this.buttons.render();
        if (! Canvas.CML.blueprint) {
            dialog.$el.append(this.buttons.el);
        }
        this.$el.append(dialog.el);
        this.localCategories.fetch();
        this.welcome();

        return this;
    },

    /*
     * Create dialog help link to Canvas Board Picker doc page 
     */
    helpLink: function () {
        var a = AJS.$('<a></a>', {
            href: 'https://apps.appf.re/cbdc/doc/getstarted/iWeUng',
            target: '_blank',
            "class" : 'cml-help-link'
        }).html('Help');

        help = AJS.$('<div></div>', {
            'class' : 'dialog-help-link'
        }).html(a);

        return help;
    },

    renderButtons: function() {
        this.buttons.$el.append(this.make("buton", {"class": "aui-button back", "disabled" : "disabled"},AJS.I18n.getText('adhoclists.rte.previous')));
        this.buttons.$el.append(this.make("button", {"class": "aui-button aui-button-primary insert", "disabled" : "disabled"},AJS.I18n.getText('insert.name')));
        this.buttons.$el.append(this.make("a", {"class": "aui-button close aui-button-link"},AJS.I18n.getText('cancel.name')));
        this.$insert = this.buttons.$(".insert");
        this.$back = this.buttons.$(".back");
    },

    renderCategories: function(categories) {
        categories.each(function(category) {
            var menuItem = new AUI.PageMenuItem({
                target: category.get("id"),
                caption: category.get("name")
            });
            this.menu.$el.append(menuItem.render().el);
            menuItem.$el.addClass(category.get("id"));
                if (categories.exchange) {
                    menuItem.$el.addClass("exchange")
            }
            
            var panel = new Canvas.View.TableListings({
                id: category.get("id"),
                    model: category
            });
            this.panels.$el.append(panel.el);
        },this);
    },
    
    renderRemoteMenu: function() {
        this.cancelTimeout();
        this.discoverMoreBoardsMenuItem.remove();
        this.menu.$el.append(new AUI.PageMenuSeparator({
            caption: AJS.I18n.getText("adhoccanvas.exchange.boards.discover")
        }).render().el);
        this.renderCategories(this.remoteCategories);
        AJS.$(".page-menu-item.exchange:first button").click()
    },
    
    
    renderMenu: function() {
        this.renderCategories(this.localCategories);
        this.discoverMoreBoardsMenuItem = new AUI.PageMenuItem({
                    target: "discover-boards",
                    caption: AJS.I18n.getText("adhoccanvas.exchange.boards.discover"),
                    select: _.bind(function() {
                        this.oTimeout = window.setTimeout(this.timeout.bind(this), this.waitTimeout);
                        this.remoteCategories.fetch({
                            error: this.timeout
                        });
                    }, this)
                });
        this.menu.$el.append(this.discoverMoreBoardsMenuItem.render().el);
        
        if(this.welcomeSkip || Canvas.CML.blueprint) {
            AJS.$(".page-menu-item:first button").click();
        }
    },
    
    /**
     * If no welcome skip cookie is detected, show welcome panel
     */
     welcome: function () {
        if(! this.welcomeSkip && ! Canvas.CML.blueprint) {
            var view = new WelcomeView(this.welcomeCookie);
            this.panels.$el.append(view.render().el);
            console.log("navigate to welcome");
            this.history.navigate("welcome", {trigger: true});
        }
    },
    
    timeout: function () {
        AJS.messages.warning('.dialog-page-menu', {
            body: "Sorry, we can't connect right now!"
        });
        
        AJS.$('.dialog-page-menu .selected').removeClass('loading');
    },
    
    startTimeout: function () {
        this.oTimeout = window.setTimeout(this.timeout.bind(this), this.waitTimeout);
    },
    
    cancelTimeout: function () {
        window.clearTimeout(this.oTimeout);
    }
});

var WelcomeView = Backbone.View.extend({
    id: 'welcome',
    className: 'dialog-panel-body panel-body welcome',
    template: Canvas.Picker.welcome,
    cookie: '',
    
    events: {
        'click #skipmessage': 'skipmessage'
    },
    
    initialize: function (cookie) {
        this.cookie = cookie;
    },
    
    render: function () {
        console.log("WelcomeView : render");
        this.$el.html(this.template());
        return this;
    },
    /**
     * Checkbox creates a cookie to skip welcome message
     */
    skipmessage: function (evt) {
        var checked = evt.currentTarget.checked;
        if(checked && this.cookie) {
            Canvas.CML.Cookies.create(this.cookie, 'skip', 30 * 12 * 5); // 5 years
            AJS.$(".page-menu-item:first button").click();
        }
        else {
            Canvas.CML.Cookies.erase(this.cookie);
        }
    }
});


Canvas.Router = Backbone.Router.extend({
    routes: {
        "category/:id" : "category",
        "welcome" : "welcome"
    },
    
    category: function (id) {
        AJS.$('.' + id + ' button').click();
    },
    
    welcome: function () {
        console.log("Backbone router : welcome");
        AUI.DialogPanels.selectPanel(AJS.$('#welcome'), null);
        AJS.$('.page-menu-item').removeClass("selected loading");
    }
});