/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var requirejs = require || requirejs;

// Function that calculates the visible area of a jquery element
// according to current screen size and scroll position
function getViewport($board) {
    var $header = $('#header');
    var $mainHeader = $('#main-header');
    var mainHeaderTop = parseInt($mainHeader.css('top'));
    if (isNaN(mainHeaderTop)) {
        mainHeaderTop = $mainHeader.position().top;
    }
    var headerHeight = Math.max($header.height(), $mainHeader.height() + mainHeaderTop);
    var viewport = { top: 0, bottom: 0 };
    var windowScroll = $(window).scrollTop();
    var windowHeight = $(window).height();
    var windowBottom = windowHeight + windowScroll;
    var offset = $board.offset();
    if (offset) {
        var height = $board.height();
        var bottom = offset.top + height;
        viewport.top = Math.max(windowScroll - offset.top + headerHeight, 0);
        viewport.bottom = Math.max((windowBottom - bottom) * -1, 0);
        viewport.height = Math.max(height - viewport.top - viewport.bottom, 0);
    }
    return viewport;
}

// Function that determines if the given jquery element
// is within the visible range of the screen
function isWithinVisibleArea($board) {
    var windowScroll = $(window).scrollTop();
    var windowHeight = $(window).height();
    var windowBottom = windowHeight + windowScroll;
    var offset = $board.offset();
    var bottom = offset.top + $board.height();
    return offset.top < windowBottom && bottom > windowScroll;
};

requirejs(['cb', 'jquery', 'underscore'], function (CB, $, _) {
    CB.toInit(function() {
        var buffer = []; // will contain all the boards that have to load
        var boards = []; // will contain all the boards that have been loaded

        // lazy load boards sequentially one after another
        // gets the first visible board from the buffer and renders it
        // since on the onLoad option of each board there is another call to this function
        // as soon as the first visble board has finished loading
        // the next visible board will be processed
        // and so on, and so forth...
        function processBuffer() {
            var b, board;
            $.each(buffer, function(i, item) {
                // get the first board from the buffer
                // that is within the visible area
                if (isWithinVisibleArea(item.$el)) {
                    b = i;
                    board = item;
                    return false;
                }
                return true;
            });
            if (board) {
                // if there was a board within the visible area, load it
                board.render();
                // move the board out of the buffer
                // so it doesn't get processed again
                boards.push(board);
                buffer.splice(b, 1);
            }
        };

        // create iframes for all board macros in the document
        // and store them in the buffer
        $('.cb-board[data-cb-board]').each(function (i, item) {
            var $target = $(item);
            var customData = { board: $target.data('cbBoard') };
            var spaceKey = $target.data('cbSpaceKey') || CB.getParam('spaceKey');
            var pageId = $target.data('cbPageId') || CB.getParam('pageId');
            var board = new CB.iframe({
                id: CB.format('board-{0}-{1}', pageId, i),
                url: CB.format('/boards/render/{0}/{1}', spaceKey, pageId),
                height: '200',
                customData: customData,
                loadEvent: 'AppLoad',
                reloadEvent: 'AppReload',
                onResize: function() {
                    // get the current viewport after resize
                    this.send('viewport', getViewport(this.$el));
                },
                onLoad: function() {
                    // lazy load next board within the visible area
                    processBuffer();
                },
                query: { readOnly: $target.data('cbReadonly') }
            });
            $target.append(board.$el);
            buffer.push(board);
        });

        // begin lazy loading of boards
        processBuffer();

        // load boards that are within the visible area after table sort
        $('table').on('sortEnd', processBuffer);

        // update board viewport when the user scrolls
        $(window).scroll(_.debounce(function () {
            $.each(boards, function (i, board) {
                if (isWithinVisibleArea(board.$el)) {
                    // viewport will only be updated
                    // if the board is within the visible area
                    board.send('viewport', getViewport(board.$el));
                }
            });
            // lazy load boards that enter the visible area after scrolling
            processBuffer();
        }, 66));

        // update iframe sizes when the user resizes the window
        $(window).resize(_.debounce(function () {
            $.each(boards, function (i, board) {
                board.send('autoResize');
            });
        }, 66));

        // start austoscrolling when a dragged card is close to the viewport edge
        var autoScroll;
        CB.connect('startScroll', function (increment, source) {
            // until stopScroll is triggered
            // or the scroll position  gets to the edge of the iframe,
            // it will keep autoscrolling
            autoScroll = window.setInterval(function() {
                var viewport = getViewport($(source.frameElement));
                if (
                    (increment > 0 && viewport.bottom > 0)
                    || (increment < 0 && viewport.top > 0)
                ) {
                    $(window).scrollTop($(window).scrollTop() + increment);
                    const event = document.createEvent('UIEvents');
                    event.initUIEvent('scroll', true, true, window, 1);
                    document.body.dispatchEvent(event);
                } else {
                    window.clearInterval(autoScroll);
                }
                $(window).one('mouseUp', function() { window.clearInterval(autoScroll); });
            }, 100);
        });
        // stop autoscrolling
        CB.connect('stopScroll', function () {
            window.clearInterval(autoScroll);
        });
    });
});
