/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var debugo, evento;

var Adhoc = Adhoc || {};
Adhoc.Board = Adhoc.Board || {};

Adhoc.Board.textAreaHintShown = false;
Adhoc.Board.textAreaEnterCount = 0;


(function ($) {
    var addPageDialog;
    var heightAdjusted = false;

    var adjustContainerHeights = function () {
        containersFullHeight(false);
        try {
            // Match height: Across the board
            $("table.matchheight").each(function () {
                // reset styles of containers
                resetContainerHeights($(this));

                // if the board is in a hidden container div, display it until we calculate the height
                var $hiddenContainer = $(this).parents(':hidden').last();
                $hiddenContainer.show();

                // find max height
                var max = 0;
                var $columns = $("td", $(this));
                $columns.each(function (i, col) {
                    var colHeight = $(col).height();
                    if (colHeight > max) {
                        max = colHeight;
                    }
                });

                // hide the container again
                $hiddenContainer.css('display', '');

                // adjust heights
                if (max) {
                    $('.boardcontainer', $columns).each(function (i, col) {
                        $(col).css("min-height", max + 'px');
                    });
                }

                heightAdjusted = true;
            });

            // Match height: Across the rows
            $("tr.matchheight").each(function () {
                // reset styles of containers
                resetContainerHeights($(this));

                var max = 0;
                var $columns = $("td", $(this));

                // if the board is in a hidden container div, display it until we calculate the height
                var $hiddenContainer = $(this).parents(':hidden').last();
                $hiddenContainer.show();

                // first let's adjust the containers heights, based on number of containers
                for (var i = 1; i < 10; i++) {
                    for (var j = 0; j < i; j++) {
                        max = 0;
                        var checked = [];
                        $columns.each(function () {
                            var containers2 = $(".boardcontainer", $(this));
                            if (containers2.length === i) {
                                checked.push(containers2[j]);
                                var height = $(containers2[j]).height();
                                max = height > max ? height : max;
                            }
                        });
                        if (max) {
                            for (var k = 0; k < checked.length; k++) {
                                $(checked[k]).css("min-height", "" + max + "px");
                            }
                        }
                    }
                }

                // now let's do a final adjustments, based on the total height
                max = 0;
                var heights = [];
                $columns.each(function () {
                    var height = 0;
                    var gap = 0;
                    $(".boardcontainer", $(this)).each(function () {
                        height += gap + $(this).height();
                        gap += 4;
                    });
                    heights.push(height);
                    if (height > max) {
                        max = height;
                    }
                });

                var ix = 0;
                $columns.each(function () {
                    var height = heights[ix++];
                    if (height < max) {
                        var containers2 = $(".boardcontainer", $(this));
                        var lastContainer = $(containers2[containers2.length - 1]);
                        var newHeight = lastContainer.height() + (max - height);
                        if (newHeight) {
                            lastContainer.css("min-height", "" + newHeight + "px");
                        }
                    }
                });

                // hide the container again
                $hiddenContainer.css('display', '');

                heightAdjusted = true;
            });

            containersFullHeight(true);
        } catch (e) {
            console.error(e);
        }

        // adjust vertical align of header and footer of each row
        alignVertical('.cml-vertical', '.row-header');
    };

    var containersFullHeight = function (apply) {
        if (apply && heightAdjusted) {
            $('.adhocboard').addClass('cnv-height-adjusted');
        } else {
            $('.adhocboard').removeClass('cnv-height-adjusted');
        }
    };

    var resetContainerHeights = function (scope) {
        var $containers = $(".boardcontainer", scope);
        $containers.removeAttr("style");
    };

    var setSpinner = function ($container) {
        var $messageBox = $(".adhocerror", $container);
        Adhoc.addMediumSpinner($messageBox);
        $messageBox.removeClass("hidden");
    };

    var hideSpinner = function ($container) {
        var $messageBox = $(".adhocerror", $container);
        $messageBox.addClass("hidden").html("");
    };

    var combineLabelKeys = function (key1, key2) {
        var separator = "";
        // since "" is a valid key the combined key can be ",filterlabel"
        if (key2.length) {
            separator = ",";
        }
        return key1 + separator + key2;
    };

    var move = function ($card, $from, $to) {
        var $toBoard = $to.parents(".adhocboard");
        var toFilter = new Canvas.Filter($toBoard);
        var $fromBoard = $from.parents(".adhocboard");
        var fromFilter = new Canvas.Filter($fromBoard);
        var $nextCard = $card.next(".boardcontent");
        var filterLabels = "";

        if ($from.find(".boardcontent").length === 0) {
            $from.append("<div class='emptyspace'></div>");
        }

        $to.find(".emptyspace").remove();

        var params = {
            fromColour: $(".containerheader", $from.parents(".boardcontainer")).css("background-color"),
            toColour: $(".containerheader", $to.parents(".boardcontainer")).css("background-color"),
            fromTitle: $from.attr("containercaption"),
            toTitle: $to.attr("containercaption"),
            notify: Adhoc.Board.notifyOnMove && ($toBoard.find("input#sendnotifications").val() || false)

        };
        if ($nextCard.length > 0) {
            var nextPageId = $nextCard.attr("pageid");
            if (nextPageId) {
                params.nextId = nextPageId;
            } else {
                params.nextCard = $(".currentBody", $nextCard).text();
            }
        }

        var pageId = $card.attr("pageid");
        var url, body = "";
        params.from = $from.attr("containerid");
        params.to = $to.attr("containerid");
        if (pageId && pageId > 0) {
            url = AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + pageId + "/move";
            params.containingPageId = AJS.params.pageId || 0;
            params.fromFilter = fromFilter.labels;
            params.toFilter = toFilter.labels;
        } else {
            url = AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + (AJS.params.pageId || 0) + "/card/move";
            body = $.trim($(".currentBody", $card).text());
        }
        url = url + "?" + $.param(params);
        $.ajax({
            type: "POST",
            cache: false,
            contentType: "application/json",
            data: body,
            url: url,
            success: function (action, textStatus, jqXHR) {
                adjustContainerHeights();
            },
            error: function (jqXHR, textStatus, errorThrown) {
                Adhoc.common.errorHandler($to, jqXHR, textStatus, errorThrown);
            }
        });
        return false;
    };

    /*
     * Adjust vertical align of vertical headers and footers of each row
     * Calculates header and footer
     */
    alignVertical = function (target, parent) {
        AJS.$(parent + ' ' + target).each(function () {
            var $item = AJS.$(this),
                $parent = $item.parents(parent),
                hContainer = $parent.height(),
                h = $item.width();

            // We want to save the original height for further recalculations
            // because after positioning we will force max-width to 16px
            if ($item.attr('data-vheight')) {
                h = $item.attr('data-vheight');
            } else {
                $item.attr('data-vheight', h);
            }

            $item.css({
                'margin-bottom': ((hContainer - h) / 2) + 'px',
                'display': 'block'
            });

            // hack for IE9
            if (navigator.userAgent.indexOf('MSIE 9') > 0) {
                $item.css({
                    'width': '16px'
                });
            }

            $parent.css({
                'max-width': '16px',
                'width': '16px'
            });
        });
    };

    /**
     * If press ESC key, close the form
     */
    var enableEscCloseForm = function () {
        $('body').bind('keyup', function (evt) {
            if (evt.keyCode == 27) {
                closeAddCardForm();
            }
        });
    };

    /**
     * terminate keyup handler
     */
    disableEscCloseForm = function () {
        $('body').unbind('keyup');
    };

    /**
     * Close add card form
     */
    var closeAddCardForm = function () {
        containersFullHeight(true);
        $(".boardcontent.new").addClass("hidden");
        $(".boardcontent").trigger("unedit");
        $(".adhocerror").html('');
        disableEscCloseForm();
        endForm();
    };

    /**
     * Checkbox: press enter to save
     * swap enter to shift+enter mode depending checkbox and cookies
     * cookie is created when checkbox 'press Enter to save' is unchecked
     */
    var checkSaveKey = function (evt) {
        var cookieName = 'com.comalatech.adhoccanvas.savecardshift';
        var checked = evt.currentTarget.checked;
        if (!checked) {
            Canvas.CML.Cookies.create(cookieName, 'true', 30 * 12 * 5); // 5 years
            checkAll(".boardcontent .savekey", false);
            $('.adhoc-btn-save').show('fast');
        } else {
            // checkbox is checked, so we erase cookie
            Canvas.CML.Cookies.erase(cookieName);
            checkAll(".boardcontent .savekey", true);
            $('.adhoc-btn-save').hide('fast');
        }

        evt.stopPropagation();
    };

    var doNothing = function (evt) {
        evt.stopPropagation();
    };

    var checkContainerClick = function (event) {
        if ($(event.target).parents(".no-results").length) {
            return;      // if it's create page - let it bubble
        }
        doNothing(event);
    };

    /**
     * add handlers to form
     * checkbox 'press enter to save'
     * clicking outside card container will close form
     */
    var initForm = function ($container) {
        containersFullHeight(false);
        $(".savekey", $container).bind("click", checkSaveKey);
        $($container).bind("click", checkContainerClick);
        $('html').bind('click', closeAddCardForm);
    };

    /**
     * kill handlers defined to form
     */
    var endForm = function () {
        $(".savekey").unbind("click", checkSaveKey);
        $(".boardcontent").unbind("click", checkContainerClick);
        $('html').unbind('click', closeAddCardForm);
    };

    var enableAddCard = function ($container) {
        var $form = $(".boardcontent.new", $container);

        // first close previous opened forms
        closeAddCardForm();

        $form
            .removeClass("hidden newpage")
            .addClass("newcard");

        $(".description", $form).removeClass("hidden");
        $(".cross-space-search", $form).addClass("hidden");

        $("textarea", $form)
            .removeClass("hidden")
            .val("")
            .css("height", "37px")
            .focus();

        $("input.newcontainerpage", $form)
            .addClass("hidden")
            .val("");

        // If press ESC key, close the form
        enableEscCloseForm();
        initForm($form);
    };

    var enableAddPage = function ($container) {
        var $form = $(".boardcontent.new", $container);

        // first close previous opened forms
        closeAddCardForm();

        $form
            .removeClass("hidden newcard")
            .addClass("newpage");

        $(".description", $form).addClass("hidden");
        $(".cross-space-search", $form).removeClass("hidden");

        $("textarea", $form)
            .addClass("hidden")
            .val("");

        $("input.newcontainerpage", $form)
            .removeClass("hidden")
            .val("")
            .focus();


        // If press ESC key, close the form
        enableEscCloseForm();
        initForm($(".boardcontent.new", $container));
    };

    /**
     * Find checkboxes matching 'selector' and un/check them
     */
    var checkAll = function (selector, bCheck) {
        $(selector).each(function (i, chk) {
            chk.checked = bCheck;
        });
    };

    var configSaveCheckbox = function () {
        // Set checkbox 'press enter to save' depending on cookie
        if (Canvas.CML.Cookies.read('com.comalatech.adhoccanvas.savecardshift') == 'true') {
            checkAll(".boardcontent .savekey", false);
        } else {
            $('.adhoc-btn-save').hide();
        }
    };

    var initCards = function (trigger) {
        // Set checkbox 'press enter to save' depending on cookie
        configSaveCheckbox();
        if (trigger) {
            Adhoc.initializePagePopupDialog($(trigger).get(0));
        }
    };

    var _renderNewCard = function ($trigger, card, $container) {
        var continueAdding = true;
        var newCard = Adhoc.sanitizeBoardContent(AJS.template.load("board-page").fill(card).toString());

        if ($trigger && $trigger.hasClass("cardtopage")) {
            var $oldCard = $trigger.parents(".boardcontent");
            $oldCard.after(newCard);
            $oldCard.remove();
            continueAdding = false;
        } else {
            $(".containerbody .boardcontent.new", $container).after(newCard);
        }
        $container.find(".emptyspace").remove();
        $("input.newcontainerpage", $container).val("");
        endForm();
        initCards('#pageid_' + card.id);
        adjustContainerHeights();
        if (continueAdding) {
            enableAddPage($container);
        }
        hideSpinner($container);
    };

    /**
     * add a page to a container. For existing pages apply the label, for new pages, add the page and apply the label
     * @param $container - the container the page will be added to
     * @param $trigger - has info wheather it was a page or card that has been added. needed for rendering the added card
     * @param pageName - optional: used for new pages that have been added via a popup.
     * @returns {boolean}
     */
    var addPage = function ($container, $trigger, pageName) {
        setSpinner($container);
        pageName = pageName || "";
        $(".boardcontent.new form.aui div.autocomplete", $container).html("");
        var $theBoard = $container.parents(".adhocboard");
        var type = $theBoard.attr("boardType");
        var filter = new Canvas.Filter($theBoard);
        var containerId = $container.attr("containerid");
        var name = $("input.newcontainerpage", $container).val();
        if (pageName.length > 0) {
            name = pageName;
        }
        var wiki = $("textarea.newcardbody", $container).val();
        var pageId = AJS.params.pageId || 0;
        var notify = Adhoc.Board.notifyOnMove && ($theBoard.find("input#sendnotifications").val() || false);
        var toColour = $(".containerheader", $container).css("background-color");
        var toTitle = $(".containerbody", $container).attr("containercaption");
        var pageSpaceKey = $("input.selected-page-space-key", $container).val();
        pageSpaceKey = pageSpaceKey || AJS.params.spaceKey;


        $(".boardcontent.new", $container).addClass("hidden");
        $.ajax({
            type: "POST",
            contentType: "application/json",
            cache: false,
            url: AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + AJS.params.spaceKey + "/page/add?" +
                "type=" + type +
                "&containerId=" + containerId +
                "&filterKey=" + filter.labels +
                "&title=" + encodeURIComponent(name) +
                "&pageId=" + pageId +
                "&parentId=" + filter.parentId +
                "&toTitle=" + toTitle +
                "&color=" + toColour +
                "&notify=" + notify +
                "&pageSpaceKey=" + pageSpaceKey,
            data: wiki,
            success: function (card, textStatus, jqXHR) {
                card.contextPath = AJS.Confluence.getContextPath();
                _renderNewCard($trigger, card, $container);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                Adhoc.common.errorHandler($container, jqXHR, textStatus, errorThrown);
            }
        });
        return false;
    };


    var addCard = function ($container, $trigger) {
        var $card = $(".boardcontent.new", $container);
        setSpinner($card);

        var type = $(".adhocboard").attr("boardType");
        var $theBoard = $container.parents(".adhocboard");
        var filterLabels = "";
        var containerId = $container.attr("containerid");
        var body = $.trim($("textarea", $card).val());
        var pageId = AJS.params.pageId || 0;
        var notify = Adhoc.Board.notifyOnMove && ($theBoard.find("input#sendnotifications").val() || false);
        var toColour = $(".containerheader", $container).css("background-color");
        var toTitle = $(".containerbody", $container).attr("containercaption");

        $.ajax({
            type: "POST",
            contentType: "application/json",
            cache: false,
            url: AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + pageId +
                "/card/add" +
                "?containerId=" + combineLabelKeys(containerId, filterLabels) +
                "&notify=" + notify +
                "&color=" + toColour +
                "&toTitle=" + toTitle,
            data: body,
            success: function (card, textStatus, jqXHR) {
                $(".boardcontent.new", $container).addClass("hidden");

                card.contextPath = AJS.Confluence.getContextPath();
                var newCard = Adhoc.sanitizeBoardContent(AJS.template.load("board-card").fill(card).toString());
                $(".containerbody .boardcontent.new", $container).after(newCard);
                var $newCard = $("#" + card.elementId);
                $(".renderedbody", $newCard).html(card.bodyHtml || card.body);
                $container.find(".emptyspace").remove();
                $("input.newcontainerpage", $container).val("");
                initCards('#link-' + card.elementId);
                adjustContainerHeights();
                hideSpinner($container);
                enableAddCard($container);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                Adhoc.common.errorHandler($card, jqXHR, textStatus, errorThrown);
            }
        });
        return false;
    };

    var updateCard = function ($card) {
        setSpinner($card);

        var $container = $card.parents(".boardcontainer");
        var $theBoard = $container.parents(".adhocboard");
        var filterLabels = "";
        var key = combineLabelKeys($container.attr("containerkey"), filterLabels);
        var pageId = AJS.params.pageId || 0;
        var oldBody = $(".currentBody", $card).text();
        var newBody = $("textarea", $card).val();
        var url = AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + pageId + "/card/update?containerId=" + key;
        $.ajax({
            type: "POST",
            contentType: "application/json",
            cache: false,
            data: JSON.stringify({
                body: oldBody,
                newBody: newBody
            }),
            url: url,
            success: function (card, textStatus, jqXHR) {
                var $before = $card.prev();
                $card.remove();
                card.contextPath = AJS.Confluence.getContextPath();
                var updatedCard = Adhoc.sanitizeBoardContent(AJS.template.load("board-card").fill(card).toString());
                $before.after(updatedCard);
                var $newCard = $("#" + card.elementId);
                $(".renderedbody", $newCard).html(card.bodyHtml || card.body);
                endForm();
                initCards('#link-' + card.elementId);
                hideSpinner($container);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                Adhoc.common.errorHandler($card, jqXHR, textStatus, errorThrown);
            }
        });

    };

    var removeCard = function ($card) {
        var $container = $card.parents(".boardcontainer");
        var $theBoard = $container.parents(".adhocboard");
        var filterLabels = "";
        var params = {
            key: combineLabelKeys($container.attr("containerkey"), filterLabels),
            containingPageId: AJS.params.pageId || 0
        };
        var cardPageId = $card.attr("pageid") || 0;
        var url, message, body = "";
        var title = $(".cardheading a.cardtitle", $card).html();
        if (cardPageId > 0) {
            url = AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + cardPageId + "/remove";
            message = AJS.I18n.getText("adhoclists.boards.cards.remove.page.caption");
        } else {
            url = AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + AJS.params.pageId + "/card/remove";
            message = AJS.I18n.getText("adhoclists.boards.cards.remove.caption");
            body = $(".currentBody", $card).text();
        }
        url = url + "?" + $.param(params);
        Adhoc.doIfconfirm(title, message, function () {
            setSpinner($container);
            $.ajax({
                type: "POST",
                contentType: "application/json",
                cache: false,
                data: body,
                url: url,
                success: function (card, textStatus, jqXHR) {
                    $card.remove();
                    hideSpinner($container);
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    Adhoc.common.errorHandler($card, jqXHR, textStatus, errorThrown);
                }
            });
        });
    };

    /**
     * Function to remove all the cards from a given board
     * @param boardId
     */
    var removeBoardCards = function (boardId) {
        var $board = AJS.$(boardId);
        var containerList = [];

        // Retrieve board containers and referenced pages
        var $containers = $board.find('.boardcontainer');
        $containers.each(function () {
            var $container = $(this);
            var referencedPages = [];
            containerList.push({
                container: $container.attr('containerid'),
                pages: referencedPages
            });
            $container.find('a.pagelink').each(function () {
                referencedPages.push($(this).attr('pageid'));
            });
        });

        var title = $board.find('.board-caption > h3').text();
        var message = AJS.I18n.getText("adhoclists.boards.cards.remove.all.caption");

        Adhoc.doIfconfirm(title, message, function () {
            // Show Activity Spinners
            $containers.each(function () {
                setSpinner($(this));
            });

            $.ajax({
                type: "POST",
                contentType: "application/json",
                cache: false,
                data: JSON.stringify(containerList),
                url: Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + AJS.params.pageId + "/remove-cards",
                success: function (data, textStatus, jqXHR) {
                    // Remove cards from containers
                    $board.find('div[id].boardcontent').each(function () {
                        $(this).remove();
                    });
                    // Hide activity spinners
                    $containers.each(function () {
                        hideSpinner($(this));
                    });
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    // Hide activity spinners
                    $containers.each(function () {
                        hideSpinner($(this));
                    });
                    // Show error
                    Adhoc.common.errorHandler($board, jqXHR, textStatus, errorThrown);
                }
            });
        });
    };

    var getAllKeysFromBoard = function ($board) {
        var allKeys = "";
        var separator = "";
        $(".boardcontainer", $board).each(function () {
            var keys = $(this).attr("containerid");
            if (keys.length) {
                allKeys += separator + keys;
                separator = ":";      // special character - not part of the label
            }
        });
        return allKeys;
    };

    var morePages = function (event) {
        event.preventDefault();
        var $container = $(this).parents(".boardcontainer");
        var $board = $container.parents(".adhocboard");
        var filter = new Canvas.Filter($board);
        var allBoardKeys = getAllKeysFromBoard($board);
        var label = $container.attr("containerkey");
        var spaceKey = AJS.params.spaceKey;
        var filterLabels = filter.labels;
        var parentId = filter.parentId;
        var pageId = AJS.params.pageId || 0;
        var start = $container.find(".boardcontent").length - 1;
        var maxRows = $("#boardmaxrows", $board).val() || 20;
        var $spinner = $container.find(".morespinner");
        var $moreCaption = $container.find(".morecaption");
        $moreCaption.hide();
        Adhoc.addSmallSpinner($spinner);
        $.ajax({
            type: "GET",
            cache: false,
            url: AJS.Confluence.getContextPath() + "/rest/adhoclists/latest/boards/" + spaceKey + "/more" +
                "?label=" + label +
                "&labels=" + filterLabels +
                "&allKeys=" + allBoardKeys +
                "&start=" + start +
                "&maxRows=" + maxRows +
                "&parentId=" + parentId +
                "&pageId=" + pageId,
            success: function (moreContent, textStatus, jqXHR) {
                var $moreBtn = $container.find(".containerfooter");
                if (moreContent.pagesHtml) {
                    $(moreContent.pagesHtml).insertBefore($moreBtn);
                }
                $spinner.empty();
                if (!moreContent.more) {
                    $moreBtn.remove();
                } else {
                    $moreCaption.show();
                }
                adjustContainerHeights();
                initCards();
            },
            error: function (jqXHR, textStatus, errorThrown) {
                Adhoc.common.errorHandler($container, jqXHR, textStatus, errorThrown);
            }
        });
    };

    var _findPage = function (spaceKey, title, success, error) {
        _requestRPC({
            method: 'getPage',
            data: [spaceKey, title],
            success: function (response) {
                success(response);
            },
            error: function (err) {
                error(err);
            }
        });
    };

    var _requestRPC = function (options) {
        var method = options.method;
        if (options.debug) {
            console.log('_requestRPC', options);
        }
        $.ajax({
            url: AJS.Confluence.getContextPath() + '/rpc/json-rpc/confluenceservice-v2/' + method,
            type: 'POST',
            contentType: "application/json; charset=utf-8",
            data: JSON.stringify(options.data),
            dataType: "json",
            success: options.success,
            error: options.error
        });
    };

    var _verifyPage = function (pageName, done, fail) {
        _findPage(AJS.params.spaceKey, pageName, function (page) {  // success
            if (page.length === 0) {
                fail();
                return;
            }
            var id = page.id || 0;
            if (id === 0) {
                fail();
                return;
            }
            done();
        }, function (err) {
            fail();
        });
    };

    var createAddPageDialog = function ($contents, trigger, pageName, cardBody) {
        cardBody = cardBody || "";
        $contents.empty();
        var $trigger = $(trigger);
        var $container = $trigger.parents(".boardcontainer");
        var $adhocboard = $trigger.parents(".adhocboard");
        var templateId = 0;
        if (cardBody.length == 0) {
            templateId = $adhocboard.find("#newPageTemplateId").val() || 0;
        }
        var containerKey = $container.attr("containerId");
        var confluenceBuildNumber = parseInt($('input[name="confluenceBuildNumber"]').val());
        // Confluence 6.0.3 build number is 7102 and it does not support wiki markup as part of the editor content
        var isConfluence6_0_3_orHigher = confluenceBuildNumber >= 7102;
        var createUrl = AJS.contextPath() + "/pages/createpage-entervariables.action" +
            "?templateId=" + templateId +
            "&spaceKey=" + AJS.params.spaceKey +
            "&title=" + encodeURI(pageName) +
            "&newSpaceKey=" + AJS.params.spaceKey +
            "&fromPageId=" + AJS.params.pageId +
            //          adding page labels after page has been confirmed  "&createPageLabelsString=" + containerKey +
            "&wysiwygContent=" + (isConfluence6_0_3_orHigher ? cardBody.replace(/{.*?}/g, '') : cardBody) +
            "&boardTemplate=true";
        var width = 960, height = 640;
        var left = (screen.width / 2) - (width / 2);
        var top = (screen.height / 2) - (height / 2);
        var popup = window.open(createUrl, '_blank',
            "height=" + height +
            ",width=" + width +
            ",top=" + top +
            ",left=" + left +
            ",status=no" +
            ",toolbar=no" +
            ",menubar=no" +
            ",location=no");
        $('body').bind('boardTemplate.savePage', function (ev, newTitle) {
            console.log("Received: boardTemplate.savePage", newTitle);
            setTimeout(function () {
                popup.close();
                _verifyPage(newTitle, function () { // done()
                    addPage($container, $trigger, newTitle);
                }, function () {  // fail()
                    hideSpinner($container);
                    console.error("Page not created:", newTitle)
                });
            }, 1400);
            $('body').unbind('boardTemplate.savePage');
        });
    };

    AJS.toInit(function ($) {
        AJS.log("init view boards");
        Adhoc.viewMode = true;
        var sortableOptions = {
            over: function (event, ui) {
                $(".boardcontent.new").addClass("hidden");
                $(".boardcontainer").removeClass("hover");
                $(this).parents(".boardcontainer").addClass("hover");
            },
            items: "div.boardcontent:not(.readonly)",
            stop: function (event, ui) {
                $(".boardcontainer").removeClass("hover");
                var $from = $(this);
                var $to = ui.item.parents(".containerbody");
                var notify = ($("input#sendnotifications", ui.item.parents(".adhocboard")).val().toLowerCase() === 'true') || false;
                var fromEqualsTo = ($from.attr("containerid") == $to.attr("containerid"));
                if (!Adhoc.Board.notifyOnMoveDontAsk && notify && !fromEqualsTo) {
                    var movePageDialog = Adhoc.Board.getMovePageDialog();
                    movePageDialog.dialog.on("hide", function() {
                        move(ui.item, $from, $to);
                    });
                    movePageDialog.show();
                    return true;
                }
                move(ui.item, $from, ui.item.parents(".containerbody"));
            },
            connectWith: ".containerbody"
        };
        if (AJS.params.versionNumber.indexOf("4.2") >= 0) {
            AJS.log("will apply fix to CONF-25069");
            sortableOptions.start = function (event, ui) {
                $("body:not('.quickcommentfixed')").css("overflow-x", "visible").addClass("quickcommentfixed");
            };
        }
        var sortable = $(".adhocboard:not(.readonly) .containerbody").sortable(sortableOptions);
        if (!$.browser.mozilla) {
            try {
                sortable.disableSelection();
            } catch (e) {
            }
        }

        // Enable/Disable cross-space search
        $(document).on('change', ".cross-space-search input:checkbox", function (event) {
            if ($(this).is(':checked')) {
                $(this).parent().siblings(".autocomplete-page").attr("data-spacekey", "");
            } else {
                $(this).parent().siblings(".autocomplete-page").attr("data-spacekey", "AJS.params.spaceKey");
            }
        });

        // Existing page selected event handling
        $(document).on('selected.autocomplete-content', ".autocomplete-page", function (event) {
            var spaceKeyField = $(this).siblings(".selected-page-space-key");
            var pageName = spaceKeyField.val();
            // Switch values to fix known issue with autocomplete component: https://jira.atlassian.com/browse/CONF-33227
            spaceKeyField.val($(this).val());
            $(this).val(pageName);
            $(this).focus();
        });

        // Make sure the spaceKeyField is empty unless selected from autocomplete component
        $(document).on('keyup', ".autocomplete-page", function (event) {
            var keyCode = event.keyCode;
            // If the user presses a key that is not Enter
            if (keyCode != 13) {
                var spaceKeyField = $(this).siblings(".selected-page-space-key");
                spaceKeyField.val("");
            }
        });

        $(document).on('click', ".addboardpage", function (event) {
            event.preventDefault();
            var $container = $(event.currentTarget).parents(".boardcontainer");
            enableAddPage($container);
        });

        $(document).on('click', ".addboardcard", function (event) {
            event.preventDefault();
            var $container = $(event.currentTarget).parents(".boardcontainer");
            enableAddCard($container);
        });

        $(document).on('click', ".boardcontainer .morerows", morePages);

        $(document).on('submit', ".boardcontent.new form", function (event) {
            event.preventDefault();
            var $container = $(this).parents(".boardcontainer");
            if ($(".boardcontent.new", $container).hasClass("newcard")) {
                addCard($container);
            } else {
                var $board = $container.parents(".adhocboard");
                var template = $(".newPageTemplate", $board).text();
                $("textarea.newcardbody", $container).val(template);
                addPage($container);
            }
        });

        $(document).on('keyup', ".boardcontent.new input", function (event) {
            event.preventDefault();
            if (event.keyCode === 27) {
                $(".boardcontent.new").addClass("hidden");
            }
        });
        
        $(document).on('keyup', ".boardcontent.new input", function (event) {
            if ($(this).val().length > 0) {
                $(this).addClass("hasvalue");
            } else {
                $(this).removeClass("hasvalue");
            }
        });

    // Add new card Form

        $(".boardcontent textarea.newcardbody")
        // auto-grow the textarea
            .elastic()

        // submit the card with "enter" key (or shift+enter)   
        $(document).on('keypress', ".boardcontent textarea.newcardbody", function (event) {
            var key = event.keyCode;

            // not ESC key
            if (key != 27) {
                // checkbox 'press Enter to save' is checked
                if (!Canvas.CML.Cookies.read('com.comalatech.adhoccanvas.savecardshift')) {
                    // shift + enter key: next line
                    if (key == 13 && event.shiftKey) {
                        // do nothing, let jump to next line
                    }
                    // ENTER key: submit card
                    else if (key == 13) {
                        event.preventDefault();
                        $(this).parents("form").submit();
                    }
                } else {
                    // SHIFT+ENTER key: submit card
                    if (key == 13 && event.shiftKey) {
                        event.preventDefault();
                        $(this).parents("form").submit();
                    }
                }
            }
        });

        initCards();

        /**
         * Opens a new window dialog to create a new page
         * @private
         */
        function _addNewPage(context) {
            $(".boardcontent.new").addClass("hidden");
            var newPageTitle = $("input.newcontainerpage", $(context).parents("form.aui.newcard")).val();
            createAddPageDialog($(".add-page-dialog"), context, newPageTitle);
        }

        // MIKE 20190919 I'd say this is never used anymore. OBSOLETE?
        addPageDialog = new AJS.Dialog({
            width: 640,
            height: 480,
            id: "newPageDialog",
            closeOnOutsideClick: true
        });
        addPageDialog.addPage("add-page-dialog").addHeader("");

        $(document).on('keypress', "form.newcard .newcontainerpage", function (event) {
            var keyCode = event.keyCode;
            // If the user press Enter
            if (keyCode == 13) {
                var $createNewPageLink = $(this).parent().find('a.no-results');
                if ($createNewPageLink.length) {
                    event.preventDefault();
                    _addNewPage(this);
                }
            }
        });
        $(document).on('click', ".boardcontentbody .aui .no-results", function (event) {
            event.preventDefault();
            _addNewPage(this);
        });
        $(document).on('click', "#newPageDialog input.button.submit", function (event) {
            event.preventDefault();
            var $container = $("#newPageDialog").data("container");
            var $trigger = $("#newPageDialog").data("trigger");
            $("input.newcontainerpage", $container).val($("#newPageDialog .pagetitle").val());
            $("textarea.newcardbody", $container).val($("#newPageDialog .pagewikibody").val());
            addPageDialog.hide();
            addPage($container, $trigger);
        });
        $(document).on('click', "#newPageDialog a.cancel-add", function (event) {
            event.preventDefault();
            addPageDialog.hide();
        });
        $(document).on('click', ".cardtopage", function (event) {
            event.preventDefault();
            var cardId = $(event.currentTarget).data('card');
            var $card = $('#' + cardId);
            var title = $(".cardtitle", $card).text();
            var body = $(".currentBody", $card).text().replace(title, "");
            createAddPageDialog($(".add-page-dialog"), event.currentTarget, title, body);
        });
        $(document).on('click', ".removecard", function (event) {
            event.preventDefault();
            var cardId = $(event.currentTarget).data('card');
            var $card = $('#' + cardId);
            removeCard($card);
        });
        // card editing
        $(document).on('unedit', ".boardcontent", function (event) {
            $(".cardtitle", this).removeClass("hidden");
            $(".editcard", this).addClass("hidden");
            disableEscCloseForm();
        });
        $(document).on('edit', ".boardcontent", function (event) {
            // first close previous opened forms
            closeAddCardForm();

            $(".cardtitle", this).addClass("hidden");
            $(".editcard", this).removeClass("hidden");
            $(".cardlink", this).removeClass("hovering"); // stop card preview

            // focus on end of content
            $textarea = $(".editcard textarea", this);
            var tmp = $textarea.val();
            $textarea.val('')
                .focus()
                .val(tmp)
                .elastic();

            configSaveCheckbox();
            enableEscCloseForm();
            initForm(this);
        });
        $(document).on('click', ".cardlink", function (event) {
            event.preventDefault();
            $(".boardcontent").trigger("unedit");
            var $card = $(this).parents(".boardcontent");
            $card.trigger("edit");
        });
        $(document).on('submit', ".editcard", function (event) {
            event.preventDefault();
            var $card = $(this).parents(".boardcontent");
            updateCard($card);
        });

        // Set show/hide description option for each board in the page
        // Persist selection by cookie and per board
        // cookie's name is "com.comalatech.adhoccanvas.hidedesc." + PageID + BoardID
        var optionTextHide = AJS.I18n.getText('adhoclists.help.boards.description.hide.label');
        var optionTextShow = AJS.I18n.getText('adhoclists.help.boards.description.show.label');

        $.each($(".adhocboard"), function (i, board) {
            var boardid = '#' + $(board).attr("id");
            var id = boardid.split('#board-')[1];
            var boardoptions = '#boardoptions-' + id;
            var cookie = "com.comalatech.adhoccanvas.hidedesc." + AJS.params.pageId + "." + id;

            // if no descriptions on boards, hide the option from dropdown
            $containerdescription = $(boardid + ' .containerdescription');
            if ($containerdescription.length === 0) {
                $(boardoptions + ' .options-desc').hide();
            } else {
                var hideDescription = readCookie(cookie);
                if (hideDescription === 'true') {
                    $containerdescription.toggleClass('hidden');
                }
                $(boardoptions + ' .options-desc a').text(hideDescription === 'true' ? optionTextShow : optionTextHide);

                // Toggle show descriptions
                $(document).on('click', boardoptions + ' .options-desc a', function () {
                    $(boardid + ' .containerdescription').toggleClass('hidden');
                    adjustContainerHeights();
                    var isHidden = $(boardid + ' .containerdescription').hasClass('hidden');
                    $(boardoptions + ' .options-desc a').text(isHidden ? optionTextShow : optionTextHide);
                    // set expire date to 5 years
                    createCookie(cookie, isHidden, 30 * 12 * 5);
                });
            }

            $(document).on('click', boardoptions + ' .options-remove-cards', function (e) {
                e.preventDefault();
                removeBoardCards(boardid);
            });

            // set dropdown board options
            $(boardid + ' .board-caption').dropDown('Standard');
        });

        adjustContainerHeights();

        // Track if at least one board is embedded in page
        Canvas.CML.analytics.trackEvent('board', 'view');
    });

    // TODO These cookie functions are duplicated from welcome.js We should only define once
    //      have to move out of this closure
    var createCookie = function (name, value, days) {
        var expires;
        if (days) {
            var date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toGMTString();
        } else {
            expires = "";
        }
        document.cookie = name + "=" + value + expires + "; path=/";
    };

    var readCookie = function (name) {
        var nameEQ = name + "=";
        var ca = document.cookie.split(';');
        for (var i = 0; i < ca.length; i++) {
            var c = ca[i];
            while (c.charAt(0) == ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0)
                return c.substring(nameEQ.length, c.length);
        }
        return null;
    };

    var eraseCookie = function (name) {
        createCookie(name, "", -1);
    };

})(AJS.$);
