/*
 * Copyright (c) 2022 Appfire Technologies, LLC.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Appfire EULA"
 * (https://appfire.com/eula/) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

var Canvas = Canvas || {};

Canvas.StatefulView = Backbone.View.extend({
    defaults: {},
    options: {},
    initialize: function (options) {
        this.options = _.defaults(options || {}, this.defaults);
        _.bindAll(this, 'setState', 'getState', 'resetState', 'stateChanged', 'afterInit');
        this.state = this.getStateModel();
        this.listenTo(this.state, 'change', function (state, options) {
            var changed = Object.keys(state.changed);
            if (changed.length > 0) {
                _.defer(this.stateChanged, changed, options);
            }
        });
        this.afterInit();
    },
    getStateModel: function () {
        return new Backbone.Model(this.options);
    },
    setState: function (attr, val) {
        return typeof val !== 'undefined' ? this.state.set(attr, val) : this.state.set(attr);
    },
    unsetState: function (attr) { return this.state.unset(attr); },
    getState: function (attr) { return this.state.get(attr); },
    resetState: function () { return this.state.clear(); },
    afterInit: function () {
        // to be overriden
    },
    stateChanged: function () {
        // to be overriden
    }
});

Canvas.CML = Canvas.CML || {
    VERSION: '0.3',

    allVersions: function () {
        if(console.group) {
            console.group('Confluence versions');
        }

        console.log('Confluence:', AJS.params.versionNumber);
        console.log('AJS:', AJS.version);
        console.log('jQuery:', jQuery.fn.jquery);
        console.log('jQuery UI:', jQuery.ui.version);

        console.log('Backbone:', Backbone.VERSION);
        console.log('Underscore:', _.VERSION);

        if(console.group) {
            console.groupEnd();
            console.group('Comalatech versions');
        }

        console.log('CML:', this.VERSION);
        console.log('Canvas:', Canvas.VERSION);

        if(console.group) {
            console.groupEnd();
        }
    },

    /*
     * Equivalent to Adhoc.getUrlParams()
     */
    getUrlParameters: function() {
        var map = {};
        var parts = window.location.href.replace(/[?&]+([^=&]+)=([^&]*)/gi, function(m,key,value) {
            map[key] = value;
        });
        return map;
    },

    /*  copied from Comala Skull (comala-ext)
        When C4C is using Skull, we can remove this and reference directly to comala.js */
    analytics: (function () {
        var version = '0.0';

        var init = function () { };

        var trackPage = function (pageUrl) { };

        var trackEvent = function (category, action, label, value) { };

        var trackName = function (newPrefix) { };

        return {
            init: init,
            trackPage: trackPage,
            trackEvent: trackEvent,
            trackName: trackName,
            VERSION: version
        };
    })(),

    /**
     * Generic class object for handling cookies
     */
    Cookies: {
        create: function(name, value, days) {
            var expires;
            if (days) {
                var date = new Date();
                date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
                expires = "; expires=" + date.toGMTString();
            } else {
                expires = "";
            }
            document.cookie = name + "=" + value + expires + "; path=/";
        },

        read: function(name) {
            var nameEQ = name + "=";
            var ca = document.cookie.split(';');
            for (var i = 0; i < ca.length; i++) {
                var c = ca[i];
                while (c.charAt(0) == ' ') c = c.substring(1, c.length);
                if (c.indexOf(nameEQ) == 0)
                    return c.substring(nameEQ.length, c.length);
            }
            return null;
        },

        erase: function(name) {
            this.create(name, "", -1);
        }
    },
    /* WRAPPERS 4 AJS & CONFLUENCE */
    contextPath: AJS.contextPath(),

    iframe: Canvas.StatefulView.extend({
        className: 'cml-iframe-wrapper',
        defaults: {
            id: null,
            url: '',
            query: {},
            absolute: false,
            customData: null,
            width: '100%',
            height: '100%',
            css: '',
            scrolling: 'no',
            onLoad: function () { }
        },
        getStateModel: function () {
            return new Backbone.Model({ loaded: false });
        },
        afterInit: function () {
            _.bindAll(this, 'load');
            this.params = _.extend({}, this.options.query, {
                cp: this.contextPath,
                xdm_e: AJS.params.baseUrl,
                loc: AJS.params.userLocale || AJS.params.actionLocale,
                user: AJS.params.remoteUser,
                rom: AJS.Meta.get('access-mode') === 'READ_ONLY' ? true : false,
                atlToken: AJS.params.atlToken,
            });
        },
        stateChanged: function (changed) {
            var _self = this;
            var $iframe = _self.$iframe;
            if ($iframe && changed.indexOf('loaded') !== -1) {
                _.defer(function () {
                    var loading = !_self.getState('loaded');
                    $iframe.toggleClass('cml-loading', loading);
                    if (!loading) {
                        var customData = _self.options.customData;
                        var onLoad = _self.options.onLoad;
                        if (customData) {
                            _self.send('customData', customData);
                        }
                        onLoad();
                    }
                });
            }
        },
        getUrl: function (query) {
            var params = _.extend({}, this.params, query || {});
            var queryString = _(Object.keys(params)).map(function (paramKey) {
                return paramKey + '=' + encodeURIComponent(this.params[paramKey]);
            }, this).join('&');
            var baseUrl = Canvas.CML.contextPath + '/plugins/servlet/adhoccanvas';
            return (this.options.absolute ? '' : baseUrl) + this.options.url + '?' + queryString;
        },
        render: function (query) {
            this.$el.html('<iframe frameborder="0" ALLOWTRANSPARENCY="true"'
                + ' src="' + this.getUrl(query) + '"'
                + ' id="' + this.options.id + '_iframe"'
                + ' width="' + (_.isUndefined(this.options.width) ? '100%' : this.options.width) + '"'
                + ' height="' + (_.isUndefined(this.options.height) ? '100%' : this.options.height) + '"'
                + ' scrolling="' + this.options.scrolling + '"'
                + ' class="cml-iframe canvas' + (this.options.css ? ' ' + this.options.css : '') + ' cml-loading"'
                + ' style="position: absolute; border: none"'
                + ' />'
            );
            this.$iframe = this.$el.find('iframe');
            this.$iframe.on('load', this.load);
            return this.el;
        },
        load: function (e) {
            this.setState('loaded', true);
        },
        reload: function (query) {
            this.setState('loaded', false);
            this.$iframe.attr('src', this.getUrl(query));
        },
        send: function (event, data) {
            var iframe = this.$iframe.get(0);
            if (iframe && iframe.contentWindow) {
                // this.send(iframe.contentWindow, event, data);
            }
        }
    }),

    // based on Workflows' CW.Dialog
   Dialog: Canvas.StatefulView.extend({
        tagName: 'section',
        defaults: {
            id: '',
            type: '',
            title: '',
            content: '',
            className: '',
            hint: '',
            actions: [],
            size: '',
            width: 600,
            height: 225,
            customData: null,
            closeable: true,
            absolute: false,
            path: '',
            query: {},
            open: false,
            remove: false,
            onLoad: function () { },
            onClose: function () { },
            onOpen: function () { }
        },
        events: {
            'click .aui-dialog2-footer-actions .aui-button': 'clickAction'
        },

        getStateModel: function () {
            return new Backbone.Model(_.pick(this.options, ['open']));
        },

        afterInit: function () {
            _.bindAll(this, 'show', 'hide', 'onShow', 'onHide', 'initializeDialog');
            this.$el.attr('class', 'aui-layer aui-dialog2 cml-modal-dialog ' + this.options.className);
            this.initializeDialog();
        },

        initializeDialog: function () {
            if (this.$el.parents().length === 0) {
                AJS.$('body').append(this.render());
            }
            this.dialog = AJS.dialog2(this.el);

            this.dialog.on('show', this.onShow);
            this.dialog.on('hide', this.onHide);

            if (this.options.open) {
                this.show();
            }
        },

        show: function (query) {
            if (query) {
                this.options.query = _.extend({}, this.options.query, query);
            }
            this.$el.addClass('cml-modal-dialog-opening');
            this.dialog.show();
            this.resize();
            return this;
        },

        onShow: function (e) {
            var _self = this;
            this.setState('open', true);
            _.defer(function () {
                _self.$el.removeClass('cml-modal-dialog-opening');
                _self.$el.addClass('cml-modal-dialog-open');
                _self.center();
                _self.options.onOpen(e);
            });
        },

        hide: function () {
            this.dialog.hide();
            return this;
        },

        onHide: function (e) {
            if (e.type !== '_aui-internal-layer-hide') {
                this.send('dialogClosed');
                this.options.onClose(e);
                if (this.options.remove) {
                    this.remove();
                } else {
                    this.$el.removeClass('cml-modal-dialog-closing');
                    this.$el.removeClass('cml-modal-dialog-open');
                }
            }
        },

        toggle: function () {
            if (this.getState('open')) {
                this.hide();
            } else {
                this.show();
            }
        },

        resize: function (height) {
            var windowHeight = $(window).height() - 150;
            var h = windowHeight;
            if (height) {
                h = Math.min(height, windowHeight);
            } else {
                h = Math.min(this.options.height, windowHeight);
            }

            if (this.getState('open') && h) {
                this.$el.css('height', h + 'px');
                this.center();
            }
        },

        center: function () {
            // vertically center the dialog
            var dialogHeight = this.$el.height();
            var windowHeight = $(window).height();
            var top = (windowHeight - dialogHeight) / 2;
            this.$el.css('top', top + 'px');
        },

        send: function (event, data) {
            if (this.iframe) {
                this.iframe.send(event, data);
            }
        },

        getActions: function () {
            return this.options.actions;
        },

        clickAction: function (event) {
            var $target = $(event.target);
            $.each(this.getActions(), function (i, action) {
                if (action.id === $target.attr('id') && action.onClick) {
                    action.onClick(event);
                    return false;
                }
            });
        },

        getContent: function () {
            return this.$el.find('.aui-dialog2-content');
        },

        setContent: function (content) {
            this.getContent().html(content);
            return this;
        },

        _renderHeader: function () {
            if (this.options.title) {
                this.$el.append('<header class="aui-dialog2-header cml-dialog-header">' +
                    '<h2 class="aui-dialog2-header-main">' + this.options.title + '</h2>' +
                    '</header>');
            }
        },

        _renderContent: function () {
            var $content = $('<div class="aui-dialog2-content cml-dialog-content" />');
            if (this.options.content) {
                $content.append(this.options.content);
                this.$el.append($content);
            } else if (this.options.path) {
                var iframe = new Canvas.CML.iframe({
                    id: this.options.id + '_content',
                    url: this.options.path,
                    absolute: this.options.absolute,
                    customData: this.options.customData,
                    query: this.options.query,
                    width: '100%',
                    height: '100%',
                    onLoad: this.options.onLoad
                });
                this.iframe = iframe;
                this.$el.append(iframe.render());
            }
        },

        _renderFooter: function () {
            // parse dialog buttons
            var actions = '';
            $.each(this.getActions(), function (i, action) {
                actions += '<button id="' + action.id + '" class="aui-button aui-button-' + action.style + '">' +
                    action.label + '</button>';
            });
            if (this.options.hint || actions) {
                this.$el.append('<footer class="aui-dialog2-footer cml-dialog-footer">' +
                    '<div class="aui-dialog2-footer-actions"> ' + actions + '</div>' +
                    '<div class="aui-dialog2-footer-hint">' + this.options.hint + '</div>' +
                    '</footer>');
            }
        },

        render: function () {
            var width = typeof this.options.width === 'string' ? this.options.width : this.options.width + 'px';
            var height = typeof this.options.height === 'string' ? this.options.height : this.options.height + 'px';
            this.$el.attr({
                'role': 'dialog',
                'aria-hidden': 'true',
                'data-aui-remove-on-hide': this.options.remove,
                'data-aui-modal': !this.options.closeable,
                'style': 'height: ' + height + ';',
            });

            // add specific class for warning dialogs
            if (this.options.type) {
                this.$el.addClass('aui-dialog2-' + this.options.type);
            }
            if (this.options.size) {
                this.$el.addClass('aui-dialog2-' + this.options.size);
            } else {
                this.$el.css('width', width);
            }

            // reset dialog's contents
            this.$el.html('');

            // append dialog header
            this._renderHeader();
            // append contents or render iframe
            this._renderContent();
            //append dialog footer
            this._renderFooter();

            return this.el;
        }
    }),

    closeDialogs: function (event) {
        AJS.dialog2('.cml-modal-dialog:visible').hide();
        if (event) {
            event.preventDefault();
        }
    }
};

(function ($, AJS) {
    var spaceKey = AJS.$('meta[name=space-key]').attr("content");
    if (spaceKey && spaceKey.length > 0) {
        Canvas.CML.spaceKey = spaceKey;
    }
    Canvas.CML.blueprint = Canvas.CML.getUrlParameters().blueprint;
})(AJS.$, AJS);