(function ($) { 

	// Replaces the first receiver with an input field configured according to the options. Returns the jQuery-wrapped input field.
	// options:
	//		type - field type (text, select)
	//		value - initial value for the field
	//		placeholder - placeholder text for the field
	//		options - used only for select field type, provides the options for the select (same format as select2)
	//		select2 - used only for select field types, provides the select2 parameters (e.g. { data: data, tags: true })
	//		initialize - callback method (see below)
	//		submit - callback method (see below)
	//		cancel - callback method (see below)
	//		validate - callback method (see below)
	//
	// initialize callback will have "this" be the text field
	//
	// submit callback will have "this" be the jQuery-wrapped original text element, and receive three parameters:
	//   event: submit event
	//   oldValue: value of text before edit
	//   newValue: value after edit
	// In the submit callback, you can call the following on "this":
	//   revert(): to restore the original text and original value
	//   apply():  to restore the original text with the edited value
	//
	// If cancel callback is provided, it will have the same "this" and parameters as above, EXCEPT event
	//
	// If validate callback is provided, it will have the same "this" and two parameters:
	//   inputField: the input field that is being displayed to the user
	//   newValue: value in the input field being validated
	// the validate callback should return false to reject the value (and also provide feedback to the user)
	$.fn.editInPlaceInputText = function(options) {
		options = options || {};

		var type = options.type || "text";
		
		var $input;
		if (type == "text") $input = $("<input>").attr("type", "text").addClass("text");
		else if (type == "select") $input = $("<input>").attr("type", "hidden");

		if (options.placeholder) $input.attr("placeholder", options.placeholder);
		
        this.first().each(function() {
			var $element = $(this);
			var value = typeof(options.value) != "undefined" ? options.value : $element.data("value") || $element.text();
			$input.data("original", value).val(value);

			var $acceptButton = ThemePress.auiIcon("success", "saver");
			var $cancelButton = ThemePress.auiIcon("close-dialog", "closer");
			var $form = $("<form>").addClass("aui edit-in-place").append($input).append($acceptButton).append($cancelButton);
			$element.after($form).hide();
			
			$element.revert = function (event) {
				$form.remove();
				$element.show();
				if (options.cancel) options.cancel.call($element, $input.data("original"), $input.val());
			};
			$element.apply = function () {
				var newValue = $.trim($input.val());
				$element.text(newValue);
				$form.remove();
				$element.show();
			};

			$acceptButton.click(function() { $form.submit(); });
			$cancelButton.click($element.revert);
			$input.keydown(function (event) { if (event.which == 27) $element.revert(); });
			$form.submit(function (event) { 
				var newValue = $.trim($input.val());
				if (options.validate && !options.validate.call($element, $input, newValue)) $input.focus();
				else if (options.submit) options.submit.call($element, event, $input.data("original"), newValue); 
				return false;
			});
			
			if (options.initialize) options.initialize.call($input);
			
			if (type == "select") {
				if (options.select2) $input.auiSelect2(options.select2);
				else $input.auiSelect2();
			}
			
			$input.focus();
        });

		return $input;
	};

	// options is a list of either strings (display and value will be the same) or {display: "Xxxx", value: "xxx"} to control the displayed values.
	//
	// submitCallback will have "this" be the jQuery-wrapped original text element, and receive three parameters:
	//   event: submit event
	//   oldValue: value of text before edit
	//   newValue: value after edit
	// In submitCallback, you can call the following on "this":
	//   revert(): to restore the original text and original value
	//   apply():  to restore the original text with the edited value
	//
	// If cancelCallback is provided, it will have the same "this" and parameters as above, EXCEPT event
	// 
	// >>>>>>>> WARNING: THIS IS UNFINISHED AND NOT USED BY ANYONE <<<<<<<<<<<<
	//
	$.fn.editInPlaceSelectField = function(options, submitCallback, cancelCallback) {
        return this.each(function() {
			var element = $(this);
			var originalVal = element.text();
			var $select = $("<select>").addClass("select").data("original", originalVal);
			$.each(options, function (i, value) {
				var $option = $("<option>");
				if (typeof(value) == "string") $option.text(value);
				else $option.attr("value", value.value).text(value.display);
				$select.append($option);
			});
			var $cancelButton = ThemePress.auiIcon("close-dialog", "closer");
			var $form = $("<form>").addClass("aui").append($select).append($cancelButton);
			element.replaceWith($form);
			
			element.revert = function () {
				$form.replaceWith(element);
				if (cancelCallback) cancelCallback.call(element, $select.data("original"), $select.val());
			};
			element.apply = function () {
				element.text($select.val());
				$form.replaceWith(element);
			};

			$cancelButton.click(element.revert);
			$select.keydown(function (event) { if (event.which == 27) element.revert(); });
			$form.submit(function (event) { submitCallback.call(element, event, $select.data("original"), $select.val()); return false; });
			
			$select.focus();
        });
	};

})(jQuery);
