/**
 * jQuery plugin to allow controlled and formatted input for time. Works on any <input /> element.
 * 
 * Accepts input of virtually any form (3, 330, 03:30a 3:30 AM are equivalent, as are 3p 1530 330p, 03:30 PM)
 * and formats the date in hh:mm PM (12-hour clock) format. Formats in 24-hour clock in target element, if specified.
 * Uses timezone of current user.
 * 
 * Usage: 
 * 
 * 	$("input").timefield();
 * 	$("input").timefield({option1: value1, . . .});
 *
 * Options:
 *
 * 	target: selector for input field to put 
 * 
 * See http://docs.jquery.com/Plugins/Authoring for an explanation of the structure of this plugin.
 */
(function( $ ) {

	// Defaults
	var settings = {
		target: false
    };

	var pad = function (n) { return (n < 10 ? "0" : "") + n.toString(); }

	var methods = {
		
		version: function () {
			return "1.0";
		},

		init : function(options) {

			return this.each(function() {
				var $this = $(this);
				var data = $this.data('timefield');

				// If the plugin hasn't been initialized yet
				if (!data) {

					$(this).data('timefield', {
						settings : $.extend({}, settings, options)
					});

					$this.bind("change.timefield", methods.adjustFormat);
				}
         
			});
		},

		destroy : function(options) {
			return this.each(function() {
				var $this = $(this);
				var data = $this.data('timefield');
				$this.unbind('.timefield');
				data.timefield.remove();
				$this.removeData('timefield');
			});
		},

		adjustFormat : function(event) {
			var timeExp = /([012][0-9]|0?\d):?(\d{0,2})\s*([ap]?)m?/i;

			var $this = $(this);
			var after = $.trim($this.val());
			var afterTime = after.match(timeExp) || ["", "0", "0", ""];
			var h = parseInt(pad(afterTime[1]));
			var m = parseInt(pad(afterTime[2]));
			var p = afterTime[3].toLowerCase();
			var isPM = h > 11 && p != "a" || p == "p";
			if (h == 12 && !isPM) h = 0;
			else if (h < 12 && isPM) h += 12;
			var date = new Date(2012, 9, 15, h, m);
			methods.setTime.call($this, date);
		},
		
		setTime: function (date) {
			var h12 = function (d) { 
				if (d.getHours() == 0) return 12;
				return d.getHours() > 12 ? d.getHours() - 12 : d.getHours();
			}
			return this.each(function() {
				var $this = $(this);
				if (typeof(date) == "string") {
					$this.val(date);
					methods.adjustFormat.call(this);
				}
				else {
					$this.val(h12(date) + ":" + pad(date.getMinutes()) + (date.getHours() > 11 ? " PM" : " AM"));
					$(methods.setting.call($this, "target")).val(pad(date.getHours()) + pad(date.getMinutes()));
				}
			});
		},

		setting: function (value) {
			return this.data('timefield').settings[value];
		},

		update : function(content) {
			return this.each(function() {
				console.log("Whatever update means");
			});
		}
	};

 	$.fn.timefield = function(method) {
  
		// jQuery plugin standard for calling methods on the object (see http://docs.jquery.com/Plugins/Authoring#Namespacing)
		if (methods[method]) return methods[ method ].apply(this, Array.prototype.slice.call(arguments, 1));
		else if (typeof method === "object" || !method) return methods["init"].apply(this, arguments);
		else $.error("Method " +  method + " does not exist on jQuery.timefield");

	};
})( jQuery );
