/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.migration.agent.service.guardrails.usage;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.confluence.setup.BootstrapManager;
import com.atlassian.core.util.FileUtils;
import com.atlassian.migration.agent.dto.InstanceMetadataDto;
import com.atlassian.migration.agent.json.Jsons;
import com.atlassian.migration.agent.service.guardrails.logs.PageType;
import com.atlassian.migration.agent.service.guardrails.logs.UsageMetricsNodeData;
import com.atlassian.migration.agent.service.guardrails.usage.DailyUsageDetails;
import com.atlassian.migration.agent.service.guardrails.usage.DailyUsageSummary;
import com.atlassian.migration.agent.service.guardrails.util.CsvBuilder;
import com.atlassian.migration.agent.service.guardrails.util.SerializationUtil;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import jakarta.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DailyUsageMetricsStore {
    public static final int DAYS_PRODUCT_ANALYTICS_COLLECTION = 14;
    private static final Logger log = LoggerFactory.getLogger(DailyUsageMetricsStore.class);
    private static final Pattern REGEX_YYYYMMDD = Pattern.compile("\\d{8}");
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter CSV_DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final String USAGE_METRICS_FOLDER = "usage-metrics";
    private static final String USAGE_FILE_PREFIX = "confluence-usage-metrics-";
    private static final String COMBINED_USAGE_FILE_NAME = "combined-usage-metrics";
    private final BootstrapManager bootstrapManager;

    public DailyUsageMetricsStore(BootstrapManager bootstrapManager) {
        this.bootstrapManager = bootstrapManager;
    }

    public boolean hasData(String nodeId, LocalDate date) {
        Path dateDir = this.dateDir(date);
        return this.nodeFile(dateDir, nodeId).exists() || this.combinedFile(dateDir).exists();
    }

    public void storePartial(DailyUsageDetails metrics) throws IOException {
        File file = this.nodeFile(this.dateDir(metrics.getDate()), ((UsageMetricsNodeData)Iterables.getOnlyElement(metrics.getNodes())).getId());
        SerializationUtil.saveJson(file, metrics);
    }

    public void combine(Instant now, List<String> requiredNodes) {
        LocalDate today = now.atZone(ZoneId.systemDefault()).toLocalDate();
        this.getDirectories(today).forEach((date, dir) -> {
            try {
                this.combineDateIfNeeded(now, (LocalDate)date, (File)dir, requiredNodes);
            }
            catch (Exception e) {
                log.info("Failed to compute daily usage metrics for {}", (Object)dir.getName(), (Object)e);
            }
        });
    }

    public int getProgress(LocalDate today) {
        return (int)(100L * this.streamSummaryFiles(today).count() / 14L);
    }

    public void cleanup() {
        this.getDirectories(LocalDate.now());
    }

    private Map<LocalDate, File> getDirectories(LocalDate today) {
        LocalDate min = today.minusDays(14L);
        File[] dirs = this.getMetricsDir().toFile().listFiles();
        if (dirs == null) {
            return Collections.emptyMap();
        }
        ArrayList<File> toDelete = new ArrayList<File>();
        HashMap<LocalDate, File> result = new HashMap<LocalDate, File>();
        for (File dir : dirs) {
            String name = dir.getName();
            if (!REGEX_YYYYMMDD.matcher(name).matches()) continue;
            LocalDate date = LocalDate.from(DATE_FORMATTER.parse(name));
            if (date.isBefore(min)) {
                toDelete.add(dir);
                continue;
            }
            if (!date.isBefore(today)) continue;
            result.put(date, dir);
        }
        toDelete.forEach(FileUtils::recursiveDelete);
        return result;
    }

    private File nodeFile(Path dateDirectory, String nodeId) {
        return dateDirectory.resolve(USAGE_FILE_PREFIX + nodeId).toFile();
    }

    private File combinedFile(Path dateDirectory) {
        return dateDirectory.resolve(COMBINED_USAGE_FILE_NAME).toFile();
    }

    private Path dateDir(LocalDate localDate) {
        return this.getMetricsDir().resolve(DATE_FORMATTER.format(localDate));
    }

    private void combineDateIfNeeded(Instant now, LocalDate today, File dir, List<String> requiredNodes) throws IOException {
        File combined = this.combinedFile(dir.toPath());
        if (combined.exists()) {
            return;
        }
        File[] files = dir.listFiles(file -> file.getName().startsWith(USAGE_FILE_PREFIX));
        if (files == null) {
            return;
        }
        List<File> filteredFiles = Stream.of(files).filter(file -> !SerializationUtil.isTemporaryFileName(file.getName())).collect(Collectors.toList());
        if (!this.isCompleted(now, filteredFiles, requiredNodes)) {
            return;
        }
        DailyUsageDetails.DailyUsageDetailsBuilder builder = new DailyUsageDetails.DailyUsageDetailsBuilder().date(today);
        for (File file2 : filteredFiles) {
            builder.add(SerializationUtil.deserializeJson(file2, DailyUsageDetails.class));
        }
        builder.addNodesIfMissing(requiredNodes);
        SerializationUtil.saveJson(combined, builder.build().toDailyUsageSummary());
        for (File file3 : files) {
            Files.delete(file3.toPath());
        }
    }

    protected boolean isCompleted(Instant now, List<File> files, List<String> requiredNodes) {
        Optional<Instant> maxLastModified = files.stream().map(File::lastModified).max(Comparator.naturalOrder()).map(Instant::ofEpochMilli);
        return maxLastModified.isPresent() && (files.size() >= requiredNodes.size() || now.isAfter(maxLastModified.get().plus(Duration.ofDays(1L))));
    }

    private Path getMetricsDir() {
        return this.bootstrapManager.getSharedHome().toPath().resolve(USAGE_METRICS_FOLDER);
    }

    @javax.annotation.Nullable
    @Nullable
    public Path createDailyUsageMetricsCsv(Path csvFilePath, InstanceMetadataDto instanceMetadataDto, List<DailyUsageSummary> usageMetrics) throws IOException {
        if (usageMetrics.isEmpty()) {
            return null;
        }
        return new CsvBuilder<DailyUsageSummary>().addColumn("date", usage -> CSV_DATE_FORMATTER.format(usage.getDate())).addColumn("active_users", DailyUsageSummary::getUniqueUsers).addColumn("peak_hour_users", DailyUsageSummary::getPeakHourUsers).addColumn("peak_hour_requests", DailyUsageSummary::getPeakHourRequests).addColumn("nodes", summary -> Jsons.valueAsString(summary.getNodes())).addColumn("server_id", instanceMetadataDto.getServerId()).addColumn("assessment_starting_date", instanceMetadataDto.getAssessmentDate()).build(csvFilePath, usageMetrics);
    }

    @javax.annotation.Nullable
    @Nullable
    public Path createPageTrafficDistributionCsv(Path csvFilePath, InstanceMetadataDto instanceMetadataDto, List<DailyUsageSummary> usageMetrics) throws IOException {
        if (usageMetrics.isEmpty()) {
            return null;
        }
        EnumMap counts = new EnumMap(PageType.class);
        for (DailyUsageSummary metrics : usageMetrics) {
            metrics.getRequestsTypeCount().forEach((type, count) -> counts.merge((PageType)((Object)type), count, Integer::sum));
        }
        counts.remove((Object)PageType.UNKNOWN);
        counts.remove((Object)PageType.REST);
        long sum = counts.values().stream().mapToLong(Integer::longValue).sum();
        List data = counts.entrySet().stream().sorted(Map.Entry.comparingByValue().reversed()).collect(Collectors.toList());
        return new CsvBuilder<Map.Entry>().addColumn("action", entry -> this.formatPageTypeName((PageType)((Object)((Object)entry.getKey())))).addColumn("traffic", entry -> String.format("%.2f%%", (double)((long)((Integer)entry.getValue() * 10000) / sum) / 100.0)).addColumn("id", Map.Entry::getKey).addColumn("server_id", instanceMetadataDto.getServerId()).addColumn("assessment_starting_date", instanceMetadataDto.getAssessmentDate()).build(csvFilePath, data);
    }

    private String formatPageTypeName(PageType pageType) {
        String name = pageType.name();
        name = name.toLowerCase().replace("_", " ");
        String[] words = name.split("\\s");
        StringBuilder sb = new StringBuilder();
        for (String word : words) {
            sb.append(Character.toUpperCase(word.charAt(0))).append(word.substring(1)).append(" ");
        }
        return sb.toString().trim();
    }

    public List<DailyUsageSummary> listSummaries(LocalDate today) {
        return this.listSummariesInternal(today);
    }

    @VisibleForTesting
    protected List<DailyUsageSummary> listSummariesInternal(LocalDate today) {
        return this.streamSummaryFiles(today).flatMap(file -> {
            try {
                return Stream.of(SerializationUtil.deserializeJson(file, DailyUsageSummary.class));
            }
            catch (Exception e) {
                log.info("Error reading file {}", file);
                return Stream.empty();
            }
        }).sorted(Comparator.comparing(DailyUsageSummary::getDate)).collect(Collectors.toList());
    }

    public List<DailyUsageSummary> listExampleSummaries(LocalDate today) {
        ArrayList<DailyUsageSummary> summaries = new ArrayList<DailyUsageSummary>();
        for (int i = 1; i <= 14; ++i) {
            int val = i;
            summaries.add(new DailyUsageSummary(today.minusDays(i), (List<UsageMetricsNodeData>)ImmutableList.of((Object)UsageMetricsNodeData.builder().id("local").dataCollectionStatus(UsageMetricsNodeData.DataCollectionStatus.COMPLETE).nodeStatus(UsageMetricsNodeData.NodeStatus.AVAILABLE).build()), 11456 + i * 10, 5200 - i * 10, 250000 + i * 1234, Stream.of(PageType.values()).collect(Collectors.toMap(Function.identity(), type -> Math.abs(type.toString().hashCode() / 2) % 1000 + val))));
        }
        return summaries;
    }

    private Stream<File> streamSummaryFiles(LocalDate today) {
        return this.getDirectories(today).values().stream().map(dateDir -> this.combinedFile(dateDir.toPath())).filter(File::exists);
    }
}

