/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.migration.agent.service.impl;

import com.atlassian.migration.agent.dto.ConfigDto;
import com.atlassian.migration.agent.dto.util.MissingConfigurationException;
import com.atlassian.migration.agent.entity.Config;
import com.atlassian.migration.agent.logging.LoggingContextAwareExecutorService;
import com.atlassian.migration.agent.store.ConfigStore;
import com.atlassian.migration.agent.store.tx.PluginTransactionTemplate;
import com.atlassian.util.concurrent.Lazy;
import com.atlassian.util.concurrent.Supplier;
import io.atlassian.util.concurrent.ThreadFactories;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.PreDestroy;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);
    private static final long LOWER_CACHE_TIME = Duration.of(30L, ChronoUnit.SECONDS).toMillis();
    private static final long UPPER_CACHE_TIME = Duration.of(60L, ChronoUnit.SECONDS).toMillis();
    private final ConfigStore configStore;
    private final PluginTransactionTemplate ptx;
    private final ExecutorService executorService;
    private final AtomicBoolean isRefreshing;
    private volatile long lastCacheCheck;
    private volatile Config cachedConfig;
    private final Supplier<Void> initialiseConfigSupplier;
    private final Supplier<Void> refreshCacheFirstTime;

    public ConfigurationService(ConfigStore configStore, PluginTransactionTemplate ptx) {
        this.configStore = configStore;
        this.ptx = ptx;
        this.executorService = new LoggingContextAwareExecutorService(Executors.newSingleThreadExecutor(ThreadFactories.namedThreadFactory((String)this.getClass().getName())));
        this.initialiseConfigSupplier = Lazy.supplier(() -> {
            this.initialiseConfig();
            return null;
        });
        this.refreshCacheFirstTime = Lazy.supplier(() -> {
            this.refreshCache();
            return null;
        });
        this.isRefreshing = new AtomicBoolean(false);
        this.lastCacheCheck = System.currentTimeMillis();
    }

    @PreDestroy
    private void cleanup() {
        this.executorService.shutdown();
    }

    private void initialiseConfig() {
        try {
            this.configStore.getConfig();
        }
        catch (MissingConfigurationException e) {
            log.info("No configuration presently found, creating empty config");
            this.ptx.write(this.configStore::createEmptyConfig);
        }
    }

    public void refreshCacheAsync() {
        if (this.isRefreshing.compareAndSet(false, true)) {
            this.executorService.execute(this::refreshCache);
        }
    }

    public void refreshCache() {
        try {
            this.initialiseConfigSupplier.get();
            this.cachedConfig = this.configStore.getConfig();
        }
        catch (Exception e) {
            log.error("Unable to fetch fresh configuration");
        }
        finally {
            this.lastCacheCheck = System.currentTimeMillis();
            this.isRefreshing.set(false);
        }
    }

    public ConfigDto getCachedConfig() {
        long cacheAge = this.getCacheAge();
        if (this.cachedConfig == null) {
            this.refreshCacheFirstTime.get();
        } else if (cacheAge > UPPER_CACHE_TIME) {
            this.cacheUpperLimitExceeded();
        } else if (cacheAge > LOWER_CACHE_TIME) {
            this.refreshCacheAsync();
        }
        return this.toDto(this.cachedConfig);
    }

    private synchronized void cacheUpperLimitExceeded() {
        if (this.getCacheAge() > UPPER_CACHE_TIME && this.isRefreshing.compareAndSet(false, true)) {
            this.refreshCache();
        }
    }

    private long getCacheAge() {
        return System.currentTimeMillis() - this.lastCacheCheck;
    }

    public ConfigDto getConfig() {
        this.refreshCacheFirstTime.get();
        return this.toDto(this.configStore.getConfig());
    }

    public void upsertConfig(ConfigDto configDto) {
        Config config = new Config(configDto);
        this.ptx.write(() -> {
            this.configStore.updateConfig(config);
            this.refreshCache();
        });
    }

    private ConfigDto toDto(Config config) {
        return new ConfigDto(config);
    }
}

