/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.migration.agent.service.impl;

import com.atlassian.migration.agent.dto.ConfigDto;
import com.atlassian.migration.agent.dto.util.MissingConfigurationException;
import com.atlassian.migration.agent.entity.Config;
import com.atlassian.migration.agent.logging.LoggingContextAwareExecutorService;
import com.atlassian.migration.agent.store.ConfigStore;
import com.atlassian.migration.agent.store.tx.PluginTransactionTemplate;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);
    private static final long LOWER_CACHE_TIME = Duration.of(30L, ChronoUnit.SECONDS).toMillis();
    private static final long UPPER_CACHE_TIME = Duration.of(60L, ChronoUnit.SECONDS).toMillis();
    private final ConfigStore configStore;
    private final PluginTransactionTemplate ptx;
    private final AtomicBoolean isRefreshing;
    private final ExecutorService executorService;
    private volatile long lastCacheCheck;
    private volatile Config cachedConfig;

    public ConfigurationService(ConfigStore configStore, PluginTransactionTemplate ptx) {
        this.configStore = configStore;
        this.ptx = ptx;
        this.executorService = new LoggingContextAwareExecutorService(Executors.newSingleThreadExecutor());
        this.initialiseConfig();
        this.isRefreshing = new AtomicBoolean(false);
        this.lastCacheCheck = System.currentTimeMillis();
    }

    @PostConstruct
    public void setup() {
        this.refreshCache();
    }

    @PreDestroy
    private void cleanup() {
        this.executorService.shutdown();
    }

    private void initialiseConfig() {
        try {
            this.configStore.getConfig();
        }
        catch (MissingConfigurationException e) {
            log.debug("No configuration presently found, creating empty config");
            this.ptx.write(this.configStore::createEmptyConfig);
        }
    }

    public void refreshCacheAsync() {
        if (this.isRefreshing.compareAndSet(false, true)) {
            this.executorService.execute(this::refreshCache);
        }
    }

    public void refreshCache() {
        try {
            this.cachedConfig = this.configStore.getConfig();
        }
        catch (Exception e) {
            log.error("Unable to fetch fresh configuration");
        }
        finally {
            this.lastCacheCheck = System.currentTimeMillis();
            this.isRefreshing.set(false);
        }
    }

    public ConfigDto getCachedConfig() {
        long cacheAge = this.getCacheAge();
        if (cacheAge > UPPER_CACHE_TIME) {
            this.cacheUpperLimitExceeded();
        } else if (cacheAge > LOWER_CACHE_TIME) {
            this.refreshCacheAsync();
        }
        return this.toDto(this.cachedConfig);
    }

    private synchronized void cacheUpperLimitExceeded() {
        if (this.getCacheAge() > UPPER_CACHE_TIME && this.isRefreshing.compareAndSet(false, true)) {
            this.refreshCache();
        }
    }

    private long getCacheAge() {
        return System.currentTimeMillis() - this.lastCacheCheck;
    }

    public ConfigDto getConfig() {
        return this.toDto(this.configStore.getConfig());
    }

    public void upsertConfig(ConfigDto configDto) {
        Config config = new Config(configDto);
        this.ptx.write(() -> {
            this.configStore.updateConfig(config);
            this.refreshCache();
        });
    }

    private ConfigDto toDto(Config config) {
        return new ConfigDto(config);
    }
}

