define('confluence-copy-space-plugin/options', [
    'jquery',
    'ajs',
    'underscore',
], function ($,
             AJS,
             _) {
    'use strict';

    function onFormSubmit(e) {
        e.preventDefault();

        const formData = {
            'oldKey': document.getElementById('oldKey').value,
            'newName': document.getElementById('newName').value,
            'newKey': document.getElementById('newKey').value,
            'copyPages': document.getElementById('copyPages').checked,
            'copyBlogPosts': document.getElementById('copyBlogPosts').checked,
            'copyAttachments': document.getElementById('copyAttachments').checked,
            'copyComments': document.getElementById('copyComments').checked,
            'copyLabels': document.getElementById('copyLabels').checked,
            'keepMetaData': document.getElementById('keepMetaData').checked
        };

        fetch(AJS.contextPath() + '/rest/copy/1.0/copy', {
            body: JSON.stringify(formData),
            cache: 'no-store',
            method: 'POST',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        }).then(function (response) {
            return response.json();
        }).then(function (result) {
            if (result['statusCode'] && result['statusCode'] !== 200) {
                let isConfluenceInReadOnlyMode = result['statusCode'] === 405 && result['reason'] === "READ_ONLY";
                let message = isConfluenceInReadOnlyMode ? AJS.I18n.getText("copyspace.validation.read.only.mode") : result['message'];
                showPluginMessage(message);
                return;
            }

            longRunningTask(result['taskId'], formData['newKey']);
        }).catch(function (error) {
            hideLongRunningTaskDiv();
            showPluginMessage(error);
        });
    }

    function longRunningTask(taskId, newKey) {
        let REGULAR_PROGRESS_POLL_INTERVAL = 1000;
        let QUICK_PROGRESS_POLL_INTERVAL = 172;
        let timeout;
        let timesPolled = 0;

        showLongRunningTaskDiv();

        let process = function () {
            fetch(AJS.contextPath() + '/rest/api/longtask/' + taskId).then(function (response) {
                return response.json();
            }).then(function (data) {
                let message = getMessages(data);
                document.getElementById("taskCurrentStatus").innerText = message;

                if (!data.successful) {
                    hideLongRunningTaskDiv();
                    showPluginMessage(AJS.I18n.getText("copyspace.result.message.failure"));
                    clearTimeout(timeout);
                    return;
                }

                let percentage = +data.percentageComplete;
                if (percentage > 0) {
                    document.getElementById("taskCompletedBar").style.width = percentage + "%";
                    document.getElementById("taskRemainedBar").style.width = (100 - percentage) + "%";
                }

                if (percentage >= 100) {
                    window.location.href = AJS.contextPath() + '/display/' + newKey
                    clearTimeout(timeout);
                    return;
                }
                updateStatusPoller();
            }).catch(function (error) {
                hideLongRunningTaskDiv();
                showPluginMessage(error);
                clearTimeout(timeout);
            });
        };
        let updateStatusPoller = function () {
            timeout = setTimeout(process,
                timesPolled++ < 5 ? QUICK_PROGRESS_POLL_INTERVAL : REGULAR_PROGRESS_POLL_INTERVAL);
        };
        process();
    }

    function showLongRunningTaskDiv() {
        document.getElementById('spaceCopyingLongRunningTask').style.display = 'block';
        document.getElementById('copySpaceForm').style.display = 'none';
    }

    function hideLongRunningTaskDiv() {
        document.getElementById('spaceCopyingLongRunningTask').style.display = 'none';
        document.getElementById('copySpaceForm').style.display = 'block';
    }

    function getMessages(data) {
        let message = AJS.I18n.getText("copyspace.progress.message.started");
        if (data.messages && data.messages.length) {
            let messageText = data.messages[data.messages.length - 1].translation;
            try {
                //messages from copy space plugin come as single string and messages from copy pages come as JSON object with message key
                let messageJson = JSON.parse(messageText);
                let statusKey = messageJson.status.key;
                if (statusKey === 'STATUS_COPYING_PAGES' || statusKey === 'STATUS_COPIED_PAGES' || statusKey === 'STATUS_RELINK_PAGES') {
                    message = AJS.I18n.getText("copyspace.progress.message.pages.in.progress")
                }
            } catch (e) {
                message = messageText;
            }
        }
        return message;
    }

    function onPluginMessageClose() {
        $("#cp-alert-message").css("display", "none")
        $("#cp-alert-message-bg").css("display", "none")
    }

    function onCancelClick() {
        const oldKey = document.getElementById('oldKey').value;
        window.location.href = AJS.contextPath() + '/display/' + oldKey;
    }

    function showPluginMessage(message) {
        $("div .aui-dialog2-content").text(message)
        $("#cp-alert-message").css("display", "flex")
        $("#cp-alert-message-bg").css("display", "flex")
    }

    let keyValidationRequest;

    function validateSpaceKey() {
        const keyField = $('#newKey');
        const deferred = $.Deferred();
        const value = $.trim(keyField.val());

        if (!value) {
            deferred.reject(AJS.I18n.getText('copyspace.validation.space.key.empty'));
            return;
        }

        if (keyValidationRequest) {
            keyValidationRequest.abort();
        }

        keyValidationRequest = $.getJSON(AJS.contextPath() + '/ajax/spaceavailable.action', { key: value }).done(function(data) {
            if (!data.available && data.message) {
                deferred.reject(data.message);
            } else {
                deferred.resolve();
            }
            keyValidationRequest = false;
        });

        const promise = deferred.promise();
        const errorContainer = $('#keyError');
        promise.done(function() {
            errorContainer.text('').addClass('hidden');
        });
        promise.fail(function(message) {
            errorContainer.text(message).removeClass('hidden');
        });
    }

    let debouncedValidation = _.debounce(validateSpaceKey, 500);

    function generateSpaceKeyBasedOnSpaceName() {
        const newNameValue = document.getElementById('newName').value;
        //KeyGenerator is added in default confluence file 'jquery.keygen.js'
        document.getElementById('newKey').value = window.KeyGenerator.call(this).generateKey(newNameValue);
        debouncedValidation();
    }

    return {
        setup: function () {
            AJS.trigger('analyticsEvent', { name: 'confluence.confluence-copyspace.tab.open',
                data: {
                    originalSpaceKey: document.getElementById('oldKey').value,
                    originalSpaceId: document.getElementById('oldSpaceId').value
            }});
            document.getElementById('copy-space-form-close-button').onclick = onPluginMessageClose;
            document.getElementById('cancelCopySpace').onclick = onCancelClick;
            document.getElementById('copySpaceForm').onsubmit = onFormSubmit;
            document.getElementById('newName').onkeyup = generateSpaceKeyBasedOnSpaceName;
            document.getElementById('newKey').onkeyup = debouncedValidation;
        }
    }

});

