/*
 * Copyright (c) 2019 Appfire Technologies, Inc.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Bob Swift Atlassian Add-ons EULA"
 * (https://bobswift.atlassian.net/wiki/x/WoDXBQ) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */

/**
 * Provides custom behaviour in the macro browser for the Graphviz macro.
 */
(function($) {

    var MACROS_LIST = ['graphviz', 'flowchart', 'digraph', 'graph'];
    var MacroConfig = function() {};

    MacroConfig.prototype.fields = {
        "enum" : {
            "template" : function(param, options) {  // parameter with name: template
                var input = '';
                var paramDiv = AJS.$(Confluence.Templates.MacroBrowser.macroParameterSelect());
                var templatesDropDown = AJS.$("select", paramDiv);
                loadTemplates(templatesDropDown,param);
                return new AJS.MacroBrowser.Field(paramDiv, templatesDropDown, options);
            }
        },
        "string" : {
            "profile" : function(param, options) { // parameter with name: profile

                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterSelect());
                var input = $("select", paramDiv);

                // we need to do some trickery when the value is first set, and when the value is subsequently changed, so set this up.
                if (options == null) {
                    options = {};
                }
                options.setValue = function(value) {
                    var targetOption = input.find("option[value='" + value + "']");
                    if (targetOption.length == 0) {
                        var option = $("<option/>");
                        option.val(value);
                        option.text(value);
                        input.append(option);
                    }
                };

                profilesBindAsyncDropDown(input);
                return new AJS.MacroBrowser.Field(paramDiv, input, options);
            }
        }
    };
    
    function loadTemplates(templatesDropDown,param) {
        getTemplatesAsync(param,
                function(templates) {
                    if(isMacroBodyExists()){
                        AJS.$("#macro-param-div-template").hide();
                    } else {
                        if (!templates.length) {
                            AJS.log("Appfire: The configured template list is empty.");
                            return;
                        }
                        templatesDropDown.empty();
                        templatesDropDown.append("<option></option>");
                        // add templates to select list
                        $.each(templates, function(index, template) {
                                templatesDropDown.append("<option value=\"" + template.templateName + "\" data-template-desc=\"" + template.description + "\">"+template.templateName +"</option>");
                        });
                    }
                },
                function(xhr, textStatus, errorThrown) {
                    AJS.log("Failed to retrieve templates from Confluence " + textStatus + " - " + errorThrown);
                }
            ); 
    }
    
    /**
     * This triggers when ever we change the Template drop down options to append its description.
     */
    $(document).on("change", "#macro-param-template", function(e) {
        AJS.$('#macro-param-div-template').find(".template-description").remove();
        var templateDesc = AJS.$(this).find('option[value="'+$(this).val()+'"]').data('templateDesc')
        if(templateDesc == undefined) {
            templateDesc = '';
        }
        AJS.$('#macro-param-div-template').append('<div class="macro-param-desc template-description">'+templateDesc+'</div>');
    });
        
    AJS.MacroBrowser.Macros["graphviz"] = new MacroConfig();
    AJS.MacroBrowser.Macros["flowchart"] = new MacroConfig();
    AJS.MacroBrowser.Macros["graph"] = new MacroConfig();
    AJS.MacroBrowser.Macros["digraph"] = new MacroConfig();
    AJS.MacroBrowser.Macros["plantuml"] = new MacroConfig();

 /**
     * Populates Select element with the list of configured profiles.
     * @param dropDown JQuery selector for the select element to be populated.
     */
    function profilesBindAsyncDropDown(dropDown) {
        // Load the profiles from Active object
        getProfilesAsync(function(profiles) {
            if (!profiles.length) {
                AJS.log("Configured Graphviz profiles were not in the expected format.");
                return;
            }
            var currentValue = dropDown.val();
            dropDown.empty();
            dropDown.append($("<option/>").attr("value", ""));

            $.each(profiles, function(index, profile) {
                var option = $("<option />");
                option.val(profile);
                option.text(profile);
                dropDown.append(option);
            });
            // restore the currently selected value.
            dropDown.val(currentValue);
        }, function(xhr, textStatus, errorThrown) {
            AJS.log("Failed to retrieve profiles from Confluence " + textStatus + " - "
                    + errorThrown);
        });
    }

    /**
     * Asynchronously retrieves the current set of profiles.
     *
     * @param successHandler Callback to invoke if the retrieval is successful
     * @param errorHandler Callback to invoke if the retrieval fails.
     */
    function getProfilesAsync(successHandler, errorHandler) {
        AJS.$.ajax({
            async : true,
            url : AJS.params.contextPath+ "/rest/bobswift/graphviz/1.0/profile/macroprofiles/",
            dataType : "json",
            timeout : 10000, // 10 seconds,
            error : function(xhr, textStatus, errorThrown) {
                if (errorHandler && typeof (errorHandler) == "function") {
                    errorHandler(xhr, textStatus, errorThrown);
                }
            },
            success : function(data) {
                if (successHandler && typeof (successHandler) == "function") {
                    successHandler(data);
                }
            }
        });
    }
    /**
     * Asynchronously retrieves the current set of templates.
     *
     * @param successHandler Callback to invoke if the retrieval is successful
     * @param errorHandler Callback to invoke if the retrieval fails.
     */
    function getTemplatesAsync(param, successHandler, errorHandler) {
        AJS.$.ajax({
            async: true,
            url: AJS.params.contextPath + "/rest/bobswift/graphviz/latest/templates?macroName="+param.macroName,
            dataType: "json",
            timeout: 10000, // 10 seconds,
            error: function(xhr, textStatus, errorThrown) {
                if (errorHandler && typeof(errorHandler) == "function") {
                    errorHandler(xhr, textStatus, errorThrown);
                }
            },
            success: function(data) {
                if (successHandler && typeof(successHandler) == "function") {
                    successHandler(data);
                }
            }
        });
    };

    
    /**
     * On Save of macro editor getting the syntax of the selected template and copy it to the macro body.
     * We are overriding the native method here to replace the macrobody when there is no macrobody and a template is chosen.
     * We need to be careful if there is any change in future in the native method tinymce.confluence.macrobrowser.macroBrowserComplete 
     */
    tinymce.confluence.macrobrowser.macroBrowserComplete = function (macro) {
        var t = tinymce.confluence.macrobrowser;
        var macroRenderRequest = {
                contentId: AJS.Meta.get("content-id") || "0",
                macro: {
                    name: macro.name,
                    params: macro.params,
                    defaultParameterValue: macro.defaultParameterValue,
                    body: replaceMacroBody(macro) //Except this piece, everything else is native tinymce.confluence.macrobrowser.macroBrowserComplete method in tinymce-macrobrowser.js
                }
            };
        AJS.Rte.BookmarkManager.restoreBookmark();
        if (t.editedMacroDiv) {
            tinymce.confluence.MacroUtils.insertMacro(macroRenderRequest, t.editedMacroDiv);
            delete t.editedMacroDiv;
        } else {
            tinymce.confluence.MacroUtils.insertMacro(macroRenderRequest);
        }
    };
    
    /**
     * Replace macro body only if template param is disabled and it is one of the graphviz macros
     */
   function replaceMacroBody(macro){
      var selectedTemplate = $("#macro-param-template").val();
      if (selectedTemplate) {
          /*We will be getting the wiki syntax of the selected template  only  when there is no macro body
           previously and it should be our Graphviz macro list mentioned.*/  
          if(!isMacroBodyExists() && MACROS_LIST.indexOf(macro.name) > -1 ) {
              var templateObj = $.parseJSON(getTemplateBody(selectedTemplate)); 
              return templateObj.wikiSyntax;
          }
      }
      return macro.bodyHtml;
   } 
   
   function isMacroBodyExists() {
       if(AJS.MacroBrowser.selectedMacroDefinition != undefined){
            var macroBody = AJS.MacroBrowser.selectedMacroDefinition.body;
            if(macroBody) {
                if(macroBody.trim() && macroBody.trim().length > 0 ){
                    return true;
                }
            }
       }
       return false;
   }
    /**
     * Retrieves the macro body of the selected template.
     *
     * @param templateName selected template name from drop down
     */
    function getTemplateBody(templateName) {
        var templateObj = '';
        AJS.$.ajax({
            async: false,
            type: 'GET',
            contentType: 'application/json',
            url: AJS.params.contextPath + "/rest/bobswift/graphviz/latest/templatebody?templateName="+templateName,
            dataType: 'text',
            success: function( data, textStatus, jqXHR)
            {
                if(jqXHR.status == 200){
                    templateObj  = data;
                }
            },
            error: function( jqXHR, textStatus, errorThrown)
            {
                return false;
            },
            complete: function( jqXHR, textStatus, errorThrown)
            {
                return false;
            }
        });
        return templateObj;
    };
})(AJS.$);
