/*
 * Copyright (c) 2017. GLiNTECH Pty Ltd.
 */

import Immutable from "immutable";
import ImmutableStore from "./ImmutableStore.js";
import AppDispatcher from "../dispatchers/AppDispatcher.js";
import BlogWatchersConstants from "../constants/BlogWatchersConstants.js";
import analytics from "../analytics.js";

const BASE_URL = `${AJS.contextPath()}/rest/blogwatchersresource/latest`;

class BlogWatchersUserGroupStore extends ImmutableStore {
    constructor(props) {
        super(props);
        AppDispatcher.register((action) => {
            switch (action.type) {
                case BlogWatchersConstants.ActionTypes.CHOOSE_RESULT:
                    this._addItem(action.data);
                    break;
                case BlogWatchersConstants.ActionTypes.REMOVE_WATCHING_USER:
                    this._deleteUser(action.data);
                    break;
                case BlogWatchersConstants.ActionTypes.REMOVE_WATCHING_GROUP:
                    this._deleteGroup(action.data);
                    break;
                case BlogWatchersConstants.ActionTypes.SET_SPACE_OPTIONS:
                    this._setSpaceOptions(action.data);
                    break;
            }
        });
    }

    getInitialData() {
        return {
            users: Immutable.List([]),
            groups: Immutable.List([]),
            spaceOptions: Immutable.Map(),
            licenseDetailsModel: Immutable.Map(),
            spaceKey: "",
        };
    }

    setInitData(data) {
        this.set(
            "groups",
            Immutable.List(data.groups).sortBy((g) => g.name)
        );
        this.set(
            "users",
            Immutable.List(data.users).sortBy((u) => u.username)
        );
        this.set("spaceKey", data.spaceKey);
        this.set("spaceOptions", Immutable.Map(data.spaceOptions));
        this.set("licenseDetailsModel", Immutable.Map(data.licenseDetailsModel));
    }

    getAll() {
        return {
            groups: this.get("groups").toArray(),
            users: this._getUsers().toArray(),
            spaceKey: this.get("spaceKey"),
            spaceOptions: this.get("spaceOptions").toJS(),
            licenseDetailsModel: this.get("licenseDetailsModel").toJS(),
        };
    }

    getSpaceOptions() {
        return this.get("spaceOptions").toJS();
    }

    _addItem(item) {
        switch (item.type) {
            case "group":
                this._addGroup(item);
                break;
            case "user":
                this._addUser(item);
        }
    }

    _addUser(user) {
        this._putUser(user).then(() => {
            var users = this._getUsers();
            if (!users.find((u) => u.username === user.username)) {
                this._setUsers(users.push(user).sortBy((u) => u.username));
                analytics("addUser");
            }
        });
    }

    _putUser(user) {
        return AJS.$.ajax(`${this._getBaseUrl()}/watcher`, {
            type: "PUT",
            contentType: "application/json",
            data: JSON.stringify({
                username: user.username,
            }),
        });
    }

    _deleteUser(user) {
        var request = AJS.$.ajax(`${this._getBaseUrl()}/watcher`, {
            type: "DELETE",
            contentType: "application/json",
            data: JSON.stringify({
                username: user.username,
            }),
        });

        request.then(() => {
            this._setUsers(this._getUsers().filterNot((u) => u.username === user.username));
            analytics("removeUser");
        });
    }

    _addGroup(group) {
        this._putGroup(group).then(() => {
            var groups = this._getGroups();
            if (!groups.find((g) => g.name === group.name)) {
                this._setGroups(groups.push(group).sortBy((u) => u.name));
                analytics("addGroup");
            }
        });
    }

    _putGroup(group) {
        return AJS.$.ajax(`${this._getBaseUrl()}/group`, {
            type: "PUT",
            contentType: "application/json",
            data: JSON.stringify({
                groupName: group.name,
            }),
        });
    }

    _deleteGroup(group) {
        var url = `${this._getBaseUrl()}/group`;
        var request = AJS.$.ajax(url, {
            type: "DELETE",
            contentType: "application/json",
            data: JSON.stringify({
                groupName: group.name,
            }),
        });

        request.then(() => {
            this._setGroups(this._getGroups().filterNot((g) => g.name === group.name));
            analytics("deleteGroup");
        });
    }

    _setSpaceOptions(options) {
        this.set("spaceOptions", Immutable.Map(options));
        var url = `${this._getBaseUrl()}/spaceOptions`;
        AJS.$.ajax(url, {
            type: "PUT",
            contentType: "application/json",
            data: JSON.stringify(options),
        });
        analytics(options.disableOptOut ? "optoutDisabled" : "optoutEnabled");
    }

    _getBaseUrl() {
        return `${BASE_URL}/${this.get("spaceKey")}`;
    }

    _getGroups() {
        return this.get("groups");
    }

    _setGroups(groups) {
        this.set("groups", groups);
    }

    _getUsers() {
        return this.get("users");
    }

    _setUsers(users) {
        this.set("users", users);
    }
}

export default new BlogWatchersUserGroupStore();
