/*
 * Decompiled with CFR 0.152.
 */
package com.yubico.webauthn;

import com.google.common.primitives.Bytes;
import com.upokecenter.cbor.CBORObject;
import com.yubico.webauthn.data.ByteArray;
import com.yubico.webauthn.data.COSEAlgorithmIdentifier;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.HashMap;
import java.util.stream.Stream;

final class WebAuthnCodecs {
    private static final ByteArray EC_PUBLIC_KEY_OID = new ByteArray(new byte[]{42, -122, 72, -50, 61, 2, 1});
    private static final ByteArray P256_CURVE_OID = new ByteArray(new byte[]{42, -122, 72, -50, 61, 3, 1, 7});
    private static final ByteArray P384_CURVE_OID = new ByteArray(new byte[]{43, -127, 4, 0, 34});
    private static final ByteArray P512_CURVE_OID = new ByteArray(new byte[]{43, -127, 4, 0, 35});
    private static final ByteArray ED25519_ALG_ID = new ByteArray(new byte[]{48, 5, 6, 3, 43, 101, 112});

    WebAuthnCodecs() {
    }

    static ByteArray ecPublicKeyToRaw(ECPublicKey key) {
        int fieldSizeBytes = Math.toIntExact(Math.round(Math.ceil((double)key.getParams().getCurve().getField().getFieldSize() / 8.0)));
        byte[] x = key.getW().getAffineX().toByteArray();
        byte[] y = key.getW().getAffineY().toByteArray();
        byte[] xPadding = new byte[Math.max(0, fieldSizeBytes - x.length)];
        byte[] yPadding = new byte[Math.max(0, fieldSizeBytes - y.length)];
        Arrays.fill(xPadding, (byte)0);
        Arrays.fill(yPadding, (byte)0);
        return new ByteArray(Bytes.concat({4}, xPadding, Arrays.copyOfRange(x, Math.max(0, x.length - fieldSizeBytes), x.length), yPadding, Arrays.copyOfRange(y, Math.max(0, y.length - fieldSizeBytes), y.length)));
    }

    static ByteArray rawEcKeyToCose(ByteArray key) {
        int coseCrv;
        COSEAlgorithmIdentifier coseAlg;
        byte[] keyBytes = key.getBytes();
        int len = keyBytes.length;
        int lenSub1 = keyBytes.length - 1;
        if (len != 64 && len != 96 && len != 132 && (keyBytes[0] != 4 || lenSub1 != 64 && lenSub1 != 96 && lenSub1 != 132)) {
            throw new IllegalArgumentException(String.format("Raw key must be 64, 96 or 132 bytes long, or start with 0x04 and be 65, 97 or 133 bytes long; was %d bytes starting with %02x", keyBytes.length, keyBytes[0]));
        }
        int start = len == 64 || len == 96 || len == 132 ? 0 : 1;
        int coordinateLength = (len - start) / 2;
        HashMap<Long, Object> coseKey = new HashMap<Long, Object>();
        coseKey.put(1L, 2L);
        switch (len - start) {
            case 64: {
                coseAlg = COSEAlgorithmIdentifier.ES256;
                coseCrv = 1;
                break;
            }
            case 96: {
                coseAlg = COSEAlgorithmIdentifier.ES384;
                coseCrv = 2;
                break;
            }
            case 132: {
                coseAlg = COSEAlgorithmIdentifier.ES512;
                coseCrv = 3;
                break;
            }
            default: {
                throw new RuntimeException("Failed to determine COSE EC algorithm. This should not be possible, please file a bug report.");
            }
        }
        coseKey.put(3L, coseAlg.getId());
        coseKey.put(-1L, coseCrv);
        coseKey.put(-2L, Arrays.copyOfRange(keyBytes, start, start + coordinateLength));
        coseKey.put(-3L, Arrays.copyOfRange(keyBytes, start + coordinateLength, start + 2 * coordinateLength));
        return new ByteArray(CBORObject.FromObject(coseKey).EncodeToBytes());
    }

    static PublicKey importCosePublicKey(ByteArray key) throws IOException, InvalidKeySpecException, NoSuchAlgorithmException {
        CBORObject cose = CBORObject.DecodeFromBytes(key.getBytes());
        int kty = cose.get(CBORObject.FromObject(1)).AsInt32();
        switch (kty) {
            case 1: {
                return WebAuthnCodecs.importCoseEdDsaPublicKey(cose);
            }
            case 2: {
                return WebAuthnCodecs.importCoseEcdsaPublicKey(cose);
            }
            case 3: {
                return WebAuthnCodecs.importCoseRsaPublicKey(cose);
            }
        }
        throw new IllegalArgumentException("Unsupported key type: " + kty);
    }

    private static PublicKey importCoseRsaPublicKey(CBORObject cose) throws NoSuchAlgorithmException, InvalidKeySpecException {
        RSAPublicKeySpec spec = new RSAPublicKeySpec(new BigInteger(1, cose.get(CBORObject.FromObject(-1)).GetByteString()), new BigInteger(1, cose.get(CBORObject.FromObject(-2)).GetByteString()));
        return KeyFactory.getInstance("RSA").generatePublic(spec);
    }

    private static PublicKey importCoseEcdsaPublicKey(CBORObject cose) throws NoSuchAlgorithmException, InvalidKeySpecException {
        ByteArray curveOid;
        int crv = cose.get(CBORObject.FromObject(-1)).AsInt32Value();
        ByteArray x = new ByteArray(cose.get(CBORObject.FromObject(-2)).GetByteString());
        ByteArray y = new ByteArray(cose.get(CBORObject.FromObject(-3)).GetByteString());
        switch (crv) {
            case 1: {
                curveOid = P256_CURVE_OID;
                break;
            }
            case 2: {
                curveOid = P384_CURVE_OID;
                break;
            }
            case 3: {
                curveOid = P512_CURVE_OID;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown COSE EC2 curve: " + crv);
            }
        }
        ByteArray algId = WebAuthnCodecs.encodeDerSequence(WebAuthnCodecs.encodeDerObjectId(EC_PUBLIC_KEY_OID), WebAuthnCodecs.encodeDerObjectId(curveOid));
        ByteArray rawKey = WebAuthnCodecs.encodeDerBitStringWithZeroUnused(new ByteArray(new byte[]{4}).concat(x).concat(y));
        ByteArray x509Key = WebAuthnCodecs.encodeDerSequence(algId, rawKey);
        KeyFactory kFact = KeyFactory.getInstance("EC");
        return kFact.generatePublic(new X509EncodedKeySpec(x509Key.getBytes()));
    }

    private static ByteArray encodeDerLength(int length) {
        if (length <= 127) {
            return new ByteArray(new byte[]{(byte)length});
        }
        if (length <= 65535) {
            if (length <= 255) {
                return new ByteArray(new byte[]{-127, (byte)length});
            }
            return new ByteArray(new byte[]{-126, (byte)(length >> 8), (byte)(length & 0xFF)});
        }
        throw new UnsupportedOperationException("Too long: " + length);
    }

    private static ByteArray encodeDerObjectId(ByteArray oid) {
        return new ByteArray(new byte[]{6, (byte)oid.size()}).concat(oid);
    }

    private static ByteArray encodeDerBitStringWithZeroUnused(ByteArray content) {
        return new ByteArray(new byte[]{3}).concat(WebAuthnCodecs.encodeDerLength(1 + content.size())).concat(new ByteArray(new byte[]{0})).concat(content);
    }

    private static ByteArray encodeDerSequence(ByteArray ... items) {
        ByteArray content = Stream.of(items).reduce(ByteArray::concat).orElseGet(() -> new ByteArray(new byte[0]));
        return new ByteArray(new byte[]{48}).concat(WebAuthnCodecs.encodeDerLength(content.size())).concat(content);
    }

    private static PublicKey importCoseEdDsaPublicKey(CBORObject cose) throws InvalidKeySpecException, NoSuchAlgorithmException {
        int curveId = cose.get(CBORObject.FromObject(-1)).AsInt32();
        switch (curveId) {
            case 6: {
                return WebAuthnCodecs.importCoseEd25519PublicKey(cose);
            }
        }
        throw new IllegalArgumentException("Unsupported EdDSA curve: " + curveId);
    }

    private static PublicKey importCoseEd25519PublicKey(CBORObject cose) throws InvalidKeySpecException, NoSuchAlgorithmException {
        ByteArray rawKey = new ByteArray(cose.get(CBORObject.FromObject(-2)).GetByteString());
        ByteArray x509Key = WebAuthnCodecs.encodeDerSequence(ED25519_ALG_ID, WebAuthnCodecs.encodeDerBitStringWithZeroUnused(rawKey));
        KeyFactory kFact = KeyFactory.getInstance("EdDSA");
        return kFact.generatePublic(new X509EncodedKeySpec(x509Key.getBytes()));
    }

    static String getJavaAlgorithmName(COSEAlgorithmIdentifier alg) {
        switch (alg) {
            case EdDSA: {
                return "EDDSA";
            }
            case ES256: {
                return "SHA256withECDSA";
            }
            case ES384: {
                return "SHA384withECDSA";
            }
            case ES512: {
                return "SHA512withECDSA";
            }
            case RS256: {
                return "SHA256withRSA";
            }
            case RS384: {
                return "SHA384withRSA";
            }
            case RS512: {
                return "SHA512withRSA";
            }
            case RS1: {
                return "SHA1withRSA";
            }
        }
        throw new IllegalArgumentException("Unknown algorithm: " + (Object)((Object)alg));
    }

    static String jwsAlgorithmNameToJavaAlgorithmName(String alg) {
        switch (alg) {
            case "RS256": {
                return "SHA256withRSA";
            }
        }
        throw new IllegalArgumentException("Unknown algorithm: " + alg);
    }
}

