/* global require */
(function() {
  var $ = require('jquery');
  var enabled = window.name === 'structure-confluence-page';
  if (!enabled) return;
  var traceFocusEnabled = false;

  var structureMessageChannel;

  try {

    initializeStructureFrameWindow();
    applyOurStyle();
    makeLinksOpenInNewWindow();

    structureMessageChannel = new StructureMessageChannel();
    $(function() {
      structureMessageChannel.broadcast({name: 'load', pageId: getPageId()});
      structureMessageChannel.receive('saveDraft', saveDraft);
    });
    
    postFocusBlur(structureMessageChannel);
    postKnownShortcuts(structureMessageChannel);
    postMouseEvents(structureMessageChannel);
    structureMessageChannel.receive('focus', setFocus);
    // Must not be inside $(): when the user chooses "Resume editing" to edit draft, page is reloaded and the event is 
    //  fired from inside $(), so we won't catch it.
    suppressEditorDirtyWarning();
    suppressCommentDirtyWarning();
    traceIframeFocus();

    fixAdjustMainHeaderSizeLoop();

  } catch (e) {
    log('structure-confluence-helper failed', e);
  }

  // We need to hide extra elements - application header and menus, sidebar, footer.
  // It's important that we do it here
  function applyOurStyle() {
    $('head').append('<style type="text/css" id="structure-helper-bare">' +
        // Account for stupid aui-responsive-header which breaks when no menu items are visible.
        // The failing branch is not triggered if the header is there, but invisible.
      '#header {' +
      '  visibility: hidden !important;' +
      '  position: absolute !important;' +
      '  height: 0 !important;' +
      '}' +
      '' +
      '#header-precursor,' +
      '#footer,' +
      '.ia-splitter-left,' +
      '#splitter-sidebar,' +
      '#splitter > .vsplitbar {' +
      '  display: none !important;' +
      '}' +
      '' +
        // it is set from JS via style attribute, we override it here
      '#page .ia-splitter #main {' +
      '  margin-left: 0 !important;' +
      '}' +
      '' +
      '#main {' +
      '  border: none !important;' +
      '}' +
      '' +
      '#full-height-container, body {' +
      '  background-color: white !important;' +
      '}' +
      '' +
      '#splitter {' +
      '  height: 100% !important;' +
      '}' +
      '' +
      '#splitter-content {' +
      '  left: 0 !important;' +
      '  width: 100% !important;' +
      '  height: 100% !important;' +
      '}' +
        // Do not display top-level banners
      'body > div.aui-message {' +
      '  display: none' +
      '}' +

        // Refined Wiki STR-1270
      '#edge_design_top,' +
      '#rw_side_menu,' +
      '#rw_menu_bar_wrapper {' +
      '  display:none !important;' +
      '}' +
      '#rw_wrapper {' +
      '  margin: 0 !important;' +
      '}' +
      '#rw_main {' +
      '  width:100% !important;' +
      '  padding: 0 !important;' +
      '}' +
      '#edge_design_content {' +
      '  padding: 0 !important;' +
      '}' +
      '#rw_left_column {' +
      '  display: none !important;' +
      '}' +
      '#rw_main_column {' +
      '  margin: 0 !important;' +
      '  padding: 0 !important;' +
      '}' +
      '#rw_page_right {' +
      '  border: 0 !important;' +
      '  border-radius: 0 !important;' +
      '  -webkit-border-radius: 0 !important;' +
      '}' +
      '#rw_footer_wrapper {' +
      '  display: none !important;' +
      '}' +
      '</style>');
  }

  function makeLinksOpenInNewWindow() {
    $(safeFun(function makeLinksOpenInNewWindow0() {
      if (insertBase()) {
        // Set default target for all forms and the selected list of links:
        // - "Edit" link on the view page
        // - "Delete comment" link
        // All possible dynamically inserted forms are patched, however there is no known scenario when it's necessary - it's done just in case.
        // Despite the fact that edit-page-form is created dynamically, it's actually a clone of "inlinecommentform" form,
        // which is patched on a page load, so the edit-page-form will have correct target anyway
        $('form, a#editPageLink, a#viewPageLink, li.comment-action-remove > a, ol.aui-nav-pagination a, ul.pagination a').each(updateTargetAttr);
        attachNodeInsertionListeners();
      }
    }));

    function insertBase() {
      var newBase$;
      var oldBase$ = $('base');
      var href = null;
      if (oldBase$.length) {
        href = oldBase$.attr('href');
        log("Removing existing <base> element with target=" + oldBase$.attr('target') + " and href=" + href);
        oldBase$.remove();
      }
      var head$ = $('head');
      if (!head$.length) {
        log("No <head>, won't insert <base>");
        return false;
      }

      newBase$ = $('<base>');
      newBase$.attr('target', '_blank');
      if (href) newBase$.attr('href', href);
      head$.prepend(newBase$);
      return true;
    }

    function attachNodeInsertionListeners() {
      function updateInsertedNode(e) {
        var oe = e.originalEvent;
        if (oe && oe.animationName === 'StructurePagesNodeInserted') {
          updateTargetAttr.apply(e.target);
        }
      }

      $(document)
        .on('animationstart', updateInsertedNode)
        .on('MSAnimationStart', updateInsertedNode)
        .on('webkitAnimationStart', updateInsertedNode);
    }

    function updateTargetAttr() {
      var e$ = $(this);
      if (!e$.attr('target')) {
        e$.attr('target', '_self');
      }
    }
  }

  function StructureMessageChannel() {
    var MSG_PREFIX = 'structure.confluence.';
    var structureWindow = getStructureWindow();
    
    /**
     * Sends a message to all domains. NB: do not send informations about UI events and their data there!
     * @param {{name:string}} msg
     * */
    this.broadcast = function post(msg) {
      if (!structureWindow) return;
      try {
        msg.name = MSG_PREFIX + msg.name;
        structureWindow.postMessage(msg, '*');
      } catch (e) {
        log('structureMessageChannel failed to send ' + msg.name + ' ' + e);
      }
    };

    this.receive = function(name, handler) {
      var fullName = MSG_PREFIX + name;
      var listener = function(message) {
        var data = message && message.data;
        log('received ' + JSON.stringify(data));
        if (data && data['name'] === fullName) {
          handler(data);
        }        
      };
      window.addEventListener('message', listener, false);
    };
    
    return this;

    function getStructureWindow() {
      var parent = window.parent;
      if (parent && parent.name && parent.name.indexOf('structure-confluence-page') === 0) {
        if (parent.parent && parent.parent !== parent) {
          // topWindow.parent === topWindow
          return parent.parent;
        } else {
          log('parent window does not have parent');
        }
      } else {
        log('unknown parent window ' + (parent && parent.name) + ' ' + parent);
      }
    }
  }

  function postFocusBlur(structureMessageChannel) {
    $(window).on('focus blur', function(e) {
      structureMessageChannel.broadcast({name: e.type, iframeId: 'page-' + getPageId()});
    });
    
    // For editor iframe
    AJS.bind('rte-ready', safeFun(function postEditorFocusBlur() {
      listenNestedIframe(AJS.Rte.getEditorFrame());
    }));
    // For preview iframe
    $(document).on('iframeAppended', safeFun(function postPreviewFocusBlur(e, iframe) {
      listenNestedIframe(iframe);
      tracePreviewIframeFocus(iframe);
    }));

    function listenNestedIframe(iframe) {
      $(iframe.contentWindow).on('focus blur', function (e) {
        structureMessageChannel.broadcast({name: e.type, iframeId: 'page-' + getPageId() + '-editor'});
      });

      // The editor should be focused now. For some reason, even though the focusing happens after our subscription,
      // we are not notified.
      structureMessageChannel.broadcast({name: 'focus', iframeId: 'page-' + getPageId() + '-editor'});
    }
  }
  
  function postKnownShortcuts(structureMessageChannel) {
    var modifiers = {shift: 1000, ctrl: 10000, alt: 100000, meta: 1000000};
    // See: Keyboard.js, keymap.js
    var shortcuts = {
      79: 'openIssue', // o
      1079: 'toggleIssueDetailsNoFocus', // Shift+o
      90: 'fullScreen.toggle', // z
      220: 'tabFocusPanel', // \
      // layouts
      110049 : 'layout.single',
      110050 : 'layout.double',
      110051 : 'layout.details',
      110052 : 'layout.history'
    };
    var disabledShortcuts = {
      67: 'createSubPage' // c
    };
    $(window).on('keydown', function(e) {
      var code = e.keyCode + modifiersFromEvent(e);
      if (!acceptTarget(e)) return true;
      if (disabledShortcuts[code]) return false;
      var action = shortcuts[code];
      if (action) {
        // todo domain-only
        structureMessageChannel.broadcast({name: 'action', action: action});
      }
    });

    function modifiersFromEvent(e) {
      var code = 0;
      if (e.shiftKey) code += modifiers.shift;
      if (e.ctrlKey) code += modifiers.ctrl;
      if (e.altKey) code += modifiers.alt;
      // workaround for stupid jQuery normalization
      if (e.originalEvent && e.originalEvent.metaKey) code += modifiers.meta;
      return code;
    }

    // Note: keypresses inside editor iframe are not captured on window, but keypresses in some other places are.
    // Known examples:
    // 1. Login form inputs.
    // 2. Page title input.
    function acceptTarget(e) {
      var target$ = $(e.target);
      // Don't handle keypresses in inputs. See also KeyboardManager.js::acceptTarget
      return !target$.is('input,textarea,select,button');
    }
  }

  function postMouseEvents() {
    $(window).on('mousemove', function(e) {
      structureMessageChannel.broadcast(mouseEventToMessage(e));
    });
    $(window).on('mouseup', function(e) {
      structureMessageChannel.broadcast(mouseEventToMessage(e));
    });

    function mouseEventToMessage(e) {
      var msg = {};
      msg.name = e.type;
      msg.date = new Date();
      var keys = ['screenX', 'screenY', 'ctrlKey', 'altKey', 'shiftKey', 'metaKey'];
      keys.forEach(function(key) {
        msg[key] = e[key];
      });
      return msg;
    }
  }
  
  function setFocus() {
    var rte = AJS && AJS.Rte;
    var editor = rte && rte.getEditor && rte.getEditor();
    if (editor) {
      rte.editorFocus(editor);
    } else {
      window.focus();
    }
  }
  
  function suppressEditorDirtyWarning() {
    AJS.bind('rte-ready', safeFun(function suppressEditorDirtyWarning0() {
      // Drafts are bound to unload message in the handler for the same event, we must execute after it, hence defer
      window.setTimeout(function() {
        var Drafts = getDrafts();
        if (Drafts) {
          Drafts.unBindUnloadMessage && Drafts.unBindUnloadMessage();
          log("Structure helper performed Confluence.Editor.Drafts.unBindUnloadMessage()");
          // Now bind save on beforeunload. Do all the same but suppress the message
          AJS.bind('beforeunload.editor', safeFun(function structureBeforeUnloadEditor() {
            // AJS.params.draftType is falsy if comment is being edited. 
            // If we proceed, we'll send invalid data to the server and cause exceptions to be raised.
            if (AJS.params && AJS.params.draftType) {
              Drafts.unloadMessage && Drafts.unloadMessage();
            }
          }));
        }
      }, 10);
    }));
  }
  
  function suppressCommentDirtyWarning() {
    // Hackety hack
    // The confirmation is shown by 'confluenceleavecomment' tinyMCE plugin, and there's no way to stop it except by 
    // mocking the things that it checks
    $(window).on('beforeunload', safeFun(function structureBeforeUnloadComment() {
      // Only for comments. Mirroring the logic in tinymce-adapter.js::initTinyMce() which calls getProfileByContentType()
      if (AJS.Meta && AJS.Meta.get('content-type') === 'comment') {
        var Editor = Confluence && Confluence.Editor;
        var editorIsEmpty;
        if (Editor) {
          editorIsEmpty = Editor.isEmpty;
          Editor.isEmpty = returnTrue;
          // Restore back -- if we somehow end up not navigating away
          window.setTimeout(function () {
            Editor.isEmpty = editorIsEmpty;
          }, 10);
        }
      }
    }));
  }
  
  function saveDraft() {
    var Drafts = getDrafts();
    if (!Drafts) return;
    // If comment is being edited, we're also called here. If we proceed, we'll send invalid data to the server
    // and cause exceptions to be raised.
    if (!(AJS.params && AJS.params.draftType)) return; 
    // Copied from page-editor-drafts.js::unloadMessage() @ Confluence 5.7.
    Drafts.save({
      // Skip error handler because it shows a BRIGHT RED draft save ERROR due to the browser
      // cancelling in-flight ajax requests during page unload even if the server received the
      // draft request and saved it successfully but did not respond in time.
      skipErrorHandler: true
    });
  }
  
  function getDrafts() {
    return Confluence && Confluence.Editor && Confluence.Editor.Drafts;
  }
  
  function getPageId() {
    return AJS && AJS.Meta && parseInt(AJS.Meta.get('page-id'), 10) || '*';    
  }

  function traceIframeFocus() {
    if (traceFocusEnabled) {
      window.setInterval(function traceIframeFocusJob() {
        trace('[iframe] ' + describeActiveElement(document));
      }, 500);
    }
  }

  function tracePreviewIframeFocus(iframe) {
    if (traceFocusEnabled) {
      var interval = window.setInterval(function tracePreviewIframeFocusJob() {
        if (iframe.contentDocument) {
          trace('[preview] ' + describeActiveElement(iframe.contentDocument));
        } else {
          window.clearInterval(interval);
        }
      }, 500);
    }
  }

  function describeActiveElement(doc) {
    var desc = doc.hasFocus() ? 'hasFocus ' : '';
    var activeElem = doc.activeElement;
    desc += activeElem.tagName;
    if (activeElem.id) desc += "#" + activeElem.id;
    if (activeElem.className) desc += "." + activeElem.className;
    return desc;
  }

  function fixAdjustMainHeaderSizeLoop() {
    // PAGES-219 -- this function seems to do useful work (maintaining fixed width of the header) only
    // when sidebar is resized. It also causes a debounced cycle in certain cases within Pages.
    // Since we're hiding the sidebar, it seems to be no longer needed and it fixes the problem.
    $(function() {
      try {
        require('confluence/fh/utils/dom').adjustMainHeaderSize = function() {};
      } catch (e) {
        log('failed to fix loop with adjustMainHeaderSize()', e);
      }
    });
  }

  /*
      There are some atlassian code for gadgets which calls 'jira/util/top-same-origin-window' util to top-most same origin window.
      When a confluence page is accessed via structure-page the top most window is our 'structure-frame' iframe window and not the actual
      confluence iframe, which seems to not have some necessary attributes like 'jQuery' and 'AJS'.
    */
  function initializeStructureFrameWindow() {
    window.parent.jQuery = window.jQuery;
    window.parent.AJS = window.AJS;
  }

  // Utilities
  
  function returnTrue() {
    return true;
  }
  
  function safeFun(fun) {
    return function () {
      try {
        fun.apply(this, arguments);
      } catch (e) {
        log('structure-helper function failed', e);
      }
    }
  }

  function trace(msg) {
    if (traceFocusEnabled) {
      log(msg);
    }
  }

  function log(msg, e) {
    try {
      if (console && console.log) {
        if (e) msg += ' ' + e + ' ' + e.stack;
        console.log(msg);
      }
    } catch (ignored) {}
  }
})();
