define('structure/macro/GanttMacroConfig', ['structure/macro/MacroFields'], function (MacroFields) {
  'use strict';
  const requestedAdditionalFields = {};

  const mfs = MacroFields.createMacroFields('/rest/structure-helper/1.0/ganttmacro/settings');

  function updateTimelineMode() {
    const v = mfs.fields.timelineMode?.input?.find('option:selected')?.val() || '0';
    mfs.toggleField(mfs.fields.fixedStartDate, v === "1");
    mfs.toggleField(mfs.fields.todayOffset, v === "2");
  }

  function updateWBSSection() {
    mfs.toggleField(mfs.fields.wbsWidth, mfs.fields.wbs?.input?.is(':checked'));
  }

  function createOption(key, value) {
    return `<option value="${mfs.encodeHTML(key)}">${mfs.encodeHTML(value)}</option>`;
  }

  function createOptGroup(label, items, prefix = '') {
    return `<optgroup label="${mfs.encodeHTML(label)}">
            ${items.map(item => createOption(`${prefix}${item.key}`, item.value)).join('')}
         </optgroup>`;
  }

  function excludeInvalidRestoredValue(isRestoredValueInvalid, restoredValue) {
    return (term, text, option) =>
      (!isRestoredValueInvalid || option[0].value !== restoredValue) && text.toUpperCase().indexOf(term.toUpperCase()) >= 0;
  }

  function assignBaselineOptions(field, data, restoredValue) {
    const validOptions = [
      data.noBaselineOption.key,
      `g${data.originalScheduleOption.key}`,
      ...(data.baselines?.map(b => b.key) || []),
      ...(data.sandboxes?.map(s => `g${s.key}`) || [])
    ];
    const isRestoredValueInvalid = restoredValue && !validOptions.includes(restoredValue)

    const html = [
       isRestoredValueInvalid
        ? createOption(restoredValue, WRM.I18n.getText('com.almworks.structure.confluence.helper.ganttmacro.param.baseline.invalid'))
        : '',
      createOption(data.noBaselineOption.key, data.noBaselineOption.value),
      createOption(`g${data.originalScheduleOption.key}`, data.originalScheduleOption.value),
      createOptGroup(
        WRM.I18n.getText('com.almworks.structure.confluence.helper.ganttmacro.param.baseline.group.baselines'),
        data.baselines || []
      ),
      createOptGroup(
        WRM.I18n.getText('com.almworks.structure.confluence.helper.ganttmacro.param.baseline.group.sandboxes'),
        data.sandboxes || [],
        'g'
      )
    ].join('');
    field.input.html(html);

    field.input.val(restoredValue);
    field.setValue(field.input.val());
    field.input.auiSelect2({
      matcher: excludeInvalidRestoredValue(isRestoredValueInvalid, restoredValue)
    });
  }

  function assignSandboxOptions(field, sandboxes, restoredValue) {
    const options = sandboxes?.map(o => createOption(o.key, o.value)) || [];
    const isRestoredValueInvalid = restoredValue && !sandboxes?.some(v => v.key === restoredValue);

    if (isRestoredValueInvalid) {
      options.push(createOption(
        restoredValue,
        WRM.I18n.getText('com.almworks.structure.confluence.helper.ganttmacro.param.sandbox.invalid')
      ));
    }
    field?.input?.html(options.join('') || '');

    field.input.val(restoredValue);
    field.setValue(field.input.val());
    field.input.auiSelect2({
      matcher: excludeInvalidRestoredValue(isRestoredValueInvalid, restoredValue)
    });
  }

  function updateDependentFields(appLink, data, restoredValues) {
    assignBaselineOptions(mfs.fields.baseline, data, restoredValues?.baseline);
    const sandboxes = data?.sandboxes ? [data?.noSandboxOption, ...data.sandboxes] : [data?.noSandboxOption];
    assignSandboxOptions(mfs.fields.sandbox, sandboxes, restoredValues?.sandbox);
    mfs.toggleField(mfs.fields.sandbox, data?.sandboxes);
  }

  function loadDependentFields(appLink, restoredValues) {
    if ((requestedAdditionalFields.link === appLink) && appLink) return;
    requestedAdditionalFields.link = appLink;
    let url = '/rest/structure-helper/1.0/ganttmacro/additionalsettings';
    if (appLink) {
      url = url + '?link=' + encodeURIComponent(appLink);
      const currentStructure = mfs.fields?.structure?.input?.val();
      if (currentStructure) {
        url = url + '&structureId=' + encodeURIComponent(currentStructure);
      }
    }

    const callback = (data) => {
      if (requestedAdditionalFields.link !== appLink) return;

      updateDependentFields(appLink, data, restoredValues);

      mfs.adjustAppLinksSelector(data?.authurl);
      mfs.setFieldError(mfs.fields.applink, data?.error);
      mfs.updatePreview();
    }

    fetch(mfs.contextPath + url)
      .then(r => r.json())
      .then(r => callback(r))
      .catch(error => console.log(error))
      .finally(() => delete requestedAdditionalFields.link);
  }

  mfs.onUpdateFields(function (appLink, data, restoredValues) {
    mfs.assignOptions(mfs.fields.applink, data?.applinks, restoredValues?.applink || appLink);
    mfs.assignOptions(mfs.fields.structure, data?.structures, restoredValues?.structure);
    mfs.assignOptions(mfs.fields.timelineZoom, data?.zoomLevels, restoredValues?.timelineZoom);
    mfs.assignOptions(mfs.fields.timelineMode, data?.timelineModes, restoredValues?.timelineMode);
    mfs.assignOptions(mfs.fields.fiscalYearStartMonth, data?.yearStartOptions, restoredValues?.fiscalYearStartMonth);
    mfs.updateStructureName();
    updateTimelineMode();
    updateWBSSection();

    mfs.fields.applink?.input?.prop('disabled', !data?.applinks);
    mfs.toggleField(mfs.fields.structure, data?.structures);
    mfs.toggleField(mfs.fields.timelineZoom, data?.zoomLevels);
    mfs.toggleField(mfs.fields.timelineMode, data?.timelineModes);
    mfs.toggleField(mfs.fields.fiscalYearStartMonth, data?.yearStartOptions);
    mfs.toggleField(mfs.fields.filtertype, data?.structures);
    mfs.toggleFilterFields(data?.structures && mfs.fields.filtertype?.input?.val());

    /*
      Version-dependent fields:
      `data.ganttVersion` is null for plugin versions below 4.4.1.

      Starting from 4.4.1, the Gantt plugin introduces new chart options (`schedulingConflict`, `taskIcons`)
      and deprecates two older ones (`chartWarnings`, `taskIndicators`)

      If `version` is null → assume pre-4.4.1 and show legacy fields.
      If `version` is set → use new field set introduced in 4.4.1.
     */
    const version = data?.ganttVersion;
    mfs.toggleField(mfs.fields.chartWarnings, !version);
    mfs.toggleField(mfs.fields.taskIndicators, !version);
    mfs.toggleField(mfs.fields.schedulingConflict, !!version);
    mfs.toggleField(mfs.fields.taskIcons, !!version);

    loadDependentFields(appLink, restoredValues);
  });

  function initialize() {
    mfs.macroBrowser().setMacroJsOverride('ganttmacro', {
      fields: {
        enum: {
          applink: mfs.setupAppLinksSelector('applink'),
          structure: mfs.setupSelect('structure', field => {
            field.input.on('change', () => {
              mfs.updateStructureName();
              loadDependentFields(mfs.fields?.applink?.input?.val());
            });
          }),
          filtertype: mfs.setupSelect('filtertype', field => {
            field.input.on('change', () => mfs.toggleFilterFields(field.input.val()));
            setTimeout(() => mfs.toggleFilterFields(field.getValue()));
            setTimeout(() => field.input.auiSelect2());
          }),
          timelineZoom: mfs.setupSelect('timelineZoom'),
          timelineMode: mfs.setupSelect('timelineMode', field => {
            field.input.on('change', () => updateTimelineMode());
          }),
          fiscalYearStartMonth: mfs.setupSelect('fiscalYearStartMonth'),
          baseline: mfs.setupSelect('baseline', undefined, true),
          sandbox: mfs.setupSelect('sandbox'),
          refreshInterval: mfs.setupSelect('refreshInterval', undefined, true)
        },
        string: {
          title: mfs.setupField('title', 'string'),
          textfilter: mfs.setupTextArea('textfilter'),
          jqlfilter: mfs.setupTextArea('jqlfilter'),
          sjqlfilter: mfs.setupTextArea('sjqlfilter'),
          todayOffset: mfs.setupField('todayOffset', 'string', field => {
            mfs.toggleField(field, false);
          }),
          fixedStartDate: mfs.setupField('fixedStartDate', 'string', field => {
            field.input.attr('placeholder', 'yyyy-mm-dd');
            mfs.toggleField(field, false);
          }),
          width: mfs.setupSizeField('width', 'string'),
          height: mfs.setupSizeField('height', 'string'),
          hierarchyLevel: mfs.setupField('hierarchyLevel', 'string'),
          wbsWidth: mfs.setupSizeField('wbsWidth', 'string', field => {
            mfs.toggleField(field, false);
          }),
          structurename: mfs.setupField0('structurename', '_hidden')
        },
        boolean: {
          wbs: mfs.setupField('wbs', 'boolean', field => {
            field.input.on('change', () => updateWBSSection());
          }),
          issueKey: mfs.setupField('issueKey', 'boolean'),
          summary: mfs.setupField('summary', 'boolean'),
          resourceAssignment: mfs.setupField('resourceAssignment', 'boolean'),
          criticalPath: mfs.setupField('criticalPath', 'boolean'),
          progress: mfs.setupField('progress', 'boolean'),
          dependencies: mfs.setupField('dependencies', 'boolean'),
          chartWarnings: mfs.setupField('chartWarnings', 'boolean'),
          schedulingConflict: mfs.setupField('schedulingConflict', 'boolean'),
          taskIndicators: mfs.setupField('taskIndicators', 'boolean'),
          taskIcons: mfs.setupField('taskIcons', 'boolean'),
          baselineDelays: mfs.setupField('baselineDelay', 'boolean'),
          todayMarker: mfs.setupField('todayMarker', 'boolean'),
          projectStartDay: mfs.setupField('projectStartDay', 'boolean'),
          fixVersionMarkers: mfs.setupField('fixVersionMarkers', 'boolean'),
          showSafeView: mfs.setupField('showSafeView', 'boolean'),
          customChartMarkers: mfs.setupField('customChartMarkers', 'boolean')
        }
      }
    });
  }

  return {
    initialize
  };
});
