define('structure/macro/MacroFields', ['jquery', 'wrm/context-path'], function ($, contextPathFunc) {
  const mb = window.parent.AJS.MacroBrowser;
  const requested = {};
  const filterFieldNames = ['textfilter', 'jqlfilter', 'sjqlfilter', 'savedfilter'];
  const contextPath = contextPathFunc() || '';
  const SIZE_FIELD_MAX_LENGTH = 6;

  function encodeHTML(text) {
    return text.replace(/[\u00A0-\u9999<>&]/g, i => '&#' + i.charCodeAt(0) + ';');
  }

  const createMacroFields = function (settingsUrl) {
    const fields = {};
    let updateFieldsCallback;

    function macroBrowser() {
      return mb;
    }

    function updatePreview() {
      mb.previewMacro(mb.dialog.activeMetadata);
    }

    function handleVisibilityChange() {
      if (!document.hidden) getData(fields.applink?.getValue());
    }

    function updateStructureName() {
      fields.structurename?.setValue(fields.structure?.input?.find('option:selected')?.text() || '');
    }

    function setupField0(name, type, setup) {
      return (param, options) => {
        const field = mb.ParameterFields[type](param, options);
        fields[name] = field;
        setup?.(field);
        return field;
      };
    }

    function setupField(name, type, setup) {
      return setupField0(name, type, field => {
        setup?.(field);
        field.input.on('change', () => updatePreview());
      });
    }

    function setupSizeField(name, type, setup) {
      return setupField(name, type, field => {
        setup?.(field);
        field?.input?.attr('maxlength', SIZE_FIELD_MAX_LENGTH);
      });
    }

    function setupSelect(name, setup, visible) {
      return setupField(name, 'enum', field => {
        setup?.(field);
        installErrorElement(field);
        field?.input?.closest('.macro-param-div')?.addClass('ch-macro-select');
        field?.input?.on('change', () => setFieldError(field, undefined));
        toggleField(field, visible);
      });
    }

    function setupTextArea(name, setup) {
      return setupField(name, 'string', field => {
        const textArea = $(`
          <textarea
            id="${field.input.attr('id')}"
            class="${field.input.attr('class')} ch-macro-textarea-param">
          </textarea>
        `);
        textArea.val(mb.settings?.selectedMacro?.params?.[name] || '');
        field.input.replaceWith(textArea);
        field.input = textArea;
        toggleField(field, false);
        setup?.(field);
      });
    }

    function toggleField(field, visible) {
      field?.input?.closest('.macro-param-div')?.toggle(!!visible);
    }

    function installErrorElement(field) {
      field.input.closest('.macro-param-div').append(`<div class="ch-macro-error"></div>`);
    }

    function installAuthElement(field) {
      field.input.closest('.macro-param-div').append(`
          <a class="ch-macro-link" href="#" target="_blank">${WRM.I18n.getText('structuremacro.ui.auth')}</a>
        `);
    }

    function setupAppLinksSelector(name) {
      return setupSelect(name, field => {
        const params = mb.settings?.selectedMacro?.params;
        getData(params?.applink, params);
        field.input.on('change', () => getData(field.input.val()));
        installAuthElement(field);
        adjustAppLinksSelector();
        setFieldError(field, undefined);
      }, true);
    }

    function setFieldError(field, error) {
      const error$ = field.input?.closest('.macro-param-div')?.find('.ch-macro-error');
      error$?.toggle(!!error);
      error$?.html(encodeHTML(error || ''));
    }

    function adjustAppLinksSelector(authurl) {
      const paramDiv$ = fields.applink?.input?.closest('.macro-param-div');
      if (!paramDiv$) return;
      const link$ = paramDiv$.find('.ch-macro-link');

      link$.toggle(!!authurl);
      link$.attr('href', authurl || '#');

      document.removeEventListener('visibilitychange', handleVisibilityChange);
      if (authurl) {
        document.addEventListener('visibilitychange', handleVisibilityChange);
      }
    }

    function assignOptions(field, options, restoredValue, defaultOption) {
      const allOptions = defaultOption
        ? [defaultOption, ...(options || [])]
        : options;
      const html = allOptions?.map(o => `
          <option value="${encodeHTML(o.key)}">
            ${encodeHTML(o.value)}
          </option>
        `)?.join('');
      field?.input?.html(html || '');

      if (restoredValue) {
        if (options?.some(v => v.key === restoredValue)) {
          field.input.val(restoredValue);
        } else {
          setFieldError(field, WRM.I18n.getText('structuremacro.error.invalid-parameter-this', restoredValue));
          field.input.val(defaultOption?.key);
        }
      }

      field.setValue(field.input.val());
      field.input.auiSelect2();
    }

    function toggleFilterFields(value) {
      const fieldName = value + 'filter';
      for (const [name, field] of Object.entries(fields)) {
        if (filterFieldNames.includes(name)) {
          toggleField(field, fieldName === name);
        }
      }
    }

    function onUpdateFields(updateFields) {
      updateFieldsCallback = updateFields;
    }

    function getData(appLink, restoredValues) {
      if (requested.link === appLink && appLink) return;
      requested.link = appLink;
      let url = settingsUrl;
      if (appLink) {
        url = url + '?link=' + encodeURIComponent(appLink);
      }

      const callback = (data) => {
        if (requested.link !== appLink) return;

        updateFieldsCallback(appLink, data, restoredValues);

        adjustAppLinksSelector(data?.authurl);
        setFieldError(fields.applink, data?.error);
        updatePreview();
      }

      fetch(contextPath + url)
        .then(r => r.json())
        .then(r => callback(r))
        .catch(error => console.log(error))
        .finally(() => delete requested.link);
    }

    return {
      fields,
      macroBrowser,
      updatePreview,
      setupField0,
      setupField,
      setupSizeField,
      setupSelect,
      setupTextArea,
      toggleField,
      updateStructureName,
      setupAppLinksSelector,
      assignOptions,
      encodeHTML,
      toggleFilterFields,
      setFieldError,
      adjustAppLinksSelector,
      onUpdateFields,
      contextPath
    };
  };

  return {
    createMacroFields
  }
});