require(['jquery', 'wrm/context-path'], function ($, contextPathFunc) {
  $(document).ready(function () {
    const mb = window.parent.AJS.MacroBrowser;
    const contextPath = contextPathFunc() || '';
    const requested = {};
    const filterFieldNames = ['textfilter', 'jqlfilter', 'sjqlfilter', 'savedfilter'];

    function encodeHTML(text) {
      return text.replace(/[\u00A0-\u9999<>&]/g, i => '&#' + i.charCodeAt(0) + ';');
    }

    function updatePreview() {
      mb.previewMacro(mb.dialog.activeMetadata);
    }

    class MacroFields {
      constructor() {
        this.fields = {}
      };

      handleVisibilityChange = () => {
        if (!document.hidden) this.getData(this.fields.applink?.getValue());
      };

      updateStructureName() {
        this.fields.structurename?.setValue(this.fields.structure?.input?.find('option:selected')?.text() || '');
      }

      setupField0(name, type, setup) {
        return (param, options) => {
          const field = mb.ParameterFields[type](param, options);
          this.fields[name] = field;
          setup?.(field);
          return field;
        };
      }

      setupField(name, type, setup) {
        return this.setupField0(name, type, field => {
          setup?.(field);
          field.input.on('change', () => updatePreview());
        });
      }

      setupSelect(name, setup, visible) {
        return this.setupField(name, 'enum', field => {
          setup?.(field);
          this.installErrorElement(field);
          field?.input?.closest('.macro-param-div')?.addClass('ch-macro-select');
          field?.input?.on('change', () => this.setFieldError(field, undefined));
          this.toggleField(field, visible);
        });
      }

      setupTextArea(name, setup) {
        return this.setupField(name, 'string', field => {
          const textArea = $(`
          <textarea
            id="${field.input.attr('id')}"
            class="${field.input.attr('class')} ch-macro-textarea-param">
          </textarea>
        `);
          textArea.val(mb.settings?.selectedMacro?.params?.[name] || '');
          field.input.replaceWith(textArea);
          field.input = textArea;
          this.toggleField(field, false);
          setup?.(field);
        });
      }

      toggleField(field, visible) {
        field?.input?.closest('.macro-param-div')?.toggle(!!visible);
      }

      installErrorElement(field) {
        field.input.closest('.macro-param-div').append(`<div class="ch-macro-error"></div>`);
      }

      installAuthElement(field) {
        field.input.closest('.macro-param-div').append(`
          <a class="ch-macro-link" href="#" target="_blank">${WRM.I18n.getText('structuremacro.ui.auth')}</a>
        `);
      }

      setupAppLinksSelector(name) {
        return this.setupSelect(name, field => {
          const params = mb.settings?.selectedMacro?.params;
          this.getData(params?.applink, params);
          field.input.on('change', () => this.getData(field.input.val()));
          this.installAuthElement(field);
          this.adjustAppLinksSelector();
          this.setFieldError(field, undefined);
        }, true);
      }

      setFieldError(field, error) {
        const error$ = field.input?.closest('.macro-param-div')?.find('.ch-macro-error');
        error$?.toggle(!!error);
        error$?.html(encodeHTML(error || ''));
      }

      adjustAppLinksSelector(authurl) {
        const paramDiv$ = this.fields.applink?.input?.closest('.macro-param-div');
        if (!paramDiv$) return;
        const link$ = paramDiv$.find('.ch-macro-link');

        link$.toggle(!!authurl);
        link$.attr('href', authurl || '#');

        document.removeEventListener('visibilitychange', this.handleVisibilityChange);
        if (authurl) {
          document.addEventListener('visibilitychange', this.handleVisibilityChange);
        }
      }

      assignOptions(field, options, restoredValue, defaultOption) {
        const allOptions = defaultOption
          ? [defaultOption, ...(options || [])]
          : options;
        const html = allOptions?.map(o => `
          <option value="${encodeHTML(o.key)}">
            ${encodeHTML(o.value)}
          </option>
        `)?.join('');
        field?.input?.html(html || '');

        if (restoredValue) {
          if (options?.some(v => v.key === restoredValue)) {
            field.input.val(restoredValue);
          } else {
            this.setFieldError(field, WRM.I18n.getText('structuremacro.error.invalid-parameter-this', restoredValue));
            field.input.val(defaultOption?.key);
          }
        }

        field.setValue(field.input.val());
        field.input.select2();
      }

      updateFields(appLink, data, restoredValues) {}

      getData(appLink, restoredValues) {
        if ((requested.link === appLink) && appLink) return;
        requested.link = appLink;
        let url = '/rest/structure-helper/1.0/macro/settings';
        if (appLink) {
          url = url + '?link=' + encodeURIComponent(appLink);
        }

        const callback = (data) => {
          if (requested.link !== appLink) return;

          this.updateFields(appLink, data, restoredValues);

          this.adjustAppLinksSelector(data?.authurl);
          this.setFieldError(this.fields.applink, data?.error);
          updatePreview();
          delete requested.link;
        }

        fetch(contextPath + url)
          .then(r => r.json())
          .then(r => callback(r))
          .catch(error => console.log(error));
      }
    }

    class StructureMacroFields extends MacroFields {
      toggleFilterFields(value) {
        const fieldName = value + 'filter';
        for (const [name, field] of Object.entries(this.fields)) {
          if (filterFieldNames.includes(name)) {
            this.toggleField(field, fieldName === name);
          }
        }
      }

      updateFields(appLink, data, restoredValues) {
        this.assignOptions(this.fields.applink, data?.applinks, restoredValues?.applink || appLink);
        this.assignOptions(this.fields.structure, data?.structures, restoredValues?.structure);
        this.assignOptions(this.fields.view, data?.views, restoredValues?.view, {
          key: '0',
          value: WRM.I18n.getText('structuremacro.view.default')
        });
        this.assignOptions(this.fields.savedfilter, data?.savedfilters, restoredValues?.savedfilter);
        if (restoredValues?.filtertype === 'saved' && !restoredValues?.savedfilter) {
          this.fields.filtertype.input.val('none');
          this.fields.filtertype.input.select2();
        }
        this.updateStructureName();

        this.fields.applink?.input?.prop('disabled', !data?.applinks);
        this.fields.savedfilter?.input?.prop('disabled', !data?.savedfilters);
        this.toggleField(this.fields.structure, data?.structures);
        this.toggleField(this.fields.view, data?.views);
        this.toggleField(this.fields.filtertype, data?.structures);
        this.toggleFilterFields(data?.structures && this.fields.filtertype?.input?.val());
      }
    }

    class GanttMacroFields extends MacroFields {
      updateFields(appLink, data, restoredValues) {
        this.assignOptions(this.fields.applink, data?.applinks, restoredValues?.applink || appLink);
        this.assignOptions(this.fields.structure, data?.structures, restoredValues?.structure);
        this.updateStructureName();

        this.fields.applink?.input?.prop('disabled', !data?.applinks);
        this.toggleField(this.fields.structure, data?.structures);
      }
    }

    const structureFields = new StructureMacroFields();
    mb.setMacroJsOverride('structuremacro', {
      fields: {
        enum: {
          applink: structureFields.setupAppLinksSelector('applink'),
          structure: structureFields.setupSelect('structure', field => {
            field.input.on('change', () => structureFields.updateStructureName());
          }),
          view: structureFields.setupSelect('view'),
          filtertype: structureFields.setupSelect('filtertype', field => {
            field.input.on('change', () => structureFields.toggleFilterFields(field.input.val()));
            setTimeout(() => structureFields.toggleFilterFields(field.getValue()));
            setTimeout(() => field.input.select2());
          }),
          savedfilter: structureFields.setupSelect('savedfilter')
        },
        string: {
          textfilter: structureFields.setupTextArea('textfilter'),
          jqlfilter: structureFields.setupTextArea('jqlfilter'),
          sjqlfilter: structureFields.setupTextArea('sjqlfilter'),
          title: structureFields.setupField('title', 'string'),
          width: structureFields.setupField('width', 'string'),
          height: structureFields.setupField('height', 'string'),
          structurename: structureFields.setupField0('structurename', '_hidden')
        },
        boolean: {
          editable: structureFields.setupField('editable', 'boolean')
        }
      }
    });

    const gantFields = new GanttMacroFields();
    mb.setMacroJsOverride('ganttmacro', {
      fields: {
        enum: {
          applink: gantFields.setupAppLinksSelector('applink'),
          structure: gantFields.setupSelect('structure', field => {
            field.input.on('change', () => gantFields.updateStructureName());
          })
        },
        string: {
          title: gantFields.setupField('title', 'string'),
          width: gantFields.setupField('width', 'string'),
          height: gantFields.setupField('height', 'string'),
          structurename: gantFields.setupField0('structurename', '_hidden')
        }
      }
    });
  });
});