var AJS = AJS || {};

/*
 * Functions for creating formatted strings from dates.
 */
AJS.ForumDateTimeFormatting = {

    /**
     * Converts minutes to milliseconds
     * @param {number} minutes
     * @returns {number}
     */
    minutesToMillis: function (minutes) {
        return minutes * 60 * 1000;
    },

    /**
     * timezone offset in milliseconds
     *
     * @returns {number}
     */
    timezoneOffsetMillis: function () {
        //the offset time needs to be inverted to work properly with moment functions used later by AJS.DateTimeFormatting
        return AJS.ForumDateTimeFormatting.minutesToMillis(-(new Date().getTimezoneOffset()));
    },

    /**
     * Format the date from {dateMillis} with the specified format dateFormat
     * @param {number} dateMillis
     * @param {string} dateFormat
     * @returns {string}
     */
    formatDate: function (dateMillis, dateFormat) {
        var dateToFormat = new Date(dateMillis);
        return moment(dateToFormat.valueOf()).local().format(dateFormat);
    },

    /**
     * Returns a string representing the date created from dateMillis
     * @param {number} dateMillis
     * @returns {string}
     */
    formatDateFriendly: function (dateMillis) {
        var now = new Date();
        var dateToFormat = new Date(dateMillis);
        var timezoneOffset = AJS.ForumDateTimeFormatting.timezoneOffsetMillis();
        return AJS.ForumDateTimeFormatting.friendlyFormatDateTime(dateToFormat, now, timezoneOffset);
    },

    /**
     * @param {number} dateMillis
     * @returns {string}
     */
    formatDateUK: function (dateMillis) {
        return AJS.ForumDateTimeFormatting.formatDate(dateMillis, "DD MMM YYYY HH:mm");
    },

    /**
     * @param {number} dateMillis
     * @returns {string}
     */
    formatDateUS: function (dateMillis) {
        return AJS.ForumDateTimeFormatting.formatDate(dateMillis, "MMM DD YYYY HH:mm");
    },

    /**
     * Returns date as a formatted string. Should produce results identical to the FriendlyDateFormatter Java class.
     */
    friendlyFormatDateTime: function(date, now, timezoneOffsetMillis) {
        var fourSeconds = 4 * 1000;
        var oneMinute = 60 * 1000;
        var twoMinutes = 2 * 60 * 1000;
        var fiftyMinutes = 50 * 60 * 1000;
        var ninetyMinutes = 90 * 60 * 1000;
        var fiveHours = 5 * 60 * 60 * 1000;
        var twentyFourHours = 24 * 60 * 60 * 1000;

        var timeSinceDate = now.valueOf() - date.valueOf();

        if (timeSinceDate < 0) {
            return AJS.ForumDateTimeFormatting.formatDateTime(date, timezoneOffsetMillis);
        } if (timeSinceDate === 0) {
            return AJS.I18n.getText('date.friendly.now');
        } if (timeSinceDate < fourSeconds) {
            return AJS.I18n.getText('date.friendly.a.moment.ago');
        } if (timeSinceDate < oneMinute) {
            return AJS.I18n.getText('less.than.one.min');
        } if (timeSinceDate < twoMinutes) {
            return AJS.I18n.getText('one.min.ago');
        } if (timeSinceDate < fiftyMinutes) {
            return AJS.I18n.getText('x.mins.ago', AJS.ForumDateTimeFormatting.wholeMinutesBetween(date, now));
        } if (timeSinceDate < ninetyMinutes) {
            return AJS.I18n.getText('date.friendly.about.one.hour.ago');
        } if (timeSinceDate > fiveHours && AJS.ForumDateTimeFormatting.isYesterdayRelativeTo(date, now, timezoneOffsetMillis)) {
            return AJS.I18n.getText('date.friendly.yesterday', AJS.ForumDateTimeFormatting.formatTime(date, timezoneOffsetMillis));
        } if (timeSinceDate < twentyFourHours) {
            return AJS.I18n.getText('date.friendly.about.x.hours.ago', AJS.ForumDateTimeFormatting.roundedHoursBetween(date, now));
        }

        return AJS.ForumDateTimeFormatting.formatDateWithTzOffset(date, timezoneOffsetMillis);
    },

    /**
     * Returns the date and time portions of date as a formatted string. For now, the format is hardcoded.
     */
    formatDateTime: function(date, timezoneOffsetMillis) {
        return moment(date.valueOf() + timezoneOffsetMillis).utc().format('MMM DD, YYYY HH:mm');
    },

    /**
     * Returns the date portion of date as a formatted string. For now, the format is hardcoded.
     */
    formatDateWithTzOffset: function(date, timezoneOffsetMillis) {
        return moment(date.valueOf() + timezoneOffsetMillis).utc().format('MMM DD, YYYY');
    },

    /**
     * Returns the number of minutes between from and to, rounded down.
     */
    wholeMinutesBetween: function(from, to) {
        var millisPerMinute = 60 * 1000;
        return Math.floor((to.valueOf() - from.valueOf()) / millisPerMinute);
    },

    /**
     * Returns true iff date is yesterday relative to now, in the timezone where local time is timezoneOffsetMillis milliseconds ahead of UTC.
     */
    isYesterdayRelativeTo: function(date, now, timezoneOffsetMillis) {
        var twentyFourHours = 24 * 60 * 60 * 1000;
        var dateInTimeZone = new Date(date.valueOf() + timezoneOffsetMillis);
        var yesterdayInTimeZone = new Date(now.valueOf() + timezoneOffsetMillis - twentyFourHours);
        return dateInTimeZone.getUTCFullYear() === yesterdayInTimeZone.getUTCFullYear()
            && dateInTimeZone.getUTCMonth() === yesterdayInTimeZone.getUTCMonth()
            && dateInTimeZone.getUTCDate() === yesterdayInTimeZone.getUTCDate();
    },

    /**
     * Returns the time portion of date as a formatted string. For now, the format is hardcoded.
     */
    formatTime: function(date, timezoneOffsetMillis) {
        return moment(date.valueOf() + timezoneOffsetMillis).utc().format('h:mm A');
    },

    /**
     * Returns the number of hours between from and to, rounded to nearest hour.
     */
    roundedHoursBetween: function(from, to) {
        var millisPerHour = 60 * 60 * 1000;
        return Math.round((to.valueOf() - from.valueOf()) / millisPerHour);
    }
};
