function initIssuesGanttChart({ containerId, parameters }) {
    function buildMermaidGantt(issues) {
        const tickInterval = getTickInterval(issues);
        let chart = [
            "---",
            "config:",
            "  gantt:",
            "    numberSectionStyles: 2",
            "    fontSize: 14",
            "    sectionFontSize: 14",
            "    leftPadding: 200",
            "    barHeight: 30",
            "---",
            "gantt",
            "  tickInterval " + tickInterval,
            "  dateFormat x",
            "  axisFormat %d/%m"
        ];

        issues.forEach(issue => {
            const tag = getGanttTag(issue);
            const title = issue.title.length > 30 ? issue.title.substring(0, 27) + "..." : issue.title;
            chart.push(`  section ${title}`);
            let end = issue.endDate ? `,${issue.endDate}` : "";
            chart.push(`    ${title} : ${tag},${issue.id},${issue.startDate}${end}`);
        });
        return "\n" + chart.join("\n") + "\n\n";
    }

    function getGanttTag(issue) {
        var tag = "";
        if(issue.overdue) {
            tag += "crit ";
            if(issue.state === "closed") {
                tag += ", done";
            }
        } else {
            if(issue.state === "closed") {
                tag += "done ";
            } else {
                tag += "active ";
            }
        }
        return tag.trim();
    }

    function addTooltips(issues, containerId) {
        // Create tooltip element
        const $tooltip = $('<div>').css({
            position: 'absolute',
            background: '#333',
            color: '#fff',
            padding: '8px 12px',
            borderRadius: '4px',
            pointerEvents: 'none',
            display: 'none',
            zIndex: 9999,
            fontSize: '12px',
            maxWidth: '300px',
            boxShadow: '0 2px 8px rgba(0,0,0,0.3)'
        }).appendTo('body');

        // Add tooltips to each task
        issues.forEach(issue => {
            const taskSelector = `${containerId} .mermaid rect[id*="${issue.id}"], ${containerId} .mermaid text[id="${issue.id}-text"]`;

            $(document).on('mouseenter', taskSelector, function(e) {
                const startDate = new Date(issue.startDate).toLocaleDateString();
                const closedDate = issue.closedDate ? new Date(issue.closedDate).toLocaleDateString() : '-';
                const overdueText = issue.overdue ? "⚠️ Overdue" : "";

                const tooltipContent = `
                <div><strong>Title:</strong> ${issue.title}</div>
                <div><strong>ID:</strong> ${issue.iid}</div>
                <div><strong>State:</strong> ${issue.state} ${overdueText}</div>
                <div><strong>Created At:</strong> ${startDate}</div>
                <div><strong>Closed At:</strong> ${closedDate}</div>
            `;

                $tooltip.html(tooltipContent).fadeIn(200);
            }).on('mousemove', taskSelector, function(e) {
                $tooltip.css({
                    top: e.pageY + 15 + 'px',
                    left: e.pageX + 15 + 'px'
                });
            }).on('mouseleave', taskSelector, function() {
                $tooltip.fadeOut(200);
            }).on("click", taskSelector, function() {
                window.open(issue.webUrl, "_blank");
            });
        });
    }

    function getTickInterval(issues) {
        if (!issues.length) return "1month";
        const startDates = issues.map(i => i.startDate);
        const endDates = issues.map(i => i.endDate || i.startDate);
        const minStart = Math.min(...startDates);
        const maxEnd = Math.max(...endDates);
        const diffDays = (maxEnd - minStart) / (1000 * 60 * 60 * 24);

        if (diffDays < 20) return "1day";
        if (diffDays < 120) return "1week";
        if (diffDays < 365) return "1month";
        return "1year";
    }

    $.ajax({
        url:  `${AJS.contextPath()}/rest/rumpelcoders/gitlab-api/1.0/projects/issues/chart`,
        type: "POST",
        dataType: "json",
        contentType: "application/json",
        data: JSON.stringify(parameters),
        async: true,
        error: function (xhr) {
            console.error(xhr.statusText + (": " + xhr.responseText) || "");
            $(containerId).text(xhr.statusText + (": " + xhr.responseText) || "");
        },
        success: function (response) {
            $(containerId).find(".loader").remove();
            const content = buildMermaidGantt(response);
            $(containerId).append(`<pre class="mermaid">${content}</pre>`);
            mermaid.run().then(() => {
                addTooltips(response, containerId);
                console.log("Mermaid chart rendered");
            });
        }
    });
}
