define('moveworkforward/gitlab/autoconvert', ['tinymce'], function (tinymce) {
    "use strict";

    let serverUrls = null;
    let autoconversionEnabled = true;
    const projectWithNameSpaceShift = 1;
    const projectNameShift = 2;
    const pathStartShift = 1;
    const projectIdShift = 1;
    const jobsPath = "jobs";
    const commitPath = "commit";
    const issuesPath = "issues";
    const mergeRequestsPath = "merge_requests";
    const blobPath = "blob";
    const tagsPath = "tags";
    const snippetsPath = "snippets";
    const milestonesPath = "milestones";

    let locale = $('meta[name="ajs-action-locale"]').attr("content");
    locale = locale ? locale : "en_En";
    console.log("GitLab conversion locale: " + locale);

    AJS.$.ajax({
        async: false,
        url: AJS.contextPath() + "/rest/rumpelcoders/gitlab-api/1.0/config/effective",
        dataType: 'json',
        timeout: 10000,
        error: function (xhr, textStatus, errorThrown) {
            AJS.logError(errorThrown);
            serverUrls = null;
            autoconversionEnabled = false;
        },
        success: function (response) {
            let setObj = new Set(response.map(value => value.serverUrl));
            serverUrls = Array.from(setObj);
            autoconversionEnabled = response.every(value => value.level !== "GLOBAL") || response.some(value => value.level === "GLOBAL" && value.autoconversionEnabled);
        }
    });

    function isNumeric(value) {
        return /^\d+$/.test(value);
    }

    function createMacroNode(hash, gitlabMacro, parameters, locale) {
        return AJS.$(`<img class="editor-inline-macro" 
                            src="${AJS.contextPath()}/plugins/servlet/confluence/placeholder/macro?definition=${hash}&amp;locale=${locale}&amp;version=2" 
                            data-macro-name="${gitlabMacro}"
                            data-macro-parameters="${parameters}"
                            data-macro-schema-version="1" />`);
    }


    function isAccepted(uri, keyPath) {
        let parts = uri.source.split('/');
        if (!serverUrls || !serverUrls.length > 0 || !serverUrls.some((serverUrl) => uri.source.startsWith(serverUrl))) {
            console.log("Server URL not found in the list of GitLab servers. Skipping macro creation.", uri.source);
            return false;
        }

        parts = uri.path.split('/');
        let keyIndex = parts.indexOf(keyPath);
        const result = keyIndex >= 2 && parts.length >= 3 && parts.indexOf('-') > 0;
        console.log(`GitLab autoconversion for ${keyPath} accepted: ${result}`);
        return result;
    }

    function isDefined(value) {
        return value && typeof value !== 'undefined' && value !== null && value !== '';
    }

    function processIssueMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(issuesPath);

        const issueIdShift = 1;

        if(keyIndex + issueIdShift < parts.length - 1) {
            console.log("Skipping GitLab issue macro creation.");
            done();
            return;
        }

        try {
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let issueId = keyIndex + issueIdShift == parts.length ? null : parts[keyIndex + issueIdShift];
            if(issueId && !isNumeric(issueId)) {
                console.log("Skipping GitLab issue macro creation. Id is not numeric.", issueId);
                done();
                return;
            }

            let gitlabMacro = issueId && issueId !== '' ? 'gitlab-api-issue' : 'gitlab-api-list-issues';

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            let parameters;
            let hash;
            console.log("GitLab Macro: " + gitlabMacro);
            if(gitlabMacro === 'gitlab-api-list-issues') {
                let state = uri.queryKey && uri.queryKey.state ? uri.queryKey.state : 'all';
                parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|maximumentries=10|orderby=CreatedAt|showDescription=true|sort=Descending|state=${state}|link=${encodeURIComponent(uri.source)}`;
                hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},maximumentries=10,orderby=CreatedAt,showDescription=true,sort=Descending,state=${state},link=${encodeURIComponent(uri.source)}}`);
            } else {
                parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|issue=${issueId}|showComments=true|showDescription=true|link=${encodeURIComponent(uri.source)}`;
                hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},issue=${issueId},showComments=true,showDescription=true,link=${encodeURIComponent(uri.source)}}`);
            }
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab issue macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processMilestonesMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(milestonesPath);

        const milestoneIdShift = 1;

        if(keyIndex + milestoneIdShift < parts.length - 1) {
            console.log("Skipping GitLab milestone macro creation.");
            done();
            return;
        }

        try {
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let id = keyIndex + milestoneIdShift == parts.length ? null : parts[keyIndex + milestoneIdShift];
            if(id && !isNumeric(id)) {
                console.log("Skipping GitLab milestone macro creation. Id is not numeric.", id);
                done();
                return;
            }

            let gitlabMacro = id && id !== '' ? 'gitlab-api-milestone' : 'gitlab-api-list-milestones';

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            let parameters;
            let hash;
            console.log("GitLab Macro: " + gitlabMacro);
            if(gitlabMacro === 'gitlab-api-list-milestones') {
                let state = uri.queryKey && uri.queryKey.state ? uri.queryKey.state : 'all';
                if(state === 'opened') {
                    state = 'active';
                }
                parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|maximumentries=10|state=${state}|link=${encodeURIComponent(uri.source)}`;
                hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},maximumentries=10,state=${state},link=${encodeURIComponent(uri.source)}}`);
            } else {
                parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|milestoneId=${id}|link=${encodeURIComponent(uri.source)}`;
                hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},milestoneId=${id},link=${encodeURIComponent(uri.source)}}`);
            }
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab milestone macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processCommitMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(commitPath);

        const commitHashShift = 1;

        if(keyIndex + commitHashShift < parts.length - 1) {
            console.log("Skipping GitLab commit macro creation.");
            done();
            return;
        }

        try {
            let gitlabMacro = 'gitlab-api-commit';
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let commitHash = parts[keyIndex + commitHashShift];

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace) || !isDefined(commitHash)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let parameters = `commitHash=${commitHash}|project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|showDiff=true|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:commitHash=${commitHash},project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},showDiff=true,link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab commit macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processBuildsMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(jobsPath);

        const buildIdShift = 1;

        if(keyIndex + buildIdShift < parts.length - 1) {
            console.log("Skipping GitLab builds macro creation.");
            done();
            return;
        }

        try {
            let gitlabMacro = 'gitlab-api-list-builds';
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let builId = keyIndex + buildIdShift == parts.length ? null : parts[keyIndex + buildIdShift];

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace) || isDefined(builId)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let parameters = `maxItemsShown=10|project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:maxItemsShown=10,project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab builds macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processMergeRequestsMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(mergeRequestsPath);

        const mergeRequestIdShift = 1;

        if(keyIndex + mergeRequestIdShift < parts.length - 1) {
            console.log("Skipping GitLab merge requests macro creation.");
            done();
            return;
        }

        try {
            let gitlabMacro = 'gitlab-api-list-mergerequests';
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let showClosed = uri.queryKey && uri.queryKey.state ? uri.queryKey.state === 'closed' : false;
            let showMerged = uri.queryKey && uri.queryKey.state ? uri.queryKey.state === 'merged' : false;
            let mergeRequestId = keyIndex + mergeRequestIdShift == parts.length ? null : parts[keyIndex + mergeRequestIdShift];

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace) || isDefined(mergeRequestId)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|showClosed=${showClosed}|showDescription=true|showMerged=${showMerged}|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},showClosed=${showClosed},showDescription=true,showMerged=${showMerged},link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab merge requests macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processTagMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(tagsPath);

        const tagIdShift = 1;

        if(keyIndex + tagIdShift < parts.length - 1) {
            console.log("Skipping GitLab tag macro creation.");
            done();
            return;
        }

        try {
            let gitlabMacro = 'gitlab-api-release';
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let tagId = keyIndex + tagIdShift == parts.length ? null : parts[keyIndex + tagIdShift];

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace) || !isDefined(tagId)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let parameters = `project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|showCommits=true|showMergeRequests=true|tag=${tagId}|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},showCommits=true,showMergeRequests=true,tag=${tagId},link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab tag macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processSnippetMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(snippetsPath);

        const snippetIdShift = 1;

        if(keyIndex + snippetIdShift < parts.length - 1) {
            console.log("Skipping GitLab snippet macro creation.");
            done();
            return;
        }

        try {
            let gitlabMacro = 'gitlab-api-snippet';
            let snippetId = keyIndex + snippetIdShift == parts.length ? null : parts[keyIndex + snippetIdShift];

            if (!isDefined(snippetId)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let parameters = `showDescription=true|snippetId=${snippetId}|useSnippetId=true|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:showDescription=true,snippetId=${snippetId}|useSnippetId=true,link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab snippet macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function processFileMacro(uri, node, done) {
        let parts = uri.path.split('/');
        let keyIndex = parts.indexOf(blobPath);

        const branchShift = 1;
        const fileNameShift = 2;

        try {
            let gitlabMacro = 'gitlab-api-file';
            let projectId = parts.slice(pathStartShift, keyIndex - projectIdShift).join("/");
            let projectName = parts[keyIndex - projectNameShift];
            let projectWithNameSpace = parts.slice(pathStartShift, keyIndex - projectWithNameSpaceShift).join(" / ");
            let fileName = parts.slice(keyIndex + fileNameShift).join("/");
            let branch = parts[keyIndex + branchShift];

            if (!isDefined(projectId) || !isDefined(projectName) || !isDefined(projectWithNameSpace) || !isDefined(fileName) || !isDefined(branch)) {
                console.warn("Some of the required parameters are not defined. Skipping macro creation.");
                done();
                return;
            }
            console.log("GitLab Macro: " + gitlabMacro);
            let fileType = fileName.toLocaleLowerCase().endsWith(".md") ? "Mark Down" : (fileName.toLocaleLowerCase().endsWith(".adoc") ? "Ascii" : "None");
            let parameters = `filePath=${fileName}|fileType=${fileType}|project=${encodeURIComponent(projectId)}|projectName=${projectName}|projectNameWithNamespace=${projectWithNameSpace}|reference=${branch}|showOnlyContent=true|link=${encodeURIComponent(uri.source)}`;
            let hash = btoa(`{${gitlabMacro}:filePath=${fileName},fileType=${fileType},project=${encodeURIComponent(projectId)},projectName=${projectName},projectNameWithNamespace=${projectWithNameSpace},reference=${branch},showOnlyContent=true},link=${encodeURIComponent(uri.source)}}`);
            let newNode = createMacroNode(hash, gitlabMacro, parameters, locale);
            console.log("GitLab file macro created.");
            done(newNode);
        } catch (error) {
            console.error("Error on processing link: " + error);
            done();
        }
    }

    function GitLabConverter() {
        function pasteHandler(uri, node, done) {
            let gitlabMacro;
            const start = new Date().getTime();
            try {
                if (!autoconversionEnabled || !serverUrls || serverUrls.length == 0 || node.text() !== uri.source) {
                    done();
                    console.log("GitLab autoconversion skipped");
                    return;
                }
                if (isAccepted(uri, blobPath)) {
                    processFileMacro(uri, node, done);
                } else if (isAccepted(uri, jobsPath)) {
                    processBuildsMacro(uri, node, done);
                } else if (isAccepted(uri, commitPath)) {
                    processCommitMacro(uri, node, done);
                } else if (isAccepted(uri, issuesPath)) {
                    processIssueMacro(uri, node, done);
                } else if (isAccepted(uri, mergeRequestsPath)) {
                    processMergeRequestsMacro(uri, node, done);
                } else if (isAccepted(uri, tagsPath)) {
                    processTagMacro(uri, node, done);
                } else if (isAccepted(uri, snippetsPath)) {
                    processSnippetMacro(uri, node, done);
                } else if (isAccepted(uri, milestonesPath)) {
                    processMilestonesMacro(uri, node, done);
                } else {
                    done();
                }
            } catch (error) {
                console.error("Error on processing URI: " + error);
                done();
            }
            console.log("GitLab autoconversion time:", (new Date().getTime() - start), "ms");
        }

        tinymce.plugins.Autoconvert.autoConvert.addHandler(pasteHandler);
    }

    return GitLabConverter;
});

require('confluence/module-exporter').safeRequire('moveworkforward/gitlab/autoconvert', function (GitLabConverter) {
    require('ajs').bind("init.rte", GitLabConverter);
});