/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.build.extractor.clientConfiguration.util;

import com.google.common.io.Files;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.SequenceInputStream;
import java.io.StringWriter;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.http.Header;
import org.jfrog.build.api.dependency.DownloadableArtifact;
import org.jfrog.build.api.dependency.pattern.PatternType;
import org.jfrog.build.api.search.AqlSearchResult;
import org.jfrog.build.api.util.FileChecksumCalculator;
import org.jfrog.build.api.util.Log;
import org.jfrog.build.api.util.ZipUtils;
import org.jfrog.build.extractor.builder.DependencyBuilder;
import org.jfrog.build.extractor.ci.Dependency;
import org.jfrog.build.extractor.clientConfiguration.client.artifactory.ArtifactoryManager;
import org.jfrog.build.extractor.clientConfiguration.util.ArtifactorySearcher;
import org.jfrog.build.extractor.clientConfiguration.util.DependenciesDownloader;
import org.jfrog.build.extractor.clientConfiguration.util.DependenciesDownloaderImpl;
import org.jfrog.build.extractor.clientConfiguration.util.PathsUtils;
import org.jfrog.filespecs.FileSpec;
import org.jfrog.filespecs.entities.FilesGroup;

public class DependenciesDownloaderHelper {
    private final DependenciesDownloader downloader;
    private final Log log;
    private static final int CONCURRENT_DOWNLOAD_THREADS = 3;
    public static final int MIN_SIZE_FOR_CONCURRENT_DOWNLOAD = 5120000;

    public DependenciesDownloaderHelper(DependenciesDownloader downloader, Log log) {
        this.downloader = downloader;
        this.log = log;
    }

    public DependenciesDownloaderHelper(ArtifactoryManager artifactoryManager, String workingDirectory, Log log) {
        this.downloader = new DependenciesDownloaderImpl(artifactoryManager, workingDirectory, log);
        this.log = log;
    }

    public List<Dependency> downloadDependencies(FileSpec downloadSpec) throws IOException {
        ArtifactorySearcher searcher = new ArtifactorySearcher(this.downloader.getArtifactoryManager(), this.log);
        HashSet<Dependency> resolvedDependencies = new HashSet<Dependency>();
        for (FilesGroup file : downloadSpec.getFiles()) {
            this.log.debug("Downloading dependencies using spec: \n" + file.toString());
            this.downloader.setFlatDownload(BooleanUtils.toBoolean((String)file.getFlat()));
            List<AqlSearchResult.SearchEntry> searchResults = searcher.SearchByFileSpec(file);
            Set<DownloadableArtifact> downloadableArtifacts = this.fetchDownloadableArtifactsFromResult(searchResults, Boolean.valueOf(file.getExplode()), file.getTarget());
            if (file.getSpecType() == FilesGroup.SpecType.PATTERN) {
                this.replaceTargetPlaceholders(file.getPattern(), downloadableArtifacts, file.getTarget());
            }
            resolvedDependencies.addAll(this.downloadDependencies(downloadableArtifacts));
        }
        return new ArrayList<Dependency>(resolvedDependencies);
    }

    private void replaceTargetPlaceholders(String searchPattern, Set<DownloadableArtifact> downloadableArtifacts, String target) {
        searchPattern = StringUtils.substringAfter((String)searchPattern, (String)"/");
        Pattern pattern = Pattern.compile(PathsUtils.pathToRegExp(searchPattern));
        target = (String)StringUtils.defaultIfEmpty((CharSequence)target, (CharSequence)"");
        for (DownloadableArtifact artifact : downloadableArtifacts) {
            if (StringUtils.isEmpty((CharSequence)target) || target.endsWith("/")) {
                artifact.setTargetDirPath(PathsUtils.reformatRegexp(artifact.getFilePath(), target, pattern));
                continue;
            }
            String targetAfterReplacement = PathsUtils.reformatRegexp(artifact.getFilePath(), target, pattern);
            Map<String, String> targetFileName = PathsUtils.replaceFilesName(targetAfterReplacement, artifact.getRelativeDirPath());
            artifact.setRelativeDirPath(targetFileName.get("srcPath"));
            artifact.setTargetDirPath(targetFileName.get("targetPath"));
        }
    }

    private Set<DownloadableArtifact> fetchDownloadableArtifactsFromResult(List<AqlSearchResult.SearchEntry> searchResults, boolean explode, String target) {
        HashSet<DownloadableArtifact> downloadableArtifacts = new HashSet<DownloadableArtifact>();
        for (AqlSearchResult.SearchEntry searchEntry : searchResults) {
            String path = searchEntry.getPath().equals(".") ? "" : searchEntry.getPath() + "/";
            DownloadableArtifact downloadableArtifact = new DownloadableArtifact(searchEntry.getRepo(), target, path + searchEntry.getName(), "", "", PatternType.NORMAL);
            downloadableArtifact.setExplode(explode);
            downloadableArtifacts.add(downloadableArtifact);
        }
        return downloadableArtifacts;
    }

    public List<Dependency> downloadDependencies(Set<DownloadableArtifact> downloadableArtifacts) throws IOException {
        this.log.info("Beginning to resolve Build Info published dependencies.");
        ArrayList<Dependency> dependencies = new ArrayList<Dependency>();
        HashSet<DownloadableArtifact> downloadedArtifacts = new HashSet<DownloadableArtifact>();
        for (DownloadableArtifact downloadableArtifact : downloadableArtifacts) {
            Dependency dependency = this.downloadArtifact(downloadableArtifact);
            if (dependency == null) continue;
            dependencies.add(dependency);
            downloadedArtifacts.add(downloadableArtifact);
            this.explodeDependenciesIfNeeded(downloadableArtifact);
        }
        this.removeUnusedArtifactsFromLocal(downloadedArtifacts);
        this.log.info("Finished resolving Build Info published dependencies.");
        return dependencies;
    }

    private void explodeDependenciesIfNeeded(DownloadableArtifact downloadableArtifact) throws IOException {
        if (!downloadableArtifact.isExplode()) {
            return;
        }
        String fileDestination = this.downloader.getTargetDir(downloadableArtifact.getTargetDirPath(), downloadableArtifact.getRelativeDirPath());
        this.log.info("Extracting Archive: " + fileDestination);
        File sourceArchive = new File(fileDestination);
        File parentFile = FileUtils.getFile((String[])new String[]{fileDestination}).getParentFile();
        ZipUtils.extract(sourceArchive, parentFile);
        this.log.info("Finished extracting archive to " + parentFile);
        this.log.debug("Deleting archive...");
        FileUtils.deleteQuietly((File)sourceArchive);
    }

    private void removeUnusedArtifactsFromLocal(Set<DownloadableArtifact> downloadableArtifacts) throws IOException {
        HashSet<String> forDeletionFiles = new HashSet<String>();
        HashSet<String> allResolvesFiles = new HashSet<String>();
        for (DownloadableArtifact downloadableArtifact : downloadableArtifacts) {
            String fileDestination = this.downloader.getTargetDir(downloadableArtifact.getTargetDirPath(), downloadableArtifact.getRelativeDirPath());
            allResolvesFiles.add(fileDestination);
            if (!PatternType.DELETE.equals((Object)downloadableArtifact.getPatternType())) continue;
            forDeletionFiles.add(fileDestination);
        }
        this.downloader.removeUnusedArtifactsFromLocal(allResolvesFiles, forDeletionFiles);
    }

    private Dependency downloadArtifact(DownloadableArtifact downloadableArtifact) throws IOException {
        String filePath = downloadableArtifact.getFilePath();
        String matrixParams = downloadableArtifact.getMatrixParameters();
        String uri = downloadableArtifact.getRepoUrl() + '/' + filePath;
        String uriWithParams = StringUtils.isBlank((CharSequence)matrixParams) ? uri : uri + ';' + matrixParams;
        ArtifactMetaData artifactMetaData = this.downloadArtifactMetaData(uriWithParams);
        if (StringUtils.isBlank((CharSequence)artifactMetaData.getMd5()) && StringUtils.isBlank((CharSequence)artifactMetaData.getSha1())) {
            return null;
        }
        return this.downloadArtifact(downloadableArtifact, artifactMetaData, uriWithParams, filePath);
    }

    Dependency downloadArtifact(DownloadableArtifact downloadableArtifact, ArtifactMetaData artifactMetaData, String uriWithParams, String filePath) throws IOException {
        String remotePath;
        String fileDestination = this.downloader.getTargetDir(downloadableArtifact.getTargetDirPath(), downloadableArtifact.getRelativeDirPath());
        Dependency dependencyResult = this.getDependencyLocally(artifactMetaData, fileDestination, remotePath = downloadableArtifact.getRepoUrl() + "/" + filePath);
        if (dependencyResult != null) {
            return dependencyResult;
        }
        try {
            Map<String, String> checksumsMap;
            this.log.info(String.format("Downloading '%s'...", uriWithParams));
            Map<String, String> map = checksumsMap = artifactMetaData.getSize() >= 5120000L && artifactMetaData.isAcceptRange() ? this.downloadFileConcurrently(uriWithParams, artifactMetaData.getSize(), fileDestination, filePath) : this.downloadFile(uriWithParams, fileDestination);
            if (checksumsMap == null) {
                throw new IOException("Received null checksums map for downloaded file.");
            }
            dependencyResult = this.validateChecksumsAndBuildDependency(checksumsMap, artifactMetaData, filePath, fileDestination, remotePath);
            this.log.info(String.format("Successfully downloaded '%s' to '%s'", uriWithParams, fileDestination));
            return dependencyResult;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    protected Map<String, String> downloadFile(String downloadPath, String fileDestination) throws IOException {
        File downloadedFile = this.downloader.getArtifactoryManager().downloadToFile(downloadPath, fileDestination);
        try {
            return FileChecksumCalculator.calculateChecksums(downloadedFile, "MD5", "SHA1", "SHA-256");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(String.format("Could not find checksum algorithm: %s", e.getLocalizedMessage()), e);
        }
    }

    /*
     * Loose catch block
     */
    protected Map<String, String> downloadFileConcurrently(String uriWithParams, long fileSize, String fileDestination, String filePath) throws Exception {
        File tempDir = Files.createTempDir();
        String tempPath = tempDir.getPath() + File.separatorChar + filePath;
        try {
            String[] downloadedFilesPaths = this.doConcurrentDownload(fileSize, uriWithParams, tempPath);
            try (InputStream inputStream = this.concatenateFilesToSingleStream(downloadedFilesPaths);){
                Map<String, String> map = this.downloader.saveDownloadedFile(inputStream, fileDestination);
                return map;
            }
            {
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            FileUtils.deleteDirectory((File)tempDir);
        }
    }

    private String[] doConcurrentDownload(long fileSize, String downloadPath, String tempPath) throws Exception {
        MutableBoolean errorOccurred = new MutableBoolean(false);
        String[] downloadedFilesPaths = new String[3];
        long chunkSize = fileSize / 3L;
        Thread[] workers = new Thread[3];
        long start = 0L;
        long end = chunkSize + fileSize % 3L - 1L;
        for (int i = 0; i < 3; ++i) {
            String fileDestination;
            HashMap<String, String> headers = new HashMap<String, String>();
            headers.put("Range", "bytes=" + start + "-" + end);
            downloadedFilesPaths[i] = fileDestination = tempPath + i;
            workers[i] = new Thread(() -> {
                try {
                    this.downloader.getArtifactoryManager().downloadToFile(downloadPath, fileDestination, headers);
                }
                catch (Exception e) {
                    errorOccurred.setValue(true);
                    this.printErrorToLog(e, fileDestination, downloadPath);
                }
            });
            workers[i].setName("downloader_" + i);
            workers[i].start();
            start = end + 1L;
            end += chunkSize;
        }
        for (Thread worker : workers) {
            worker.join();
        }
        if (errorOccurred.booleanValue()) {
            throw new Exception(String.format("Error occurred while downloading %s, please refer to logs for more information", downloadPath));
        }
        return downloadedFilesPaths;
    }

    private InputStream concatenateFilesToSingleStream(String[] downloadedFilesPaths) throws FileNotFoundException {
        InputStream inputStream = new FileInputStream(downloadedFilesPaths[0]);
        if (downloadedFilesPaths.length < 2) {
            return inputStream;
        }
        for (int i = 1; i < downloadedFilesPaths.length; ++i) {
            inputStream = new SequenceInputStream(inputStream, new FileInputStream(downloadedFilesPaths[i]));
        }
        return inputStream;
    }

    private Dependency getDependencyLocally(ArtifactMetaData fileMetaData, String localPath, String remotePath) throws IOException {
        if (this.downloader.isFileExistsLocally(localPath, fileMetaData.getMd5(), fileMetaData.getSha1())) {
            this.log.info(String.format("The file '%s' exists locally.", localPath));
            return new DependencyBuilder().md5(fileMetaData.getMd5()).sha1(fileMetaData.getSha1()).id(localPath.substring(localPath.lastIndexOf(String.valueOf(IOUtils.DIR_SEPARATOR)) + 1)).localPath(localPath).remotePath(remotePath).build();
        }
        return null;
    }

    protected ArtifactMetaData downloadArtifactMetaData(String url) throws IOException {
        try {
            ArtifactMetaData artifactMetaData = new ArtifactMetaData();
            for (Header header : this.downloader.getArtifactoryManager().downloadHeaders(url)) {
                String upperCaseHeader = StringUtils.upperCase((String)header.getName());
                if ("X-Checksum-Md5".toUpperCase().equals(upperCaseHeader)) {
                    artifactMetaData.setMd5(header.getValue());
                    continue;
                }
                if ("X-Checksum-Sha1".toUpperCase().equals(upperCaseHeader)) {
                    artifactMetaData.setSha1(header.getValue());
                    continue;
                }
                if ("Content-Length".toUpperCase().equals(upperCaseHeader)) {
                    artifactMetaData.setSize(NumberUtils.toLong((String)header.getValue()));
                    continue;
                }
                if (!"Accept-Ranges".toUpperCase().equals(upperCaseHeader)) continue;
                artifactMetaData.setAcceptRange("bytes".equalsIgnoreCase(header.getValue()));
            }
            return artifactMetaData;
        }
        catch (NumberFormatException e) {
            throw new IOException(e);
        }
    }

    private String validateMd5Checksum(String metadataMd5, String calculatedMd5) throws IOException {
        if (!StringUtils.equals((CharSequence)metadataMd5, (CharSequence)calculatedMd5)) {
            String errorMessage = String.format("Calculated MD5 checksum is different from original, Original: '%s' Calculated: '%s'", metadataMd5, calculatedMd5);
            throw new IOException(errorMessage);
        }
        return metadataMd5 == null ? "" : metadataMd5;
    }

    private String validateSha1Checksum(String metadataSha1, String calculatedSha1) throws IOException {
        if (!StringUtils.equals((CharSequence)metadataSha1, (CharSequence)calculatedSha1)) {
            String errorMessage = String.format("Calculated SHA-1 checksum is different from original, Original: '%s' Calculated: '%s'", metadataSha1, calculatedSha1);
            throw new IOException(errorMessage);
        }
        return metadataSha1 == null ? "" : metadataSha1;
    }

    private Dependency validateChecksumsAndBuildDependency(Map<String, String> checksumsMap, ArtifactMetaData artifactMetaData, String filePath, String fileDestination, String remotePath) throws IOException {
        String md5 = this.validateMd5Checksum(artifactMetaData.getMd5(), checksumsMap.get("MD5"));
        String sha1 = this.validateSha1Checksum(artifactMetaData.getSha1(), checksumsMap.get("SHA1"));
        return new DependencyBuilder().md5(md5).sha1(sha1).sha256(checksumsMap.get("SHA-256")).id(filePath.substring(filePath.lastIndexOf(File.separatorChar) + 1)).localPath(fileDestination).remotePath(remotePath).build();
    }

    public static File saveInputStreamToFile(InputStream inputStream, String filePath) throws IOException {
        File dest = new File(filePath);
        if (dest.exists()) {
            dest.delete();
        } else {
            dest.getParentFile().mkdirs();
        }
        FileOutputStream fileOutputStream = null;
        try {
            fileOutputStream = new FileOutputStream(dest);
            IOUtils.copyLarge((InputStream)inputStream, (OutputStream)fileOutputStream);
            File file = dest;
            return file;
        }
        catch (IOException e) {
            throw new IOException(String.format("Could not create or write to file: %s", dest.getCanonicalPath()), e);
        }
        finally {
            IOUtils.closeQuietly((InputStream)inputStream);
            IOUtils.closeQuietly((OutputStream)fileOutputStream);
        }
    }

    private void printErrorToLog(Exception e, String downloadPath, String uriWithParams) {
        StringWriter sw = new StringWriter();
        e.printStackTrace(new PrintWriter(sw));
        this.log.error(String.format("[Thread %s] downloading %s as part of file %s threw an exception: %s", Thread.currentThread().getName(), downloadPath, uriWithParams, sw.toString()));
    }

    protected static class ArtifactMetaData {
        private String sha256;
        private String sha1;
        private String md5;
        private long size;
        private boolean acceptRange;

        protected ArtifactMetaData() {
        }

        public long getSize() {
            return this.size;
        }

        public void setSize(long size) {
            this.size = size;
        }

        public String getSha256() {
            return this.sha256;
        }

        public void setSha256(String sha256) {
            this.sha256 = sha256;
        }

        public String getSha1() {
            return this.sha1;
        }

        public void setSha1(String sha1) {
            this.sha1 = sha1;
        }

        public String getMd5() {
            return this.md5;
        }

        public void setMd5(String md5) {
            this.md5 = md5;
        }

        public boolean isAcceptRange() {
            return this.acceptRange;
        }

        public void setAcceptRange(boolean acceptRange) {
            this.acceptRange = acceptRange;
        }
    }
}

