/*
 * Decompiled with CFR 0.152.
 */
package net.utoolity.atlassian.ifaws;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import net.utoolity.atlassian.dry.EncryptionFailureException;
import net.utoolity.atlassian.dry.PluginFileStorage;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class EncryptionHandler {
    private static final Logger log = LoggerFactory.getLogger(EncryptionHandler.class);
    private static final String KEY_FILE_EXTENSION = ".txt";
    private static final String KEY_STORAGE_DIR = "keys";
    private static final String CHARSET = "UTF-8";
    private static final String HASH_ALGORITHM = "SHA-1";
    private static final String KEY_ALGORITHM = "AES";
    private static final String CIPHER_ALGORITHM = "AES/CBC/PKCS5Padding";
    private static final int KEY_SIZE = 128;
    private static final int IV_SIZE = 16;
    private final SecureRandom random = new SecureRandom();
    private final PluginFileStorage pluginFileStorage;

    @Autowired
    public EncryptionHandler(PluginFileStorage pluginFileStorage) {
        this.pluginFileStorage = pluginFileStorage;
    }

    public byte[] encrypt(byte[] input) throws EncryptionFailureException {
        SecretKey key = this.getOrCreateSecretKey();
        IvParameterSpec ivParam = new IvParameterSpec(this.random.generateSeed(16));
        byte[] encryptedPayload = this.encryptPayload(input, key, ivParam);
        ByteArrayOutputStream outputStream = this.createStorageBundle(ivParam, key, encryptedPayload);
        return Base64.encodeBase64((byte[])outputStream.toByteArray());
    }

    public byte[] encrypt(String input) throws EncryptionFailureException {
        try {
            return this.encrypt(input.getBytes(CHARSET));
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to convert string to byte array";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
    }

    public byte[] decrypt(byte[] input) throws EncryptionFailureException {
        byte[] inputDecoded = Base64.decodeBase64((byte[])input);
        IvParameterSpec ivParam = this.extractIVFromStorageBundle(inputDecoded);
        SecretKey key = this.extractKeyFromStorageBundle(inputDecoded, ivParam.getIV());
        byte[] encrypted = this.extractPayloadFromStorageBundle(inputDecoded);
        byte[] output = this.decryptPayload(ivParam, key, encrypted);
        return output;
    }

    private IvParameterSpec extractIVFromStorageBundle(byte[] inputDecoded) {
        byte[] iv = Arrays.copyOfRange(inputDecoded, 0, 16);
        IvParameterSpec ivParam = new IvParameterSpec(iv);
        return ivParam;
    }

    private SecretKey extractKeyFromStorageBundle(byte[] inputDecoded, byte[] iv) throws EncryptionFailureException {
        String keyId;
        int nonPayloadSize = 16 + this.getKeyIdLength();
        byte[] keyIdBytes = Arrays.copyOfRange(inputDecoded, 16, nonPayloadSize);
        byte[] keyIdBytesDescrambled = this.descrambleKeyId(keyIdBytes, iv);
        try {
            keyId = new String(keyIdBytesDescrambled, CHARSET);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to convert byte array to string";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        SecretKey key = this.getSecretKey(keyId);
        return key;
    }

    private byte[] extractPayloadFromStorageBundle(byte[] inputDecoded) throws EncryptionFailureException {
        int nonPayloadSize = 16 + this.getKeyIdLength();
        byte[] encrypted = Arrays.copyOfRange(inputDecoded, nonPayloadSize, inputDecoded.length);
        return encrypted;
    }

    private byte[] decryptPayload(IvParameterSpec ivParam, SecretKey key, byte[] encrypted) throws EncryptionFailureException {
        byte[] output;
        Cipher cipher;
        String errorMessage = "Unable to decrypt payload";
        try {
            cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (NoSuchPaddingException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        try {
            cipher.init(2, (Key)key, ivParam);
        }
        catch (InvalidKeyException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (InvalidAlgorithmParameterException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        try {
            output = cipher.doFinal(encrypted);
        }
        catch (IllegalBlockSizeException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (BadPaddingException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        return output;
    }

    public String decryptToString(byte[] input) throws EncryptionFailureException {
        try {
            return new String(this.decrypt(input), CHARSET);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to convert byte array to string";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
    }

    public String decryptToString(String input) {
        try {
            byte[] inputBytes = input.getBytes(CHARSET);
            return this.decryptToString(inputBytes);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to convert byte array to string";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
    }

    private ByteArrayOutputStream createStorageBundle(IvParameterSpec ivParam, SecretKey key, byte[] encryptedPayload) throws EncryptionFailureException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        String errorMessage = "Unable to create storage bundle";
        try {
            outputStream.write(ivParam.getIV());
            outputStream.write(this.scrambleKeyId(key, ivParam.getIV()));
            outputStream.write(encryptedPayload);
        }
        catch (UnsupportedEncodingException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (IOException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        return outputStream;
    }

    private byte[] scrambleKeyId(SecretKey key, byte[] scrambleSource) throws UnsupportedEncodingException, EncryptionFailureException {
        byte[] keyIdBytes = this.getKeyId(key).getBytes(CHARSET);
        return this.xorByteArrays(keyIdBytes, scrambleSource);
    }

    private byte[] descrambleKeyId(byte[] keyIdBytes, byte[] scrambleSource) {
        return this.xorByteArrays(keyIdBytes, scrambleSource);
    }

    private byte[] xorByteArrays(byte[] sourceBytes, byte[] scrambleBytes) throws EncryptionFailureException {
        if (null == sourceBytes) {
            String errorMessage = "Source bytes can not be null";
            log.error(errorMessage);
            throw new EncryptionFailureException(errorMessage);
        }
        if (0 == sourceBytes.length) {
            String errorMessage = "Source bytes can not be of zero length";
            log.error(errorMessage);
            throw new EncryptionFailureException(errorMessage);
        }
        if (null == scrambleBytes) {
            String errorMessage = "Scramble bytes can not be null";
            log.error(errorMessage);
            throw new EncryptionFailureException(errorMessage);
        }
        if (0 == scrambleBytes.length) {
            String errorMessage = "Scramble bytes can not be of zero length";
            log.error(errorMessage);
            throw new EncryptionFailureException(errorMessage);
        }
        byte[] paddedScrambleBytes = this.padScrambleBytes(scrambleBytes, sourceBytes.length);
        byte[] scrambled = new byte[sourceBytes.length];
        for (int i = 0; i < scrambled.length; ++i) {
            scrambled[i] = (byte)(paddedScrambleBytes[i] ^ sourceBytes[i]);
        }
        return scrambled;
    }

    private byte[] padScrambleBytes(byte[] scrambleBytes, int desiredLength) {
        byte[] padded = new byte[desiredLength];
        if (desiredLength > scrambleBytes.length) {
            for (int targetIndex = 0; targetIndex < desiredLength; ++targetIndex) {
                int sourceIndex;
                for (sourceIndex = targetIndex; sourceIndex >= scrambleBytes.length; sourceIndex -= scrambleBytes.length) {
                }
                padded[targetIndex] = scrambleBytes[sourceIndex];
            }
        } else {
            padded = Arrays.copyOf(scrambleBytes, desiredLength);
        }
        return padded;
    }

    private byte[] encryptPayload(byte[] input, SecretKey key, IvParameterSpec ivParam) throws EncryptionFailureException {
        byte[] rawOutput;
        Cipher cipher;
        String errorMessage = "Unable to encrypt payload";
        try {
            cipher = Cipher.getInstance(CIPHER_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (NoSuchPaddingException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        try {
            cipher.init(1, (Key)key, ivParam);
        }
        catch (InvalidKeyException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (InvalidAlgorithmParameterException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        try {
            rawOutput = cipher.doFinal(input);
        }
        catch (IllegalBlockSizeException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        catch (BadPaddingException e) {
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        return rawOutput;
    }

    private SecretKey getOrCreateSecretKey() throws EncryptionFailureException {
        SecretKey key = this.tryFindExistingSecretKey();
        if (null == key) {
            key = this.createNewSecretKey();
        }
        return key;
    }

    private SecretKey tryFindExistingSecretKey() {
        String pattern = "*.txt";
        File[] matches = this.pluginFileStorage.findFiles(KEY_STORAGE_DIR, pattern);
        if (null != matches && 0 < matches.length) {
            File selected = null;
            for (int i = 0; i < matches.length; ++i) {
                if (null != selected && selected.lastModified() >= matches[i].lastModified()) continue;
                selected = matches[i];
            }
            String fileContent = null;
            try {
                fileContent = this.pluginFileStorage.readTextFile(selected);
            }
            catch (IOException e) {
                String errorMessage = "Unable to read encryption key storage file";
                log.error(errorMessage, (Throwable)e);
                throw new EncryptionFailureException(errorMessage, e);
            }
            SecretKey key = this.reconstructKeyFromFileContent(fileContent);
            return key;
        }
        return null;
    }

    private SecretKey createNewSecretKey() throws EncryptionFailureException {
        String keyStringified;
        KeyGenerator generator;
        try {
            generator = KeyGenerator.getInstance(KEY_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            String errorMessage = "Unable to generate encryption key";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        generator.init(128);
        SecretKey key = generator.generateKey();
        byte[] keyEncoded = key.getEncoded();
        try {
            keyStringified = new String(Base64.encodeBase64((byte[])keyEncoded), CHARSET);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to stringify encryption key";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        String keyId = this.getKeyId(key);
        String filename = this.getKeyFilename(keyId);
        try {
            this.pluginFileStorage.ensureDirectoryExists(KEY_STORAGE_DIR);
            this.pluginFileStorage.writeTextFile(filename, keyStringified);
        }
        catch (IOException e) {
            String errorMessage = "Unable to write encryption key file";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        return key;
    }

    private SecretKey getSecretKey(String keyId) throws EncryptionFailureException {
        String filename = this.getKeyFilename(keyId);
        String fileContent = null;
        try {
            fileContent = this.pluginFileStorage.readTextFile(filename);
        }
        catch (IOException e) {
            String errorMessage = "Unable to read encryption key file";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        SecretKey key = this.reconstructKeyFromFileContent(fileContent);
        return key;
    }

    private SecretKey reconstructKeyFromFileContent(String fileContent) throws EncryptionFailureException {
        byte[] keyStringified = null;
        try {
            keyStringified = fileContent.getBytes(CHARSET);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Unable to convert byte array to string";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        byte[] keyEncoded = Base64.decodeBase64((byte[])keyStringified);
        SecretKeySpec key = new SecretKeySpec(keyEncoded, KEY_ALGORITHM);
        return key;
    }

    private String getKeyId(SecretKey key) throws EncryptionFailureException {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance(HASH_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            String errorMessage = "Unable to create encryption key id";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        byte[] keyEncoded = messageDigest.digest(key.getEncoded());
        String keyId = new String(Hex.encodeHex((byte[])keyEncoded));
        return keyId;
    }

    private int getKeyIdLength() throws EncryptionFailureException {
        MessageDigest messageDigest;
        try {
            messageDigest = MessageDigest.getInstance(HASH_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            String errorMessage = "Unable to determine encryption key id length";
            log.error(errorMessage, (Throwable)e);
            throw new EncryptionFailureException(errorMessage, e);
        }
        return messageDigest.getDigestLength() * 2;
    }

    private String getKeyFilename(String keyId) {
        String filename = KEY_STORAGE_DIR + File.separatorChar + keyId + KEY_FILE_EXTENSION;
        return filename;
    }

    public String getCharset() {
        return CHARSET;
    }
}

