/*
 * Decompiled with CFR 0.152.
 */
package com.resolution.atlasplugins.samlsso.oidcandoauth.oidc;

import com.auth0.jwk.InvalidPublicKeyException;
import com.auth0.jwk.Jwk;
import com.auth0.jwk.JwkException;
import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.auth0.jwt.interfaces.Verification;
import com.resolution.atlasplugins.samlsso.configuration.RunningConfiguration;
import com.resolution.atlasplugins.samlsso.configuration.oidc.OidcRunningIdpConfiguration;
import com.resolution.atlasplugins.samlsso.tracker.AuthenticationTracker;
import com.resolution.samlwrapper.api.tracker.SAMLAuthenticationTracker;
import de.resolution.commons.data.StructuredData;
import java.nio.charset.StandardCharsets;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Base64;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IdTokenHelper {
    private static final Logger logger = LoggerFactory.getLogger(IdTokenHelper.class);
    private static final String RS256 = "RS256";
    private static final String EC256 = "EC256";
    private static final String RS512 = "RS512";

    private IdTokenHelper() {
    }

    @Nullable
    public static DecodedJWT decodeAndVerifyIdToken(@Nonnull String idToken, @Nonnull OidcRunningIdpConfiguration oidcConfig, @Nonnull AuthenticationTracker tracker, @Nonnull RunningConfiguration config) {
        DecodedJWT decodedButNOTverifiedJWT;
        try {
            decodedButNOTverifiedJWT = JWT.decode((String)idToken);
        }
        catch (JWTDecodeException e) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Could not decode the ID token. Cannot continue.", new Object[0]);
            tracker.setStatus(SAMLAuthenticationTracker.Status.FAILED);
            return null;
        }
        String algo = decodedButNOTverifiedJWT.getAlgorithm();
        if (algo == null) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Could not read algorithm from id_token. Cannot continue.", new Object[0]);
            return null;
        }
        if ((algo = algo.trim()).isEmpty()) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Algorithm from id_token is empty. Cannot continue.", new Object[0]);
            return null;
        }
        tracker.add(logger, SAMLAuthenticationTracker.Level.DEBUG, "Algo from id_token is {}", new Object[]{algo});
        String kid = decodedButNOTverifiedJWT.getKeyId();
        if (kid == null) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Cannot find a kid in id_token. Cannot continue.", new Object[0]);
            return null;
        }
        if ((kid = kid.trim()).isEmpty()) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Kid in id_token is empty. Cannot continue.", new Object[0]);
            return null;
        }
        tracker.add(logger, SAMLAuthenticationTracker.Level.DEBUG, "Kid from id_token is {}", new Object[]{kid});
        return IdTokenHelper.verifyIdToken(idToken, oidcConfig, tracker, algo, kid, config.getClockSkew());
    }

    @Nullable
    private static DecodedJWT verifyIdToken(@Nonnull String idToken, @Nonnull OidcRunningIdpConfiguration config, @Nonnull AuthenticationTracker tracker, @Nonnull String alg, @Nonnull String kid, int clockSkew) {
        if (clockSkew < 0) {
            logger.error("clockskew is not valid");
            return null;
        }
        PublicKey publicKey = IdTokenHelper.getPublicKey(tracker, config, kid);
        if (publicKey == null) {
            return null;
        }
        Algorithm algorithm = IdTokenHelper.checkAndRetrieveAlgorithm(tracker, alg, publicKey, config);
        if (algorithm == null) {
            return null;
        }
        try {
            Verification verification = JWT.require((Algorithm)algorithm).withIssuer(config.getIssuer()).withAudience(new String[]{config.getClientId()}).acceptLeeway((long)clockSkew);
            if (config.sendAndVerifyNonce()) {
                verification.withClaim("nonce", tracker.getNonce());
            }
            DecodedJWT decodedAndVerified = verification.build().verify(idToken);
            tracker.add(logger, SAMLAuthenticationTracker.Level.DEBUG, "Found a valid signature.", new Object[0]);
            return decodedAndVerified;
        }
        catch (JWTVerificationException e) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Could not validate signature of id token because:\n{}", new Object[]{e.getMessage()});
            return null;
        }
    }

    @Nullable
    private static PublicKey getPublicKey(@Nonnull AuthenticationTracker tracker, @Nonnull OidcRunningIdpConfiguration config, @Nonnull String kid) {
        Jwk key;
        try {
            key = config.getKeyFromKeyStore(kid);
        }
        catch (JwkException e) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Could not retrieve public key from key store because {}", new Object[]{e.getMessage()});
            return null;
        }
        if (key == null) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Cannot find a public key for kid={} . Cannot proceed", new Object[]{kid});
            return null;
        }
        tracker.add(logger, SAMLAuthenticationTracker.Level.DEBUG, "Found a public key", new Object[0]);
        try {
            return key.getPublicKey();
        }
        catch (InvalidPublicKeyException e) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Something went terribly wrong with loading the public key", new Object[]{e});
            return null;
        }
    }

    @Nullable
    private static Algorithm checkAndRetrieveAlgorithm(@Nonnull AuthenticationTracker tracker, @Nonnull String alg, @Nonnull PublicKey publicKey, @Nonnull OidcRunningIdpConfiguration config) {
        if (!config.getSupportedSigningAlgorithms().contains(alg)) {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "The sign Alg from id_token is not given via the metadata. alg={}", new Object[]{alg});
            return null;
        }
        Algorithm algorithm = null;
        if (alg.equals(RS256)) {
            algorithm = Algorithm.RSA256((RSAPublicKey)((RSAPublicKey)publicKey), null);
        } else if (alg.equals(EC256)) {
            algorithm = Algorithm.ECDSA256((ECPublicKey)((ECPublicKey)publicKey), null);
        } else if (alg.equals(RS512)) {
            algorithm = Algorithm.RSA512((RSAPublicKey)((RSAPublicKey)publicKey), null);
        } else {
            tracker.add(logger, SAMLAuthenticationTracker.Level.ERROR, "Given alg is currently not supported. Please contact our support and include this message. Found alg={}", new Object[]{alg});
        }
        return algorithm;
    }

    @Nonnull
    public static StructuredData extractClaimsFromIdToken(@Nullable DecodedJWT decodedToken) {
        if (decodedToken == null) {
            return StructuredData.create();
        }
        String str = new String(Base64.getUrlDecoder().decode(decodedToken.getPayload()), StandardCharsets.UTF_8);
        if (logger.isDebugEnabled()) {
            logger.debug("DecodedJWT as Json:\n{}", (Object)str);
        }
        return StructuredData.parseJson((String)str);
    }
}

