/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server.handler;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.eclipse.jetty.http.HttpException;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.http.QuotedQualityCSV;
import org.eclipse.jetty.io.ByteBufferOutputStream;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.util.Attributes;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.ExceptionUtil;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class ErrorHandler
implements Request.Handler {
    private static final Logger LOG = LoggerFactory.getLogger(ErrorHandler.class);
    public static final String ERROR_STATUS = "org.eclipse.jetty.server.error_status";
    public static final String ERROR_MESSAGE = "org.eclipse.jetty.server.error_message";
    public static final String ERROR_EXCEPTION = "org.eclipse.jetty.server.error_exception";
    public static final String ERROR_CONTEXT = "org.eclipse.jetty.server.error_context";
    public static final Set<String> ERROR_METHODS = Set.of("GET", "POST", "HEAD");
    public static final HttpField ERROR_CACHE_CONTROL = new PreEncodedHttpField(HttpHeader.CACHE_CONTROL, "must-revalidate,no-cache,no-store");
    boolean _showStacks = false;
    boolean _showCauses = false;
    boolean _showMessageInTitle = true;
    int _bufferSize = -1;
    String _defaultResponseMimeType = MimeTypes.Type.TEXT_HTML.asString();
    HttpField _cacheControl = new PreEncodedHttpField(HttpHeader.CACHE_CONTROL, "must-revalidate,no-cache,no-store");

    public boolean errorPageForMethod(String method) {
        return ERROR_METHODS.contains(method);
    }

    @Override
    public boolean handle(Request request, Response response, Callback callback) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("handle({}, {}, {})", new Object[]{request, response, callback});
        }
        if (this._cacheControl != null) {
            response.getHeaders().put(this._cacheControl);
        }
        int code = response.getStatus();
        String message = (String)request.getAttribute(ERROR_MESSAGE);
        Throwable cause = (Throwable)request.getAttribute(ERROR_EXCEPTION);
        if (cause instanceof HttpException) {
            HttpException httpException = (HttpException)cause;
            code = httpException.getCode();
            response.setStatus(code);
            if (message == null) {
                message = httpException.getReason();
            }
        }
        if (!this.errorPageForMethod(request.getMethod()) || HttpStatus.hasNoBody((int)code)) {
            callback.succeeded();
        } else {
            if (message == null) {
                message = cause == null ? HttpStatus.getMessage((int)code) : cause.toString();
            }
            this.generateResponse(request, response, code, message, cause, callback);
        }
        return true;
    }

    protected void generateResponse(Request request, Response response, int code, String message, Throwable cause, Callback callback) throws IOException {
        List<Charset> charsets;
        List<String> acceptable = request.getHeaders().getQualityCSV(HttpHeader.ACCEPT, QuotedQualityCSV.MOST_SPECIFIC_MIME_ORDERING);
        if (acceptable.isEmpty()) {
            if (request.getHeaders().contains(HttpHeader.ACCEPT)) {
                callback.succeeded();
                return;
            }
            acceptable = Collections.singletonList(this._defaultResponseMimeType);
        }
        if ((charsets = request.getHeaders().getQualityCSV(HttpHeader.ACCEPT_CHARSET).stream().map(s -> {
            try {
                if ("*".equals(s)) {
                    return StandardCharsets.UTF_8;
                }
                return Charset.forName(s);
            }
            catch (Throwable t) {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList())).isEmpty()) {
            charsets = List.of(StandardCharsets.ISO_8859_1, StandardCharsets.UTF_8);
            if (request.getHeaders().contains(HttpHeader.ACCEPT_CHARSET)) {
                callback.succeeded();
                return;
            }
        }
        for (String mimeType : acceptable) {
            if (!this.generateAcceptableResponse(request, response, callback, mimeType, charsets, code, message, cause)) continue;
            return;
        }
        callback.succeeded();
    }

    protected boolean generateAcceptableResponse(Request request, Response response, Callback callback, String contentType, List<Charset> charsets, int code, String message, Throwable cause) throws IOException {
        Charset charset;
        MimeTypes.Type type;
        switch (contentType) {
            case "text/html": 
            case "text/*": 
            case "*/*": {
                type = MimeTypes.Type.TEXT_HTML;
                charset = charsets.stream().findFirst().orElse(StandardCharsets.ISO_8859_1);
                break;
            }
            case "text/json": 
            case "application/json": {
                if (charsets.contains(StandardCharsets.UTF_8)) {
                    charset = StandardCharsets.UTF_8;
                } else if (charsets.contains(StandardCharsets.ISO_8859_1)) {
                    charset = StandardCharsets.ISO_8859_1;
                } else {
                    return false;
                }
                type = MimeTypes.Type.TEXT_JSON.is(contentType) ? MimeTypes.Type.TEXT_JSON : MimeTypes.Type.APPLICATION_JSON;
                break;
            }
            case "text/plain": {
                type = MimeTypes.Type.TEXT_PLAIN;
                charset = charsets.stream().findFirst().orElse(StandardCharsets.ISO_8859_1);
                break;
            }
            default: {
                return false;
            }
        }
        int bufferSize = this.getBufferSize() <= 0 ? this.computeBufferSize(request) : this.getBufferSize();
        ByteBufferPool byteBufferPool = request.getComponents().getByteBufferPool();
        RetainableByteBuffer buffer = byteBufferPool.acquire(bufferSize, false);
        try {
            boolean showStacks = this.isShowStacks();
            while (true) {
                try {
                    buffer.clear();
                    ByteBufferOutputStream out = new ByteBufferOutputStream(buffer.getByteBuffer());
                    PrintWriter writer = new PrintWriter(new OutputStreamWriter((OutputStream)out, charset));
                    switch (type) {
                        case TEXT_HTML: {
                            this.writeErrorHtml(request, writer, charset, code, message, cause, showStacks);
                            break;
                        }
                        case TEXT_JSON: 
                        case APPLICATION_JSON: {
                            this.writeErrorJson(request, writer, code, message, cause, showStacks);
                            break;
                        }
                        case TEXT_PLAIN: {
                            this.writeErrorPlain(request, writer, code, message, cause, showStacks);
                            break;
                        }
                        default: {
                            throw new IllegalStateException();
                        }
                    }
                    writer.flush();
                }
                catch (BufferOverflowException e) {
                    if (showStacks) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Disable stacks for " + String.valueOf(e));
                        }
                        showStacks = false;
                        continue;
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.warn("Error page too large: >{} {} {} {}", new Object[]{bufferSize, code, message, request, e});
                        break;
                    }
                    LOG.warn("Error page too large: >{} {} {} {}", new Object[]{bufferSize, code, message, request});
                }
                break;
            }
            if (!buffer.hasRemaining()) {
                buffer.release();
                callback.succeeded();
                return true;
            }
            response.getHeaders().put(type.getContentTypeField(charset));
            response.write(true, buffer.getByteBuffer(), new WriteErrorCallback(callback, byteBufferPool, buffer));
            return true;
        }
        catch (Throwable x) {
            if (buffer != null) {
                byteBufferPool.removeAndRelease(buffer);
            }
            throw x;
        }
    }

    protected int computeBufferSize(Request request) {
        int bufferSize = request.getConnectionMetaData().getHttpConfiguration().getOutputBufferSize();
        bufferSize = Math.min(8192, bufferSize);
        return bufferSize;
    }

    protected void writeErrorHtml(Request request, Writer writer, Charset charset, int code, String message, Throwable cause, boolean showStacks) throws IOException {
        if (message == null) {
            message = HttpStatus.getMessage((int)code);
        }
        writer.write("<html>\n<head>\n");
        this.writeErrorHtmlMeta(request, writer, charset);
        this.writeErrorHtmlHead(request, writer, code, message);
        writer.write("</head>\n<body>\n");
        this.writeErrorHtmlBody(request, writer, code, message, cause, showStacks);
        writer.write("\n</body>\n</html>\n");
    }

    protected void writeErrorHtmlMeta(Request request, Writer writer, Charset charset) throws IOException {
        writer.write("<meta http-equiv=\"Content-Type\" content=\"text/html;charset=");
        writer.write(charset.name());
        writer.write("\"/>\n");
    }

    protected void writeErrorHtmlHead(Request request, Writer writer, int code, String message) throws IOException {
        writer.write("<title>Error ");
        String status = Integer.toString(code);
        writer.write(status);
        if (this.isShowMessageInTitle() && message != null && !message.equals(status)) {
            writer.write(32);
            writer.write(StringUtil.sanitizeXmlString(message));
        }
        writer.write("</title>\n");
    }

    protected void writeErrorHtmlBody(Request request, Writer writer, int code, String message, Throwable cause, boolean showStacks) throws IOException {
        String uri = request.getHttpURI().toString();
        this.writeErrorHtmlMessage(request, writer, code, message, cause, uri);
        if (showStacks) {
            this.writeErrorHtmlStacks(request, writer);
        }
        request.getConnectionMetaData().getHttpConfiguration().writePoweredBy(writer, "<hr/>", "<hr/>\n");
    }

    protected void writeErrorHtmlMessage(Request request, Writer writer, int code, String message, Throwable cause, String uri) throws IOException {
        writer.write("<h2>HTTP ERROR ");
        String status = Integer.toString(code);
        writer.write(status);
        if (message != null && !message.equals(status)) {
            writer.write(32);
            writer.write(StringUtil.sanitizeXmlString(message));
        }
        writer.write("</h2>\n");
        writer.write("<table>\n");
        this.htmlRow(writer, "URI", uri);
        this.htmlRow(writer, "STATUS", status);
        this.htmlRow(writer, "MESSAGE", message);
        while (this._showCauses && cause != null) {
            this.htmlRow(writer, "CAUSED BY", cause);
            cause = cause.getCause();
        }
        writer.write("</table>\n");
    }

    private void htmlRow(Writer writer, String tag, Object value) throws IOException {
        writer.write("<tr><th>");
        writer.write(tag);
        writer.write(":</th><td>");
        if (value == null) {
            writer.write("-");
        } else {
            writer.write(StringUtil.sanitizeXmlString(value.toString()));
        }
        writer.write("</td></tr>\n");
    }

    protected void writeErrorPlain(Request request, PrintWriter writer, int code, String message, Throwable cause, boolean showStacks) {
        writer.write("HTTP ERROR ");
        writer.write(Integer.toString(code));
        if (this.isShowMessageInTitle()) {
            writer.write(32);
            writer.write(StringUtil.sanitizeXmlString(message));
        }
        writer.write("\n");
        writer.printf("URI: %s%n", request.getHttpURI());
        writer.printf("STATUS: %s%n", code);
        writer.printf("MESSAGE: %s%n", message);
        while (this._showCauses && cause != null) {
            writer.printf("CAUSED BY %s%n", cause);
            if (showStacks) {
                cause.printStackTrace(writer);
            }
            cause = cause.getCause();
        }
    }

    protected void writeErrorJson(Request request, PrintWriter writer, int code, String message, Throwable cause, boolean showStacks) {
        HashMap<Object, String> json = new HashMap<Object, String>();
        json.put("url", request.getHttpURI().toString());
        json.put("status", Integer.toString(code));
        json.put("message", message);
        int c = 0;
        while (this._showCauses && cause != null) {
            json.put("cause" + c++, cause.toString());
            cause = cause.getCause();
        }
        writer.append(json.entrySet().stream().map(e -> HttpField.NAME_VALUE_TOKENIZER.quote((String)e.getKey()) + ":" + HttpField.NAME_VALUE_TOKENIZER.quote(StringUtil.sanitizeXmlString((String)e.getValue()))).collect(Collectors.joining(",\n", "{\n", "\n}")));
    }

    protected void writeErrorHtmlStacks(Request request, Writer writer) throws IOException {
        Throwable th = (Throwable)request.getAttribute(ERROR_EXCEPTION);
        if (th != null) {
            writer.write("<h3>Caused by:</h3><pre>");
            try (StringWriter sw = new StringWriter();
                 PrintWriter pw = new PrintWriter(sw);){
                th.printStackTrace(pw);
                pw.flush();
                this.write(writer, sw.getBuffer().toString());
            }
            writer.write("</pre>\n");
        }
    }

    @Deprecated(since="12.0.8", forRemoval=true)
    public ByteBuffer badMessageError(int status, String reason, HttpFields.Mutable fields) {
        return null;
    }

    @ManagedAttribute(value="The value of the Cache-Control response header")
    public String getCacheControl() {
        return this._cacheControl == null ? null : this._cacheControl.getValue();
    }

    public void setCacheControl(String cacheControl) {
        this._cacheControl = cacheControl == null ? null : new PreEncodedHttpField(HttpHeader.CACHE_CONTROL, cacheControl);
    }

    @ManagedAttribute(value="Whether the error page shows the stack trace")
    public boolean isShowStacks() {
        return this._showStacks;
    }

    public void setShowStacks(boolean showStacks) {
        this._showStacks = showStacks;
    }

    @ManagedAttribute(value="Whether the error page shows the exception causes")
    public boolean isShowCauses() {
        return this._showCauses;
    }

    public void setShowCauses(boolean showCauses) {
        this._showCauses = showCauses;
    }

    @ManagedAttribute(value="Whether the error message is shown in the error page title")
    public boolean isShowMessageInTitle() {
        return this._showMessageInTitle;
    }

    public void setShowMessageInTitle(boolean showMessageInTitle) {
        this._showMessageInTitle = showMessageInTitle;
    }

    @ManagedAttribute(value="Mime type to be used when a client does not specify an Accept header, or the request did not fully parse")
    public String getDefaultResponseMimeType() {
        return this._defaultResponseMimeType;
    }

    public void setDefaultResponseMimeType(String defaultResponseMimeType) {
        this._defaultResponseMimeType = Objects.requireNonNull(defaultResponseMimeType);
    }

    protected void write(Writer writer, String string) throws IOException {
        if (string == null) {
            return;
        }
        writer.write(StringUtil.sanitizeXmlString(string));
    }

    public static Request.Handler getErrorHandler(Server server, ContextHandler context) {
        Request.Handler errorHandler = null;
        if (context != null) {
            errorHandler = context.getErrorHandler();
        }
        if (errorHandler == null && server != null) {
            errorHandler = server.getErrorHandler();
        }
        return errorHandler;
    }

    @ManagedAttribute(value="Buffer size for entire error response")
    public int getBufferSize() {
        return this._bufferSize;
    }

    public void setBufferSize(int bufferSize) {
        this._bufferSize = bufferSize;
    }

    private static class WriteErrorCallback
    implements Callback {
        private final AtomicReference<Callback> _callback;
        private final ByteBufferPool _pool;
        private final RetainableByteBuffer _buffer;

        public WriteErrorCallback(Callback callback, ByteBufferPool pool, RetainableByteBuffer retainable) {
            this._callback = new AtomicReference<Callback>(callback);
            this._pool = pool;
            this._buffer = retainable;
        }

        @Override
        public void succeeded() {
            Callback callback = this._callback.getAndSet(null);
            if (callback != null) {
                ExceptionUtil.callAndThen(this._buffer::release, callback::succeeded);
            }
        }

        @Override
        public void failed(Throwable x) {
            Callback callback = this._callback.getAndSet(null);
            if (callback != null) {
                ExceptionUtil.callAndThen(x, t -> this._pool.removeAndRelease(this._buffer), callback::failed);
            }
        }
    }

    public static class ErrorRequest
    extends Request.AttributesWrapper {
        private static final Set<String> ATTRIBUTES = Set.of("org.eclipse.jetty.server.error_message", "org.eclipse.jetty.server.error_exception", "org.eclipse.jetty.server.error_status");

        public ErrorRequest(Request request, final int status, final String message, final Throwable cause) {
            super(request, new Attributes.Synthetic(request){

                @Override
                protected Object getSyntheticAttribute(String name) {
                    return switch (name) {
                        case ErrorHandler.ERROR_MESSAGE -> message;
                        case ErrorHandler.ERROR_EXCEPTION -> cause;
                        case ErrorHandler.ERROR_STATUS -> status;
                        default -> null;
                    };
                }

                @Override
                protected Set<String> getSyntheticNameSet() {
                    return ATTRIBUTES;
                }
            });
        }

        @Override
        public Content.Chunk read() {
            return Content.Chunk.EOF;
        }

        @Override
        public void demand(Runnable demandCallback) {
            demandCallback.run();
        }

        @Override
        public String toString() {
            return "%s@%x:%s".formatted(this.getClass().getSimpleName(), this.hashCode(), this.getWrapped());
        }
    }
}

