require([
    "jquery",
    "troubleshooting/ajs",
    "aui/flag",
], function ($, AJS, flag) {
    var user = $('meta[name=ajs-remote-user]').attr('content');
    var settingsUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user;
    var severityUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user + '/notification-severity';
    var watchUrl = AJS.contextPath() + '/rest/troubleshooting/1.0/user-setting/' + user + '/watch';

    AJS.toInit(function () {
        var containerDeferred = $.Deferred();
        var preferencesDeferred = $.getJSON(settingsUrl);

        // Load the notification config panel once the troubleshootingContainerLoaded event is triggered
        // AND the UserPreferences REST call has returned.
        AJS.bind("troubleshootingContainerLoaded", containerDeferred.resolve);

        $.when(containerDeferred, preferencesDeferred).done(renderConfigPanel);
    });

    // containerResult is just used for padding with the $.when callback.
    function renderConfigPanel(containerResult, [preferences]) {
        var {severityThresholdForNotifications} = preferences;
        var notificationConfig = $(Support.HealthCheck.Templates.notificationConfig(preferences));

        var watchCheckbox = notificationConfig.find('#notification-watch');
        var severitySelect = notificationConfig.find('select#notification-settings');

        watchCheckbox.change(updateWatch);
        severitySelect.val(severityThresholdForNotifications.toLowerCase());
        severitySelect.change(updateSeverity);
        $('.troubleshooting-status-container').after(notificationConfig);
    }

    // Make a REST call to watch/unwatch for current user
    function updateWatch() {
        const checked = $(this).is(":checked");
        $.ajax({
            url: watchUrl,
            type: checked ? 'PUT' : 'DELETE',
            contentType: 'application/json'
        })
        .fail(function (e) {
            console.error('Something went wrong, could not update watch status', e);
            showErrorFlag();
        });

        AJS.trigger('analyticsEvent', {
            name: `healthcheck.notification.${checked ? 'watch' : 'unwatch'}`,
        });
    }

    // Make a REST call to enable or disable notification for current user
    function updateSeverity() {

        // Select option corresponds to Severity API Model enum - pass straight through to backend.
        var notificationSeverity = $(this).val();
        $.ajax({
            url: severityUrl,
            type: 'PUT',
            data: JSON.stringify(notificationSeverity),
            contentType: 'application/json'
        })
            .done(function () {
                // NOOP
            })
            .fail(function (e) {
                console.error('Something went wrong, could not save the notification preference', e);
                showErrorFlag();
            });

        // Trigger Atlassian Analytics event after user updated their settings
        // Available options: [undefined, major, critical], critical is for disabling all notification
        AJS.trigger('analyticsEvent', {
            name: "healthcheck.notification.config.update." + notificationSeverity,
            data: {severity: notificationSeverity}
        });
    }

    function showErrorFlag() {
        flag({type: 'error', body: AJS.I18n.getText('stp.health.settings.update.error') })
    }
});