define("troubleshooting/detect-issues/main", [
    "underscore",
    "troubleshooting/detect-issues/config",
    "troubleshooting/detect-issues/helper"
], function (_, Config, Helper) {
    'use strict';

    // Responsible for:
    // - displaying the spinner while the check is still running on the background
    // - get the current completed health check run from /check/process/{processId}/results endpoint
    // - call the displayCheckResults function to dynamically render the checks that has been completed
    var loadCheckResults = function (processReport, $container) {
        // Display the health check status container using jQuery's fadeIn() function
        $container.fadeIn("slow");

        Helper.addSpinner($container);

        // Get the process ID from response process report
        var processId = processReport.processId;
        // URL for retrieving the healthcheck results from background process
        var url = Config.instanceHealthProcessUrl + processId + "/results";

        AJS.$.ajax({
            url: url,
            dataType: "json"
        }).done(function(healthCheckResults) {
            displayCheckResults(processReport, healthCheckResults, $container);
            expandHealthCheckStatus($container, document.location.search);
        });
    };

    // What this function does:
    // - dynamically render the checks that has been completed
    // - validate if all check has been performed
    // -- If all checks has been performed, remove the spinner from page and trigger analytics
    // -- If there are checks that still runs on background, run loadCheckResults function to get the latest results every 500 miliseconds
    var displayCheckResults = function (processReport, healthCheckResults, $container) {
        // Get the health check representations from process report
        var healthChecks = processReport.healthCheckRepresentations;
        // Get the health check statuses from check results
        var statuses = healthCheckResults.statuses;

        // Group the status by isHealthy attribute
      var groups = _.groupBy(statuses, function (item) {
        if (!item.isEnabled) {
          return 'disabled';
        }
        if (item.isSoftLaunch) {
          return 'soft';
        }
        if (item.isHealthy) {
          return 'pass';
        } else {
          return 'fail';
        }
      });

        // If check results status length == health check representations length, it means that all checks run has been completed
        if (statuses.length === healthChecks.length) {
            // Remove the spinner container, we no longer need it on the page
            Helper.removeSpinner();

            // Trigger analytics event for each of healthcheck
            statuses.forEach(function (check) {
                Helper.sendAnalytics(check, '.' + (check.isHealthy ? 'pass' : 'fail') + '.done');
            });

            if (groups.fail) {
                Helper.displayBadge(groups.fail.length);
            }
        } else {
            // Else, we check the healthcheck background process again in 500 milliseconds
            setTimeout(function () {
                loadCheckResults(processReport, $container);
            }, 500);
        }

        if (groups.fail) {
            // We sort the failed results by priority, display the major error first before warn
            var failSorted = _.sortBy(groups.fail, Helper.getSeverity).reverse();
            // Then we group the failed checks by its tags
            var failGroups = Helper.groupByTags(failSorted);
            addResultsToTemplate("failed", failGroups);
        }

        if (groups.pass) {
            // We group the passed checks by its tags
            var healthyGroups = Helper.groupByTags(groups.pass);
            addResultsToTemplate("passed", healthyGroups);
        }

      if (groups.disabled) {
        // We group the disabled checks by its tags
        var disabledGroups = Helper.groupByTags(groups.disabled);
        addResultsToTemplate("disabled", disabledGroups);
      }
    };

    // Validates if a given check result in each groups has been displayed on the results container, else add the check results content to the container
    var addResultsToTemplate = function (passFail, checkGroups) {
        // Iterate over each of the health check groups
        checkGroups.forEach(function (group) {
            // Get the tag name, change the character casing to lower case, and replace space character with a "-"
            // The tag name will be used as the ID of certain elements inside the health check results container
            var groupId = 'hc-group-' + group[0].tag.toLowerCase().split(" ").join("-");

            // Get either the passed or failed check container, depending on where this function was called
            var $checkContainer = AJS.$("#" + passFail + "-check-container");
            var $statusGroupList = $checkContainer.find(".troubleshooting-status-group-list");

            // If the status group list doesn't exists, add it to the template
            if (!$statusGroupList.find("#" + groupId).length) {
                $statusGroupList.append(Stp.HealthCheck.Dynamic.Templates.healthStatusGroup({
                    groupId: groupId,
                    status: group[0]
                }));
            }

            // Iterate over each of the health check status in the group
            group.forEach(function (status) {
                var groupContentId = groupId + '-content';
                var $statusList = $statusGroupList.find("#" + groupId);

                // Get the status container, if it doesn't exists we add it to the template
                if (!$statusList.find("#" + groupContentId).length) {
                    $statusList.append(Stp.HealthCheck.Dynamic.Templates.healthStatusList({
                        groupId: groupContentId
                    }));
                }

                var statusId = status.completeKey.split(':').pop();
                var statusKey = status.completeKey;
                var $list = $statusList.find("#" + groupContentId);

                // If the check is not displayed on the page, we add it to the template
                if (!$list.find("#" + statusId).length) {
                    var status = AJS.$(Stp.HealthCheck.Dynamic.Templates.healthStatus({
                        statusId: statusId,
                        status: status
                    }));
                    AJS.$(status).find('.health-status-documentation-link').click(function (evt) {
                        Helper.sendAnalytics(this.dataset, '.clicked');
                    });
                  $list.append(status);
                  var toggle = $list.find("#health-check-enable-" + statusId);
                  toggle.on('change', function (e) {
                    var isChecked = e.target.checked;     // new value of the toggle
                    toggle.busy = true;
                    AJS.$.ajax({
                      type: "POST",
                      url: Config.enableHealthcheckUrl,
                      contentType: "application/json",
                      dataType: "json",
                      data: JSON.stringify({healthCheckKey: statusKey, enabled: isChecked})
                    })
                      .fail(function () {
                        toggle.checked = !isChecked;
                      })
                      .always(function () {
                        toggle.busy = false;
                      });
                  });
                }
            });

            // The passed/failed check container is hidden by default, hence if it's hidden here, we display it using the fadeIn() function
            if (!$checkContainer.is(":visible")) {
                $checkContainer.fadeIn("slow");
            }
        });
    };

    var expandHealthCheckStatus = function($container, queryString) {
      var searchQuery = parseQueryString(queryString);
      var healthCheckId = searchQuery && searchQuery.healthCheck;
      var $healthCheck = $container.find('#' + healthCheckId);

      if (!healthCheckId || !$healthCheck.length) {
          return;
      }

      var $healthCheckRow = $healthCheck.find('.troubleshooting-status-row');
      var $groupContainer = $healthCheck.closest('.troubleshooting-status-group');
      var $groupHeader = $groupContainer.find('.troubleshooting-status-group-header');

      var isHealthCheckExpanded = $healthCheck.hasClass('expanded');
      var isHealthCheckGroupExpanded = $groupContainer.hasClass('expanded-group');

      if (!isHealthCheckGroupExpanded) {
        $groupHeader.trigger('click');
      }

      if (!isHealthCheckExpanded) {
        $healthCheckRow.trigger('click');
        $healthCheck.addClass('highlighted');
      }
    };

    var parseQueryString = function(input) {
        if (!input) {
            return null;
        }

        var parts = input.replace(/^[?&]/, '')
            .split('&')
            .map(function(values) {
                return values.split('=');
            })
            .reduce(function(result, values) {
                result[values[0]] = values[1];

                return result;
            }, {});

        return parts;
    };

    return {
        initInstanceHealth: function () {
            // Get the health status container, and set the onclick event on the expandable child elements first
            var $container = AJS.$(".troubleshooting-status-container");
            Helper.attachHealthcheckItemClickEvent($container);

            // populate the Instance health description events
            AJS.$('#instance-health-check-link').click(function (e) {
                AJS.trigger('analyticsEvent', {name: 'stp.instance.health.description'});
            });

            AJS.$('#instance-health-check-link').attr('href', 'https://confluence.atlassian.com/support/instance-health-790796828.html#InstanceHealth-WhatHealthChecksareavailable?');

            var checkProcess = AJS.$.ajax({
                type: "POST",
                url: Config.instanceHealthProcessUrl,
                contentType: "application/json",
                dataType: "json"
            }).done(function (processReport) {
                // Trigger the following event, which is listened by a script in Support Healthcheck Plugin to render the "Last Run" div
                AJS.trigger("troubleshootingContainerLoaded");
                loadCheckResults(processReport, $container);
            });
        }
    };
});
