/* global AJS */

/**
 * Helper class for instance health, which exposes various functions for manipulating the result group, adding elements to UI, and trigger analytic events
 */
define("troubleshooting/detect-issues/helper", [
    "jquery"
], function ($) {
    // These integer values should align with those in the com.atlassian.troubleshooting.healthcheck.SupportHealthStatus.Severity
    // Java enum.
    var numericSeverity = {
        "undefined": 0,
        "minor": 1,
        "warning": 2,
        "major": 3,
        "critical": 4,
        "disabled": 5
    };

    // Analytics Check name for i18n support.
    function getI18nCheckName(completeKey, checkName) {
        var i18nNameMap = {
            "openFilesHealthCheck": "openFiles",
            "sqlModeCheck": "sqlMode",
            "licenseHealthCheck": "license",
            "endOfLifeHealthCheck": "endOfLife", // the new cross product eol health check
            "tomcatHealthCheck": "tomcat",
            "hsqlHealthCheck": "hsql",
            "jdkHealthCheck": "jdk",
            "luceneSupportHealthCheck": "lucene",
            "eolHealthCheck": "eol", // this is the Confluence eol health check
            "collationCheck": "collation",
            "maxAllowedPacketsCheck": "maxpackets",
            "characterSetCheck": "charset",
            "innoDBLogFileSizeCheck": "innodb",
            "activeObjectsSupportHealthCheck": "ao",
            "storageIndexSnapshotHealthCheck": "freeSpace",
            "attachmentHealthCheck": "attachments",
            "attachmentCustomPathHealthCheck": "attachmentCustomPath",
            "clusterIndexReplicationHealthCheck": "clusterIndexRepl",
            "clusterLockHealthCheck": "clusterLocks",
            "schedulerHealthCheck": "clusterScheduler",
            "clusterReplicationHealthCheck": "clusterCacheRepl",
            "clusterNodeTimeSynchronizationHealthCheck": "clusterTimeSync",
            "sharedHomeHealthCheck": "sharedHome",
            "secondaryAttachmentHealthCheck": "secondaryAttachments",
            "secondaryAvatarHealthCheck": "secondaryAvatars",
            "secondaryPluginHealthCheck": "secondaryPlugins",
            "secondaryIndexHealthCheck": "secondaryIndexSnapshots",
            "databaseConfigCheck": "databaseConfigCheck",
            "internalAdminCheck": "internalAdmin",
            "isolationHealthCheck": "isolation",
            "collabEditingModeCheck": "collabEditingMode",
            "gadgetFeedUrlHealthCheck": "gadgetFeedUrl",
            "gcHealthCheck": "gc",
            "threadLimitHealthCheck": "threadLimit",
            "ldapAuthenticationCheck": "ldapAuthentication",
            "crowdAuthenticationCheck": "crowdAuthentication",
            "supportedDatabaseHealthCheck": "supportedDatabase",
            "networkMountHealthCheck": "networkMount",
            "applinksStatusHealthCheck": "applinksStatus",
            "storageEngineCheck": "storageengine",
            "jTDSDriverUpgradeCheck": "jTDSDriverUpgrade",
            "fontHealthCheck": "font",
            "securityVulnerabilityCheck": "securityVulnerability",
        };

        // legacy checkname maps for older versions of SHC.
        var checkNameMap = {
            // Confluence
            "Server SQL Mode": "sqlMode",
            "HSQL database": "hsql",
            "License Expiry": "license",
            "Java Version and Vendor": "jdk",
            "Application Server Version and Vendor": "tomcat",
            "Max Allowed Packet": "maxpackets",
            "Character Set": "charset",
            "Collation": "collation",
            "Storage Engine": "storageengine",
            "InnoDB Log File Size": "innodb",
            "Internal Admin Check": "internalAdmin",
            "Collaborative Editing Mode": "collabEditingMode",
            "Garbage Collection Pause Times": "gc",
            "Thread Limit": "threadLimit",
            "Deprecated jTDS driver": "jTDSDriverUpgrade",

            //Both
            "Open Files Limit": "openFiles",

            // JIRA
            "Secondary Index Snapshots": "secondaryIndexSnapshots",
            "Secondary Plugins": "secondaryPlugins",
            "Secondary Avatars": "secondaryAvatars",
            "Secondary Attachments": "secondaryAttachments",
            "Shared Home": "sharedHome",
            "Cluster Time Synchronisation": "clusterTimeSync",
            "Cluster Cache Replication": "clusterCacheRepl",
            "Cluster Scheduler": "clusterScheduler",
            "Cluster Locks": "clusterLocks",
            "Cluster Index Replication": "clusterIndexRepl",
            "Attachments": "attachments",
            "Free space": "freeSpace",
            "Active Objects": "ao",
            "Indexing": "indexing",
            "Lucene": "lucene",
            "HSQL": "hsql",
            "Embedded database": "embeddedDb",
            "End Of Life": "eol",
            "Isolation level": "isolation",
            "Gadget feed URL": "gadgetFeedUrl",
            "LDAP Authentication": "ldapAuthentication",
            "Crowd Authentication": "crowdAuthentication",
            "Supported database": "supportedDatabase",
            "Application Links": "applinksStatus",
            "Lucene index files location": "networkMount"
        };

        // get the i18n or fail-over to the checkMapName
        if (!(typeof completeKey === 'undefined')) {
            if (completeKey != "undefined") {
                return i18nNameMap[completeKey.split(":").splice(-1)[0]];
            }
        }

        return checkNameMap[checkName];
    }

    var removeHealthCheckHighlighting = function () {
        $('.troubleshooting-status-container')
            .find('.highlighted')
            .removeClass('highlighted');
    };

    return {
        attachHealthcheckItemClickEvent: function ($container) {
            $container.on('click', '.troubleshooting-status-row', function () {
                $(this).closest('.troubleshooting-status-item').toggleClass('expanded');
                removeHealthCheckHighlighting();
            });
            $container.on('click', '.troubleshooting-status-group-header', function () {
                $(this).closest('.troubleshooting-status-group').toggleClass('expanded-group');
                removeHealthCheckHighlighting();
            });
        },
        displayBadge: function (failedSize) {
            // Inject the number with a span into the Instance Health tab if there's unhealthy checks failing:
            $('li.menu-item:contains(Instance health) strong').append("&nbsp;<span class=\"aui-badge\">" + failedSize + "</span>");
        },
        getSeverity: function (status) {
            return numericSeverity[status.severity];
        },
        groupByTags: function (statuses) {
            return _.chain(statuses)
                .groupBy(function (item) {
                    return item.tag;
                }).toArray().value()
        },
        handleError: function () {
            $(".troubleshooting-status-container").append(Stp.HealthCheck.Templates.errorMessage());
        },
        removeSpinner: function () {
            // Thanks to https://bugs.jquery.com/ticket/9310 we use hide instead of fadeOut as loading instance health
            // then going to another tab and back can cause the spinner to never go away.
            $(".health-status-spinner-container").hide();
        },
        addSpinner: function ($container) {
            // Add the spinner to the instance health div container if the spinner doesn't exists yet
            // Do nothing if the spinner is already spinning on the UI
            if (!$(".health-status-spinner-container").length) {
                $container.before(Stp.HealthCheck.Dynamic.Templates.loadingSpinner());
                $container.append(Stp.HealthCheck.Dynamic.Templates.healthcheckContainer());
            }
        },
        // This function is used to trigger healthcheck documentation click action, and is embedded in the health check status template
        // See the soy template for detect issues to see how it is implemented
        sendAnalytics: function (opts, postfix) {
            "use strict";
            var healthCheckName = getI18nCheckName(opts.completeKey, opts.name);

            // don't bother collect useless statistics.
            if (healthCheckName) {
                AJS.trigger('analyticsEvent', {
                    name: ('healthchecks.check.' + healthCheckName + postfix),
                    data: {
                        severity: numericSeverity[opts.severity],
                        isHealthy: opts.isHealthy,
                        failureReason: opts.failureReason
                    }
                });
            }
        }
    };
});
