/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.tunnel.process;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.tunnel.analytics.TunnelClientAnalyticsEventPublisher;
import com.atlassian.tunnel.analytics.TunnelUnauthorizedAnalyticsEvent;
import com.atlassian.tunnel.configuration.exception.MissingTunnelException;
import com.atlassian.tunnel.file.InletsExecutable;
import com.atlassian.tunnel.file.InletsExecutableFactory;
import com.atlassian.tunnel.process.InletsClientProcessFactory;
import com.atlassian.tunnel.process.InletsTokenFileManager;
import com.atlassian.tunnel.process.TunnelClientProcessStateManager;
import com.atlassian.tunnel.process.TunnelExecutorServiceProvider;
import com.atlassian.tunnel.process.TunnelProcessState;
import com.atlassian.tunnel.process.TunnelUnauthorizedReason;
import com.atlassian.tunnel.process.retry.BackoffHandler;
import com.atlassian.tunnel.properties.TunnelProperties;
import com.atlassian.tunnel.security.ConfidentialDataObfuscator;
import com.atlassian.utils.process.ExternalProcess;
import com.atlassian.utils.process.LineOutputHandler;
import com.atlassian.utils.process.ProcessException;
import com.atlassian.utils.process.ProcessMonitor;
import com.google.common.base.Preconditions;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

public class InletsClientManager
implements DisposableBean,
ProcessMonitor {
    public static final int MAX_UNAUTHORIZED_RETRIES = 2000;
    private static final Logger log = LoggerFactory.getLogger(InletsClientManager.class);
    private final InletsExecutableFactory inletsExecutableFactory;
    private final InletsClientProcessFactory inletsClientProcessFactory;
    private final TunnelClientProcessStateManager tunnelClientProcessStateManager;
    private final TunnelProperties tunnelProperties;
    private final ConfidentialDataObfuscator confidentialDataObfuscator;
    private final InletsTokenFileManager inletsTokenFileManager;
    private final ExecutorService executorService;
    private final AtomicReference<ExternalProcess> process = new AtomicReference();
    private final AtomicBoolean destroyed = new AtomicBoolean();
    private final TunnelClientAnalyticsEventPublisher tunnelClientAnalyticsEventPublisher;
    private InletsExecutable executable;

    public InletsClientManager(InletsExecutableFactory inletsExecutableFactory, InletsClientProcessFactory inletsClientProcessFactory, TunnelClientProcessStateManager tunnelClientProcessStateManager, TunnelProperties tunnelProperties, ConfidentialDataObfuscator confidentialDataObfuscator, InletsTokenFileManager inletsTokenFileManager, TunnelExecutorServiceProvider tunnelExecutorServiceProvider, TunnelClientAnalyticsEventPublisher tunnelClientAnalyticsEventPublisher) {
        this.inletsExecutableFactory = inletsExecutableFactory;
        this.inletsClientProcessFactory = inletsClientProcessFactory;
        this.tunnelClientProcessStateManager = tunnelClientProcessStateManager;
        this.tunnelProperties = tunnelProperties;
        this.confidentialDataObfuscator = confidentialDataObfuscator;
        this.inletsTokenFileManager = inletsTokenFileManager;
        this.executorService = tunnelExecutorServiceProvider.getExecutorService();
        this.tunnelClientAnalyticsEventPublisher = tunnelClientAnalyticsEventPublisher;
    }

    public void start() {
        log.info("Starting Secure Tunnel Client...");
        this.startInternal(true, TunnelProcessState.Status.STARTING);
    }

    public void stop() {
        this.stopInternal(false);
    }

    private void stopInternal(boolean calledFromStart) {
        ExternalProcess processToCancel = this.process.getAndSet(null);
        if (processToCancel != null) {
            try {
                if (calledFromStart) {
                    log.warn("There is a Tunnel Process already running - stopping it before starting a new one");
                } else {
                    log.info("Stopping Secure Tunnel Client...");
                }
                this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.TERMINATING));
            }
            finally {
                processToCancel.cancel();
                this.reset(null);
            }
        }
    }

    public void destroy() {
        if (this.destroyed.compareAndSet(false, true)) {
            this.destroyInternal();
        }
    }

    private void destroyInternal() {
        this.stopInternal(false);
        if (this.executable != null) {
            this.executable.delete();
            this.executable = null;
        }
    }

    @VisibleForTesting
    protected void startInternal(boolean stopProcess, TunnelProcessState.Status status) {
        Preconditions.checkState((!this.destroyed.get() ? 1 : 0) != 0);
        try {
            while ((this.process.get() != null || !this.tryStartProcess(status)) && stopProcess) {
                this.stopInternal(true);
            }
        }
        finally {
            if (this.destroyed.get()) {
                this.destroyInternal();
            }
        }
    }

    private synchronized boolean tryStartProcess(TunnelProcessState.Status status) {
        if (this.process.get() != null) {
            return false;
        }
        if (status != null) {
            this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(status));
        }
        this.startProcess();
        return true;
    }

    private synchronized void startProcess() {
        Preconditions.checkState((this.process.get() == null ? 1 : 0) != 0);
        ExternalProcess process = null;
        try {
            this.executable = this.inletsExecutableFactory.create();
            process = this.inletsClientProcessFactory.create(this.executable, this, new InletsClientOutputHandler());
            this.process.set(process);
            this.tunnelClientProcessStateManager.setErrorCode(0);
            process.start();
            this.executorService.submit(process::finish);
            this.waitForStartDelay();
            this.inletsTokenFileManager.deleteTokenFile();
            if (this.process.get() == process && process.isAlive()) {
                this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.RUNNING).resetRetryCount());
                log.debug("Tunnel is connected.");
            }
        }
        catch (MissingTunnelException e) {
            log.info("Secure Tunnel is not configured, setup skipped.");
            this.reset(process);
        }
        catch (Exception e) {
            log.warn("Failed to setup Secure Tunnel, turn on debug for stack trace: {}", (Object)e.getMessage());
            log.debug("", (Throwable)e);
            this.reset(process);
        }
    }

    private void reset(ExternalProcess process) {
        if (this.process.compareAndSet(process, null)) {
            this.tunnelClientProcessStateManager.setErrorCode(0);
            this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.NOT_RUNNING).resetRetryCount());
        }
    }

    private void waitForStartDelay() {
        this.sleep(this.tunnelProperties.getStartWaitTime(), "Waiting for tunnel process to start was interrupted");
    }

    @Override
    public void onBeforeStart(ExternalProcess process) {
        log.trace("Tunnel Process Monitor status on Before Start: {}", (Object)this.tunnelClientProcessStateManager.getState().getStatus());
    }

    @Override
    public void onAfterFinished(ExternalProcess process) {
        boolean cleared = this.process.compareAndSet(process, null);
        TunnelProcessState.Status currentStatus = this.tunnelClientProcessStateManager.getState().getStatus();
        log.trace("Tunnel Process Monitor status on After Finished: {} cleared = {}", (Object)currentStatus, (Object)cleared);
        ProcessException exception = process.getHandler().getException();
        if (exception != null) {
            this.confidentialDataObfuscator.logError(log, "Error running inlets process", exception);
        }
        if (cleared && !process.isCanceled()) {
            if (this.tunnelClientProcessStateManager.getErrorCode() == Response.Status.UNAUTHORIZED.getStatusCode()) {
                log.error("Secure Tunnel Process failed authentication");
                this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.AUTHENTICATION_FAILED));
            }
            this.scheduleRestartIfNeeded();
        }
    }

    private void scheduleRestartIfNeeded() {
        boolean isUnauthorized = this.tunnelClientProcessStateManager.getErrorCode() == Response.Status.UNAUTHORIZED.getStatusCode();
        TunnelProcessState state = this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.RESTARTING).incrementRetryCount());
        if (isUnauthorized) {
            state = this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.RESTARTING).incrementUnauthorizedRetryCount());
        } else if (state.getUnuathorizedRetryCount() > 0) {
            this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.RESTARTING).resetUnauthorizedRetryCount());
        }
        if (state.getRetryCount() > this.tunnelProperties.getMaxRetries()) {
            log.error("Maximum number of tunnel start retries reached {}", (Object)this.tunnelProperties.getMaxRetries());
            return;
        }
        if (isUnauthorized && state.getUnuathorizedRetryCount() > 2000) {
            log.error("Maximum number of tunnel start retries for unauthorized state reached");
            this.tunnelClientProcessStateManager.updateState(currentState -> currentState.withStatus(TunnelProcessState.Status.AUTHENTICATION_FAILED));
            return;
        }
        log.debug("Secure Tunnel Process terminated unexpectedly. Restarting... (retry count: {})", (Object)state.getRetryCount());
        long pauseTimeMs = BackoffHandler.determineWaitTime(state.getRetryCount(), isUnauthorized);
        long endTime = System.currentTimeMillis() + pauseTimeMs;
        this.executorService.submit(() -> this.restartAfterPause(endTime));
    }

    private void restartAfterPause(long endTime) {
        if (this.process.get() == null) {
            long remaining = endTime - System.currentTimeMillis();
            if (remaining > 0L) {
                log.debug("Pausing for {} ms", (Object)remaining);
                this.sleep(remaining, "Tunnel connection backoff interrupted");
            }
            this.startInternal(false, null);
        } else {
            log.info("Skipping restart, because new process was started in the meantime.");
        }
    }

    void sleep(long millis, String message) {
        try {
            Thread.sleep(millis);
        }
        catch (InterruptedException exception) {
            log.debug("{}: {}", (Object)message, (Object)exception.getMessage());
            Thread.currentThread().interrupt();
            throw new RuntimeException(exception);
        }
    }

    class InletsClientOutputHandler
    extends LineOutputHandler {
        public static final String LOG_LINE_FORMAT = "[INLETS CLIENT] {}";
        private static final String INVALID_TOKEN_MESSAGE = "401 Unauthorized";
        private static final String ERROR_LOG_LINE = "level=error";
        private static final String WARN_LOG_LINE = "level=warning";
        private static final String INFO_LOG_LINE = "level=info";

        InletsClientOutputHandler() {
        }

        @Override
        protected void processLine(int lineNum, String line) {
            this.logInletsProcessLine(line);
            if (line.contains(INVALID_TOKEN_MESSAGE)) {
                InletsClientManager.this.tunnelClientProcessStateManager.setErrorCode(Response.Status.UNAUTHORIZED.getStatusCode());
                this.processUnauthorizedErrorReason(line);
            }
        }

        private void logInletsProcessLine(String line) {
            if (line.contains(ERROR_LOG_LINE)) {
                InletsClientManager.this.confidentialDataObfuscator.logError(log, LOG_LINE_FORMAT, line);
            } else if (line.contains(WARN_LOG_LINE)) {
                InletsClientManager.this.confidentialDataObfuscator.logWarn(log, LOG_LINE_FORMAT, line);
            } else if (line.contains(INFO_LOG_LINE)) {
                InletsClientManager.this.confidentialDataObfuscator.logInfo(log, LOG_LINE_FORMAT, line);
            } else {
                InletsClientManager.this.confidentialDataObfuscator.logDebug(log, LOG_LINE_FORMAT, line);
            }
        }

        private void processUnauthorizedErrorReason(String line) {
            Pattern bodyPattern = Pattern.compile("(?<=Response body: ).*?(?=\")");
            Matcher matcher = bodyPattern.matcher(line);
            if (matcher.find()) {
                String reason = matcher.group();
                TunnelUnauthorizedReason tunnelUnauthorizedReason = TunnelUnauthorizedReason.fromDisplayName(reason);
                if (tunnelUnauthorizedReason == TunnelUnauthorizedReason.OTHER) {
                    log.debug("Error with unknown reason occurred for status code 401 Unauthorized");
                }
                InletsClientManager.this.tunnelClientAnalyticsEventPublisher.publish(new TunnelUnauthorizedAnalyticsEvent(tunnelUnauthorizedReason));
            }
        }
    }
}

